///////////////////////////////////////////////////////////////////////////////
//
// wxFormBuilder - A Visual Dialog Editor for wxWidgets.
// Copyright (C) 2005 José Antonio Hurtado
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// Written by
//   José Antonio Hurtado - joseantonio.hurtado@gmail.com
//   Juan Antonio Ortega  - jortegalalmolda@gmail.com
//
///////////////////////////////////////////////////////////////////////////////

#include <component.h>
#include <plugin.h>
#include <xrcconv.h>
#include <ticpp.h>

#include <wx/statline.h>
#include <wx/listctrl.h>
#include <wx/radiobox.h>
#include <wx/bmpbuttn.h>

///////////////////////////////////////////////////////////////////////////////
// Custom status bar class for windows to prevent the status bar gripper from
// moving the entire wxFB window
#if defined(__WIN32__) && wxUSE_NATIVE_STATUSBAR
class wxIndependentStatusBar : public wxStatusBar
{
public:
	wxIndependentStatusBar( wxWindow *parent, wxWindowID id = wxID_ANY, long style = wxST_SIZEGRIP, const wxString& name = wxStatusBarNameStr )
	:
	wxStatusBar( parent, id, style, name )
	{
	}

	// override this virtual function to prevent the status bar from moving the main frame
	virtual WXLRESULT MSWWindowProc(WXUINT nMsg, WXWPARAM wParam, WXLPARAM lParam)
	{
		return wxStatusBarBase::MSWWindowProc(nMsg, wParam, lParam);
	}
};
#else
typedef wxStatusBar wxIndependentStatusBar;
#endif

class wxLeftDownRedirect : public wxEvtHandler
{
private:
	wxWindow* m_window;
	IManager* m_manager;

	void OnLeftDown( wxMouseEvent& )
	{
		m_manager->SelectObject( m_window );
	}

public:
	wxLeftDownRedirect( wxWindow* win, IManager* manager )
	:
	m_window( win ),
	m_manager( manager )
	{

	}
	DECLARE_EVENT_TABLE()
};

BEGIN_EVENT_TABLE( wxLeftDownRedirect, wxEvtHandler )
	EVT_LEFT_DOWN(wxLeftDownRedirect::OnLeftDown)
END_EVENT_TABLE()

///////////////////////////////////////////////////////////////////////////////
/**
Event handler for events generated by controls in this plugin
*/
class ComponentEvtHandler : public wxEvtHandler
{
private:
	wxWindow* m_window;
	IManager* m_manager;

public:
	ComponentEvtHandler( wxWindow* win, IManager* manager )
	:
	m_window( win ),
	m_manager( manager )
	{
	}

protected:
	void OnText( wxCommandEvent& event );
	void OnChecked( wxCommandEvent& event );
	void OnChoice( wxCommandEvent& event );
	void OnTool( wxCommandEvent& event );

	DECLARE_EVENT_TABLE()
};

BEGIN_EVENT_TABLE( ComponentEvtHandler, wxEvtHandler )
	EVT_TEXT( wxID_ANY, ComponentEvtHandler::OnText )
	EVT_CHECKBOX( wxID_ANY, ComponentEvtHandler::OnChecked )
	EVT_CHOICE( wxID_ANY, ComponentEvtHandler::OnChoice )

	// Tools do not get click events, so this will help select them
	EVT_TOOL( wxID_ANY, ComponentEvtHandler::OnTool )
END_EVENT_TABLE()

///////////////////////////////////////////////////////////////////////////////
// FORMS
///////////////////////////////////////////////////////////////////////////////

// TO-DO: The "Form" type component should be drawn in the designer, so that,
//        for instance, a dark panel could be drawn for a wxFrame (N.B. ??)
class FrameFormComponent : public ComponentBase
{
public:
	wxObject* Create(IObject* /*obj*/, wxObject *parent)
	{
		wxPanel *panel = new wxPanel((wxWindow *)parent,-1);
		panel->SetBackgroundColour(wxColour(50,50,50));
		return panel;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxFrame"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty( _("title"), _("title"), XRC_TYPE_TEXT);
		if ( !obj->IsNull( _("center") ) )
		{
			xrc.AddPropertyValue( _("centered"), _("1") );
		}
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("Frame"));
		filter.AddWindowProperties();
		filter.AddProperty( _("title"), _("title"), XRC_TYPE_TEXT);
		filter.AddProperty(_("centered"), _("center"), XRC_TYPE_BITLIST);
		return filter.GetXfbObject();
	}
};

class PanelFormComponent : public ComponentBase
{
public:
	wxObject* Create(IObject* /*obj*/, wxObject *parent)
	{
		wxPanel *panel = new wxPanel((wxWindow *)parent,-1);
		return panel;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxPanel"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("Panel"));
		filter.AddWindowProperties();
		return filter.GetXfbObject();
	}
};

class DialogFormComponent : public ComponentBase
{
public:
	wxObject* Create(IObject* /*obj*/, wxObject *parent)
	{
		wxPanel *panel = new wxPanel((wxWindow *)parent,-1);
		return panel;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxDialog"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty( _("title"), _("title"), XRC_TYPE_TEXT);
		if ( !obj->IsNull( _("center") ) )
		{
			xrc.AddPropertyValue( _("centered"), _("1") );
		}
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("Dialog"));
		filter.AddWindowProperties();
		filter.AddProperty( _("title"), _("title"), XRC_TYPE_TEXT);
		filter.AddProperty(_("centered"), _("center"), XRC_TYPE_BITLIST);
		return filter.GetXfbObject();
	}
};

///////////////////////////////////////////////////////////////////////////////
// WIDGETS
///////////////////////////////////////////////////////////////////////////////

class ButtonComponent : public ComponentBase
{
public:

	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxButton* button = new wxButton((wxWindow*)parent,-1,
			obj->GetPropertyAsString(_("label")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));

		if ( obj->GetPropertyAsInteger( _("default") ) != 0 )
		{
			button->SetDefault();
		}

		return button;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxButton"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		xrc.AddProperty(_("default"),_("default"),XRC_TYPE_BOOL);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxButton"));
		filter.AddWindowProperties();
		filter.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		filter.AddProperty(_("default"),_("default"),XRC_TYPE_BOOL);
		return filter.GetXfbObject();
	}
};

class BitmapButtonComponent : public ComponentBase
{
public:

	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxBitmapButton* button = new wxBitmapButton((wxWindow*)parent,-1,
			obj->GetPropertyAsBitmap(_("bitmap")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));

		if ( obj->GetPropertyAsInteger( _("default") ) != 0 )
		{
			button->SetDefault();
		}

		if ( !obj->IsNull( _("disabled") ) )
		{
			button->SetBitmapDisabled( obj->GetPropertyAsBitmap( _("disabled") ) );
		}

		if ( !obj->IsNull( _("selected") ) )
		{
			button->SetBitmapSelected( obj->GetPropertyAsBitmap( _("selected") ) );
		}

		if ( !obj->IsNull( _("focus") ) )
		{
			button->SetBitmapFocus( obj->GetPropertyAsBitmap( _("focus") ) );
		}

		if ( !obj->IsNull( _("hover") ) )
		{
			button->SetBitmapHover( obj->GetPropertyAsBitmap( _("hover") ) );
		}
		return button;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxBitmapButton"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("bitmap"),_("bitmap"),XRC_TYPE_BITMAP);
		if ( !obj->IsNull( _("disabled") ) )
		{
			xrc.AddProperty(_("disabled"),_("disabled"),XRC_TYPE_BITMAP);
		}
		if ( !obj->IsNull( _("selected") ) )
		{
			xrc.AddProperty(_("selected"),_("selected"),XRC_TYPE_BITMAP);
		}
		if ( !obj->IsNull( _("focus") ) )
		{
			xrc.AddProperty(_("focus"),_("focus"),XRC_TYPE_BITMAP);
		}
		if ( !obj->IsNull( _("hover") ) )
		{
			xrc.AddProperty(_("hover"),_("hover"),XRC_TYPE_BITMAP);
		}
		xrc.AddProperty(_("default"),_("default"),XRC_TYPE_BOOL);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxBitmapButton"));
		filter.AddWindowProperties();
		filter.AddProperty(_("bitmap"),_("bitmap"),XRC_TYPE_BITMAP);
		filter.AddProperty(_("disabled"),_("disabled"),XRC_TYPE_BITMAP);
		filter.AddProperty(_("selected"),_("selected"),XRC_TYPE_BITMAP);
		filter.AddProperty(_("focus"),_("focus"),XRC_TYPE_BITMAP);
		filter.AddProperty(_("hover"),_("hover"),XRC_TYPE_BITMAP);
		filter.AddProperty(_("default"),_("default"),XRC_TYPE_BOOL);
		return filter.GetXfbObject();
	}
};


class TextCtrlComponent : public ComponentBase
{
public:

	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxTextCtrl* tc = new wxTextCtrl((wxWindow *)parent,-1,
			obj->GetPropertyAsString(_("value")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));

		if ( !obj->IsNull( _("maxlength") ) )
		{
			tc->SetMaxLength( obj->GetPropertyAsInteger( _("maxlength") ) );
		}

		tc->PushEventHandler( new ComponentEvtHandler( tc, GetManager() ) );

		return tc;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxTextCtrl"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("value"),_("value"),XRC_TYPE_TEXT);
        if (!obj->IsNull(_("maxlength")))
            xrc.AddProperty(_("maxlength"), _("maxlength"), XRC_TYPE_INTEGER);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxTextCtrl"));
		filter.AddWindowProperties();
		filter.AddProperty(_("value"),_("value"),XRC_TYPE_TEXT);
        filter.AddProperty(_("maxlength"), _("maxlength"), XRC_TYPE_INTEGER);
		return filter.GetXfbObject();
	}

};

void ComponentEvtHandler::OnText( wxCommandEvent& )
{
	wxTextCtrl* tc = wxDynamicCast( m_window, wxTextCtrl );
	if ( tc != NULL )
	{
		m_manager->ModifyProperty( m_window, _("value"), tc->GetValue() );
		tc->SetInsertionPointEnd();
		tc->SetFocus();
	}
}

class StaticTextComponent : public ComponentBase
{
public:

	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxStaticText* st = new wxStaticText((wxWindow *)parent,-1,
			obj->GetPropertyAsString(_("label")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));

		st->Wrap( obj->GetPropertyAsInteger( _("wrap") ) );

		return st;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		wxString name = obj->GetPropertyAsString(_("name"));
		ObjectToXrcFilter xrc(obj, _("wxStaticText"), name);
		xrc.AddWindowProperties();
		xrc.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxStaticText"));
		filter.AddWindowProperties();
		filter.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		return filter.GetXfbObject();
	}
};

class ComboBoxComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxComboBox *combo = new wxComboBox((wxWindow *)parent,-1,
			obj->GetPropertyAsString(_("value")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			0,
			NULL,
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));

		// choices
		wxArrayString choices = obj->GetPropertyAsArrayString(_("choices"));
		for (unsigned int i=0; i<choices.Count(); i++)
			combo->Append(choices[i]);

		return combo;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxComboBox"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("value"),_("value"),XRC_TYPE_TEXT);
		xrc.AddProperty(_("choices"),_("content"),XRC_TYPE_STRINGLIST);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxComboBox"));
		filter.AddWindowProperties();
		filter.AddProperty(_("value"),_("value"),XRC_TYPE_TEXT);
		filter.AddProperty(_("content"),_("choices"),XRC_TYPE_STRINGLIST);
		return filter.GetXfbObject();
	}
};

class CheckBoxComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxCheckBox *res = new wxCheckBox((wxWindow *)parent,-1,
			obj->GetPropertyAsString(_("label")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("window_style")) | obj->GetPropertyAsInteger(_T("style")));
        res->SetValue(obj->GetPropertyAsInteger(_T("checked")) != 0);

        res->PushEventHandler( new ComponentEvtHandler( res, GetManager() ) );

        return res;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxCheckBox"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		xrc.AddProperty(_("checked"),_("checked"),XRC_TYPE_BOOL);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxCheckBox"));
		filter.AddWindowProperties();
		filter.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		filter.AddProperty(_("checked"),_("checked"),XRC_TYPE_BOOL);
		return filter.GetXfbObject();
	}
};

void ComponentEvtHandler::OnChecked( wxCommandEvent& )
{
	wxCheckBox* cb = wxDynamicCast( m_window, wxCheckBox );
	if ( cb != NULL )
	{
		wxString cbValue;
		cbValue.Printf( wxT("%i"), cb->GetValue() );
		m_manager->ModifyProperty( m_window, _("checked"), cbValue );
		cb->SetFocus();
	}
}

class StaticBitmapComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		return new wxStaticBitmap((wxWindow *)parent,-1,
			obj->GetPropertyAsBitmap(_("bitmap")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("window_style")));
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxStaticBitmap"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty( _("bitmap"), _("bitmap"), XRC_TYPE_BITMAP );
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxStaticBitmap"));
		filter.AddWindowProperties();
    filter.AddProperty(_("bitmap"),_("bitmap"),XRC_TYPE_BITMAP);
		return filter.GetXfbObject();
	}
};

class XpmStaticBitmapComponent : public StaticBitmapComponent
{
};

class StaticLineComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		return new wxStaticLine((wxWindow *)parent,-1,
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxStaticLine"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxStaticLine"));
		filter.AddWindowProperties();
		return filter.GetXfbObject();
	}
};

class ListCtrlComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxListCtrl *lc = new wxListCtrl((wxWindow*)parent, -1,
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			(obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style"))) & ~wxLC_VIRTUAL);


		// Refilling
		int i,j;
		wxString buf;
		if ( (lc->GetWindowStyle() & wxLC_REPORT) != 0 )
		{
			for (i=0;i<4;i++)
			{
				buf.Printf(wxT("Label %d"),i);
				lc->InsertColumn(i, buf, wxLIST_FORMAT_LEFT, 80);
			}
		}

		for (j=0;j<10;j++)
		{
			long temp;
			buf.Printf(wxT("Cell (0,%d)"),j);
			temp = lc->InsertItem(j,buf);
			if ( (lc->GetWindowStyle() & wxLC_REPORT) != 0 )
			{
				for (i=1;i<4;i++)
				{
					buf.Printf(wxT("Cell (%d,%d)"),i,j);
					lc->SetItem(temp,i,buf);
				}
			}
		}

		return lc;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxListCtrl"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxListCtrl"));
		filter.AddWindowProperties();
		return filter.GetXfbObject();
	}
};

class ListBoxComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxListBox *listbox = new wxListBox((wxWindow*)parent, -1,
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			0,
			NULL,
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));

		// choices
		wxArrayString choices = obj->GetPropertyAsArrayString(_("choices"));
		for (unsigned int i=0; i<choices.Count(); i++)
			listbox->Append(choices[i]);

		return listbox;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxListBox"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("choices"), _("content"), XRC_TYPE_STRINGLIST);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxListBox"));
		filter.AddWindowProperties();
		filter.AddProperty(_("content"),_("choices"), XRC_TYPE_STRINGLIST);
		return filter.GetXfbObject();
	}
};

class RadioBoxComponent : public ComponentBase, public wxEvtHandler
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxArrayString choices = obj->GetPropertyAsArrayString(_("choices"));
		int count = choices.Count();
		if ( 0 == count )
		{
		    choices.Add( _("wxRadioBox must have at least one choice") );
		    count = 1;
		}

		int majorDim = obj->GetPropertyAsInteger(_("majorDimension"));
		if (majorDim < 1)
		{
			wxLogWarning(_("majorDimension must be greater than zero."));
			majorDim = 1;
		}

		wxRadioBox *radiobox = new wxRadioBox((wxWindow*)parent, -1,
			obj->GetPropertyAsString(_("label")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			choices,
			majorDim,
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));

		int selection = obj->GetPropertyAsInteger( _("selection") );
		if ( selection < count )
		{
			radiobox->SetSelection( selection );
		}

		radiobox->Connect( wxEVT_COMMAND_RADIOBOX_SELECTED, wxCommandEventHandler( RadioBoxComponent::OnRadioBox ), NULL, this );

		return radiobox;
	}

	void OnRadioBox( wxCommandEvent& event )
	{
		wxRadioBox* window = dynamic_cast< wxRadioBox* >( event.GetEventObject() );
		if ( 0 != window )
		{
			wxString value;
			value.Printf( wxT("%i"), window->GetSelection() );
			GetManager()->ModifyProperty( window, _("selection"), value );
			window->SetFocus();

			GetManager()->SelectObject( window );
		}
	}

	void Cleanup( wxObject* obj )
	{
		wxRadioBox* window = dynamic_cast< wxRadioBox* >( obj );
		if ( 0 != window )
		{
			window->Disconnect( wxEVT_COMMAND_RADIOBOX_SELECTED, wxCommandEventHandler( RadioBoxComponent::OnRadioBox ), NULL, this );
		}
		ComponentBase::Cleanup( obj );
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxRadioBox"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("label"), _("label"), XRC_TYPE_TEXT);
		xrc.AddProperty(_("selection"), _("selection"), XRC_TYPE_INTEGER );
		xrc.AddProperty(_("choices"), _("content"), XRC_TYPE_STRINGLIST);
		xrc.AddProperty(_("majorDimension"), _("dimension"), XRC_TYPE_INTEGER);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxRadioBox"));
		filter.AddWindowProperties();
		filter.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		filter.AddProperty(_("selection"), _("selection"), XRC_TYPE_INTEGER );
		filter.AddProperty(_("content"),_("choices"), XRC_TYPE_STRINGLIST);
		filter.AddProperty(_("dimension"), _("majorDimension"), XRC_TYPE_INTEGER);
		return filter.GetXfbObject();
	}
};

class RadioButtonComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxRadioButton *rb = new wxRadioButton((wxWindow *)parent,-1,
			obj->GetPropertyAsString(_("label")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));

		rb->SetValue( ( obj->GetPropertyAsInteger(_("value")) != 0 ) );
		return rb;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxRadioButton"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("label"),_("label"), XRC_TYPE_TEXT);
		xrc.AddProperty(_("value"),_("value"), XRC_TYPE_BOOL);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxRadioButton"));
		filter.AddWindowProperties();
		filter.AddProperty(_("label"),_("label"), XRC_TYPE_TEXT);
		filter.AddProperty(_("value"),_("value"), XRC_TYPE_BOOL);
		return filter.GetXfbObject();
	}
};

class StatusBarComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxStatusBar *sb = new wxIndependentStatusBar((wxWindow*)parent, -1,
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));
		sb->SetFieldsCount(obj->GetPropertyAsInteger(_("fields")));

		#ifndef __WXMSW__
		sb->PushEventHandler( new wxLeftDownRedirect( sb, GetManager() ) );
		#endif
		return sb;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxStatusBar"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("fields"),_("fields"),XRC_TYPE_INTEGER);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxStatusBar"));
		filter.AddWindowProperties();
		filter.AddProperty(_("fields"),_("fields"),XRC_TYPE_INTEGER);
		return filter.GetXfbObject();
	}
};

class MenuBarComponent : public ComponentBase
{
public:

	wxObject* Create(IObject *obj, wxObject* /*parent*/)
	{
		wxMenuBar *mb = new wxMenuBar(obj->GetPropertyAsInteger(_("style")) |
			obj->GetPropertyAsInteger(_("window_style")));
		return mb;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxMenuBar"), obj->GetPropertyAsString(_("name")));
		xrc.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxMenuBar"));
		filter.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		return filter.GetXfbObject();
	}
};

class MenuComponent : public ComponentBase
{
public:

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxMenu"), obj->GetPropertyAsString(_("name")));
		xrc.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxMenu"));
		filter.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
		return filter.GetXfbObject();
	}
};

class SubMenuComponent : public ComponentBase
{
 public:

  ticpp::Element* ExportToXrc(IObject *obj)
  {
    ObjectToXrcFilter xrc(obj, _("wxMenu"), obj->GetPropertyAsString(_("name")));
    xrc.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
    return xrc.GetXrcObject();
  }

  ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
  {
    XrcToXfbFilter filter(xrcObj, _("submenu"));
    filter.AddProperty(_("label"),_("label"),XRC_TYPE_TEXT);
    return filter.GetXfbObject();
  }
};

class MenuItemComponent : public ComponentBase
{
public:

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxMenuItem"), obj->GetPropertyAsString(_("name")));
		wxString shortcut = obj->GetPropertyAsString(_("shortcut"));
		wxString label;
		if (shortcut.IsEmpty())
			label = obj->GetPropertyAsString(_("label"));
		else
			label = obj->GetPropertyAsString(_("label")) + wxT("\\t") + shortcut;

		xrc.AddPropertyValue(_("label"), label, true);
		xrc.AddProperty(_("help"),_("help"),XRC_TYPE_TEXT);

	    if (!obj->IsNull(_("bitmap")))
	      xrc.AddProperty(_("bitmap"),_("bitmap"),XRC_TYPE_BITMAP);

		int kind = obj->GetPropertyAsInteger(_("kind"));

		if (obj->GetPropertyAsInteger(_("checked")) && (kind == wxITEM_RADIO || kind == wxITEM_CHECK))
			xrc.AddProperty(_("checked"), _("checked"), XRC_TYPE_BOOL);

		if (obj->GetPropertyAsInteger(_("enabled")) == 0)
			xrc.AddProperty(_("enabled"), _("enabled"), XRC_TYPE_BOOL);

		switch (kind)
		{
		case wxITEM_CHECK: xrc.AddPropertyValue(_("checkable"), _("1")); break;
		case wxITEM_RADIO: xrc.AddPropertyValue(_("radio"), _("1")); break;
		}

		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxMenuItem"));

		try
		{
			ticpp::Element *labelElement = xrcObj->FirstChildElement("label");
			wxString label( labelElement->GetText().c_str(), wxConvUTF8 );

			wxString shortcut;
			int pos = label.Find( wxT("\\t") );
			if ( pos >= 0 )
			{
				shortcut = label.Mid( pos + 2 );
				label = label.Left( pos );
			}

			filter.AddPropertyValue( _("label"), label, true );
			filter.AddPropertyValue( _("shortcut"), shortcut );
		}
		catch( ticpp::Exception& )
		{
		}

		filter.AddProperty(_("help"),_("help"),XRC_TYPE_TEXT);
		filter.AddProperty(_("bitmap"),_("bitmap"),XRC_TYPE_BITMAP);
		return filter.GetXfbObject();
	}
};

class SeparatorComponent : public ComponentBase
{
public:

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("separator"));
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("separator"));
		return filter.GetXfbObject();
	}
};

class ToolBarComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxToolBar *tb = new wxToolBar((wxWindow*)parent, -1,
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")) | wxTB_NOALIGN | wxTB_NODIVIDER | wxNO_BORDER);

		if (!obj->IsNull(_("bitmapsize")))
			tb->SetToolBitmapSize(obj->GetPropertyAsSize(_("bitmapsize")));
		if (!obj->IsNull(_("margins")))
		{
			wxSize margins(obj->GetPropertyAsSize(_("margins")));
			tb->SetMargins(margins.GetWidth(), margins.GetHeight());
		}
		if (!obj->IsNull(_("packing")))
			tb->SetToolPacking(obj->GetPropertyAsInteger(_("packing")));
		if (!obj->IsNull(_("separation")))
			tb->SetToolSeparation(obj->GetPropertyAsInteger(_("separation")));

		tb->PushEventHandler( new ComponentEvtHandler( tb, GetManager() ) );

		return tb;
	}

	void OnCreated( wxObject* wxobject, wxWindow* /*wxparent*/ )
	{
		wxToolBar* tb = wxDynamicCast( wxobject, wxToolBar );
		if ( NULL == tb )
		{
			// very very strange
			return;
		}

		size_t count = GetManager()->GetChildCount( wxobject );
		for ( size_t i = 0; i < count; ++i )
		{
			wxObject* child = GetManager()->GetChild( wxobject, i );
			IObject* childObj = GetManager()->GetIObject( child );
			if ( wxT("tool") == childObj->GetClassName() )
			{
				tb->AddTool( 	wxID_ANY,
								childObj->GetPropertyAsString( _("label") ),
								childObj->GetPropertyAsBitmap( _("bitmap") ),
								wxNullBitmap,
								(wxItemKind)childObj->GetPropertyAsInteger( _("kind") ),
								childObj->GetPropertyAsString( _("help") ),
								wxEmptyString,
								child
							);
			}
			else if ( wxT("toolSeparator") == childObj->GetClassName() )
			{
				tb->AddSeparator();
			}
			else
			{
				wxControl* control = wxDynamicCast( child, wxControl );
				if ( NULL != control )
				{
					tb->AddControl( control );
				}
			}
		}
		tb->Realize();

	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxToolBar"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("bitmapsize"), _("bitmapsize"), XRC_TYPE_SIZE);
		xrc.AddProperty(_("margins"), _("margins"), XRC_TYPE_SIZE);
		xrc.AddProperty(_("packing"), _("packing"), XRC_TYPE_INTEGER);
		xrc.AddProperty(_("separation"), _("separation"), XRC_TYPE_INTEGER);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxToolBar"));
		filter.AddWindowProperties();
		filter.AddProperty(_("bitmapsize"), _("bitmapsize"), XRC_TYPE_SIZE);
		filter.AddProperty(_("margins"), _("margins"), XRC_TYPE_SIZE);
		filter.AddProperty(_("packing"), _("packing"), XRC_TYPE_INTEGER);
		filter.AddProperty(_("separation"), _("separation"), XRC_TYPE_INTEGER);
		return filter.GetXfbObject();
	}
};

void ComponentEvtHandler::OnTool( wxCommandEvent& event )
{
	wxToolBar* tb = wxDynamicCast( event.GetEventObject(), wxToolBar );
	if ( NULL == tb )
	{
		// very very strange
		return;
	}

	wxObject* wxobject = tb->GetToolClientData( event.GetId() );
	if ( NULL != wxobject )
	{
		m_manager->SelectObject( wxobject );
	}
}

class ToolComponent : public ComponentBase
{
public:

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("tool"), obj->GetPropertyAsString(_("name")));
		xrc.AddProperty(_("label"), _("label"), XRC_TYPE_TEXT);
		xrc.AddProperty(_("tooltip"), _("tooltip"), XRC_TYPE_TEXT);
		xrc.AddProperty(_("statusbar"), _("longhelp"), XRC_TYPE_TEXT);
		xrc.AddProperty(_("bitmap"), _("bitmap"), XRC_TYPE_BITMAP);

		wxItemKind kind = (wxItemKind)obj->GetPropertyAsInteger(_("kind"));
		if ( wxITEM_CHECK == kind )
		{
			xrc.AddPropertyValue( wxT("toggle"), wxT("1") );
		}
		else if ( wxITEM_RADIO == kind )
		{
			xrc.AddPropertyValue( wxT("radio"), wxT("1") );
		}
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("tool"));
		filter.AddProperty(_("longhelp"), _("statusbar"), XRC_TYPE_TEXT);
		filter.AddProperty(_("tooltip"), _("tooltip"), XRC_TYPE_TEXT);
		filter.AddProperty(_("label"), _("label"), XRC_TYPE_TEXT);
		filter.AddProperty(_("bitmap"), _("bitmap"), XRC_TYPE_BITMAP);
		bool gotToggle = false;
		bool gotRadio = false;
		ticpp::Element* toggle = xrcObj->FirstChildElement( "toggle", false );
		if ( toggle )
		{
			toggle->GetTextOrDefault( &gotToggle, false );
			if ( gotToggle )
			{
				filter.AddPropertyValue( _("kind"), wxT("wxITEM_CHECK") );
			}
		}
		if ( !gotToggle )
		{
			ticpp::Element* radio = xrcObj->FirstChildElement( "radio", false );
			if ( radio )
			{
				radio->GetTextOrDefault( &gotRadio, false );
				if ( gotRadio )
				{
					filter.AddPropertyValue( _("kind"), wxT("wxITEM_RADIO") );
				}
			}
		}
		if ( !(gotToggle || gotRadio) )
		{
			filter.AddPropertyValue( _("kind"), wxT("wxITEM_NORMAL") );
		}

		return filter.GetXfbObject();
	}
};

class ToolSeparatorComponent : public ComponentBase
{
public:
	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc( obj, _("separator") );
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter( xrcObj, _("toolSeparator") );
		return filter.GetXfbObject();
	}
};

class ChoiceComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxArrayString choices = obj->GetPropertyAsArrayString(_("choices"));
		wxString *strings = new wxString[choices.Count()];
		for (unsigned int i=0; i < choices.Count(); i++)
			strings[i] = choices[i];

		wxChoice *choice = new wxChoice((wxWindow*)parent, -1,
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			(int)choices.Count(),
			strings,
			obj->GetPropertyAsInteger(_("window_style")));

		choice->SetSelection(obj->GetPropertyAsInteger(_("selection")));

		delete []strings;

		choice->PushEventHandler( new ComponentEvtHandler( choice, GetManager() ) );

		return choice;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxChoice"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("selection"), _("selection"), XRC_TYPE_INTEGER);
		xrc.AddProperty(_("choices"), _("content"), XRC_TYPE_STRINGLIST);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxChoice"));
		filter.AddWindowProperties();
		filter.AddProperty(_("selection"), _("selection"), XRC_TYPE_INTEGER);
		filter.AddProperty(_("content"),_("choices"), XRC_TYPE_STRINGLIST);
		return filter.GetXfbObject();
	}

};

void ComponentEvtHandler::OnChoice( wxCommandEvent& )
{
	wxChoice* window = wxDynamicCast( m_window, wxChoice );
	if ( window != NULL )
	{
		wxString value;
		value.Printf( wxT("%i"), window->GetSelection() );
		m_manager->ModifyProperty( m_window, _("selection"), value );
		window->SetFocus();
	}
}

class SliderComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		return new wxSlider((wxWindow *)parent,-1,
			obj->GetPropertyAsInteger(_("value")),
			obj->GetPropertyAsInteger(_("minValue")),
			obj->GetPropertyAsInteger(_("maxValue")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")) | obj->GetPropertyAsInteger(_("window_style")));
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxSlider"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("value"), _("value"), XRC_TYPE_INTEGER);
		xrc.AddProperty( _("minValue"), _("min"), XRC_TYPE_INTEGER);
		xrc.AddProperty( _("maxValue"), _("max"), XRC_TYPE_INTEGER);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxSlider"));
		filter.AddWindowProperties();
		filter.AddProperty(_("value"), _("value"), XRC_TYPE_INTEGER);
		filter.AddProperty(_("min"), _("minValue"), XRC_TYPE_INTEGER);
		filter.AddProperty(_("max"), _("maxValue"), XRC_TYPE_INTEGER);
		return filter.GetXfbObject();
	}

};

class GaugeComponent : public ComponentBase
{
public:
	wxObject* Create(IObject *obj, wxObject *parent)
	{
		wxGauge *gauge = new wxGauge((wxWindow *)parent,-1,
			obj->GetPropertyAsInteger(_("range")),
			obj->GetPropertyAsPoint(_("pos")),
			obj->GetPropertyAsSize(_("size")),
			obj->GetPropertyAsInteger(_("style")) | obj->GetPropertyAsInteger(_("window_style")));
		gauge->SetValue(obj->GetPropertyAsInteger(_("value")));
		return gauge;
	}

	ticpp::Element* ExportToXrc(IObject *obj)
	{
		ObjectToXrcFilter xrc(obj, _("wxGauge"), obj->GetPropertyAsString(_("name")));
		xrc.AddWindowProperties();
		xrc.AddProperty(_("range"), _("range"), XRC_TYPE_INTEGER);
		xrc.AddProperty(_("value"), _("value"), XRC_TYPE_INTEGER);
		return xrc.GetXrcObject();
	}

	ticpp::Element* ImportFromXrc( ticpp::Element* xrcObj )
	{
		XrcToXfbFilter filter(xrcObj, _("wxGauge"));
		filter.AddWindowProperties();
		filter.AddProperty(_("range"), _("range"), XRC_TYPE_INTEGER);
		filter.AddProperty(_("value"), _("value"), XRC_TYPE_INTEGER);
		return filter.GetXfbObject();
	}

};

///////////////////////////////////////////////////////////////////////////////

BEGIN_LIBRARY()

// forms aren't considered as windows !
ABSTRACT_COMPONENT("Frame",FrameFormComponent)
ABSTRACT_COMPONENT("Panel",PanelFormComponent)
ABSTRACT_COMPONENT("Dialog",DialogFormComponent)

WINDOW_COMPONENT("wxButton",ButtonComponent)
WINDOW_COMPONENT("wxBitmapButton",BitmapButtonComponent)
WINDOW_COMPONENT("wxTextCtrl",TextCtrlComponent)
WINDOW_COMPONENT("wxStaticText",StaticTextComponent)
WINDOW_COMPONENT("wxComboBox", ComboBoxComponent)
WINDOW_COMPONENT("wxListBox", ListBoxComponent)
WINDOW_COMPONENT("wxRadioBox", RadioBoxComponent)
WINDOW_COMPONENT("wxRadioButton",RadioButtonComponent)
WINDOW_COMPONENT("wxCheckBox", CheckBoxComponent)
WINDOW_COMPONENT("wxStaticBitmap", StaticBitmapComponent)
WINDOW_COMPONENT("wxStaticLine", StaticLineComponent)
WINDOW_COMPONENT("wxMenuBar", MenuBarComponent)
WINDOW_COMPONENT("wxMenu", MenuComponent)
WINDOW_COMPONENT("submenu", SubMenuComponent)
WINDOW_COMPONENT("wxMenuItem", MenuItemComponent)
WINDOW_COMPONENT("separator", SeparatorComponent)
WINDOW_COMPONENT("wxListCtrl", ListCtrlComponent)
WINDOW_COMPONENT("wxStatusBar", StatusBarComponent)
WINDOW_COMPONENT("wxToolBar", ToolBarComponent)
ABSTRACT_COMPONENT("tool", ToolComponent)
ABSTRACT_COMPONENT("toolSeparator", ToolSeparatorComponent)
WINDOW_COMPONENT("wxChoice", ChoiceComponent)
WINDOW_COMPONENT("wxSlider", SliderComponent)
WINDOW_COMPONENT("wxGauge", GaugeComponent)

// wxWindow style macros
MACRO(wxSIMPLE_BORDER)
MACRO(wxDOUBLE_BORDER)
MACRO(wxSUNKEN_BORDER)
MACRO(wxRAISED_BORDER)
MACRO(wxSTATIC_BORDER)
MACRO(wxNO_BORDER)
MACRO(wxTRANSPARENT_WINDOW)
MACRO(wxTAB_TRAVERSAL)
MACRO(wxWANTS_CHARS)
MACRO(wxVSCROLL)
MACRO(wxHSCROLL)
MACRO(wxALWAYS_SHOW_SB)
MACRO(wxCLIP_CHILDREN)
MACRO(wxFULL_REPAINT_ON_RESIZE)
MACRO(wxWS_EX_VALIDATE_RECURSIVELY)
MACRO(wxWS_EX_BLOCK_EVENTS)
MACRO(wxWS_EX_TRANSIENT)
MACRO(wxWS_EX_PROCESS_IDLE)
MACRO(wxWS_EX_PROCESS_UI_UPDATES)

// wxFrame style macros
MACRO(wxDEFAULT_FRAME_STYLE)
MACRO(wxICONIZE)
MACRO(wxCAPTION)
MACRO(wxMINIMIZE)
MACRO(wxMINIMIZE_BOX)
MACRO(wxMAXIMIZE)
MACRO(wxMAXIMIZE_BOX)
MACRO(wxCLOSE_BOX)
MACRO(wxSTAY_ON_TOP)
MACRO(wxSYSTEM_MENU)
MACRO(wxRESIZE_BORDER)
MACRO(wxFRAME_TOOL_WINDOW)
MACRO(wxFRAME_NO_TASKBAR)
MACRO(wxFRAME_FLOAT_ON_PARENT)
MACRO(wxFRAME_SHAPED)
MACRO(wxFRAME_EX_CONTEXTHELP)
MACRO(wxFRAME_EX_METAL)

// wxDialog style macros
MACRO(wxCAPTION)
MACRO(wxDEFAULT_DIALOG_STYLE)
MACRO(wxRESIZE_BORDER)
MACRO(wxSYSTEM_MENU)
MACRO(wxCLOSE_BOX)
MACRO(wxMAXIMIZE_BOX)
MACRO(wxMINIMIZE_BOX)
MACRO(wxSTAY_ON_TOP)
MACRO(wxDIALOG_NO_PARENT)

// wxButton
MACRO(wxBU_LEFT)
MACRO(wxBU_TOP)
MACRO(wxBU_RIGHT)
MACRO(wxBU_BOTTOM)
MACRO(wxBU_EXACTFIT)
MACRO(wxBU_AUTODRAW)

// wxStaticText
MACRO(wxALIGN_LEFT)
MACRO(wxALIGN_CENTRE)
MACRO(wxALIGN_RIGHT)
MACRO(wxST_NO_AUTORESIZE)

// wxTextCtrl
MACRO(wxTE_MULTILINE)
MACRO(wxTE_READONLY)
MACRO(wxTE_RICH)
MACRO(wxTE_AUTO_URL)
MACRO(wxTE_CAPITALIZE)
MACRO(wxTE_CENTRE)
MACRO(wxTE_CHARWRAP)
MACRO(wxTE_DONTWRAP)
MACRO(wxTE_LEFT)
MACRO(wxTE_NOHIDESEL)
MACRO(wxTE_PASSWORD)
MACRO(wxTE_PROCESS_ENTER)
MACRO(wxTE_PROCESS_TAB)
MACRO(wxTE_RICH2)
MACRO(wxTE_RIGHT)
MACRO(wxTE_WORDWRAP)

// wxStaticLine
MACRO(wxLI_HORIZONTAL)
MACRO(wxLI_VERTICAL)

// wxListCtrl
MACRO(wxLC_LIST)
MACRO(wxLC_REPORT)
MACRO(wxLC_VIRTUAL)
MACRO(wxLC_ICON)
MACRO(wxLC_SMALL_ICON)
MACRO(wxLC_ALIGN_TOP)
MACRO(wxLC_ALIGN_LEFT)
MACRO(wxLC_AUTOARRANGE)
MACRO(wxLC_EDIT_LABELS)
MACRO(wxLC_NO_SORT_HEADER)
MACRO(wxLC_NO_HEADER)
MACRO(wxLC_SINGLE_SEL)
MACRO(wxLC_SORT_ASCENDING)
MACRO(wxLC_SORT_DESCENDING)
MACRO(wxLC_HRULES)
MACRO(wxLC_VRULES)

// wxListBox
MACRO(wxLB_SINGLE)
MACRO(wxLB_MULTIPLE)
MACRO(wxLB_EXTENDED)
MACRO(wxLB_HSCROLL)
MACRO(wxLB_ALWAYS_SB)
MACRO(wxLB_NEEDED_SB)
MACRO(wxLB_SORT)

// wxRadioBox
MACRO(wxRA_SPECIFY_ROWS)
MACRO(wxRA_SPECIFY_COLS)
MACRO(wxRA_USE_CHECKBOX)

// wxRadioButton
MACRO(wxRB_GROUP)
MACRO(wxRB_SINGLE)
MACRO(wxRB_USE_CHECKBOX)

// wxStatusBar
MACRO(wxST_SIZEGRIP)

// wxMenuBar
MACRO(wxMB_DOCKABLE)

// wxMenuItem
MACRO(wxITEM_NORMAL)
MACRO(wxITEM_CHECK)
MACRO(wxITEM_RADIO)

// wxToolBar
MACRO(wxTB_FLAT)
MACRO(wxTB_DOCKABLE)
MACRO(wxTB_HORIZONTAL)
MACRO(wxTB_VERTICAL)
MACRO(wxTB_TEXT)
MACRO(wxTB_NOICONS)
MACRO(wxTB_NODIVIDER)
MACRO(wxTB_NOALIGN)
MACRO(wxTB_HORZ_LAYOUT)
MACRO(wxTB_HORZ_TEXT)

// wxTool
MACRO(wxITEM_NORMAL)
MACRO(wxITEM_CHECK)
MACRO(wxITEM_RADIO)

// wxSlider
MACRO(wxSL_AUTOTICKS)
MACRO(wxSL_BOTTOM)
MACRO(wxSL_HORIZONTAL)
MACRO(wxSL_INVERSE)
MACRO(wxSL_LABELS)
MACRO(wxSL_LEFT)
MACRO(wxSL_RIGHT)
MACRO(wxSL_SELRANGE)
MACRO(wxSL_TOP)
MACRO(wxSL_VERTICAL)
MACRO(wxSL_BOTH)

// wxComboBox
MACRO(wxCB_DROPDOWN)
MACRO(wxCB_READONLY)
MACRO(wxCB_SIMPLE)
MACRO(wxCB_SORT)

// wxCheckBox
MACRO(wxCHK_2STATE)
MACRO(wxCHK_3STATE)
MACRO(wxCHK_ALLOW_3RD_STATE_FOR_USER)

// wxGauge
MACRO(wxGA_HORIZONTAL)
MACRO(wxGA_SMOOTH)
MACRO(wxGA_VERTICAL)

//wxDialog
MACRO(wxBOTH)
SYNONYMOUS(1,wxBOTH)

END_LIBRARY()


