Check Signature Tutorial
========================

Objective
---------

In this tutorial, you will 
check that an SSH signature is for a specific message
and print the public key that was used to sign the message.


Prerequisites
-------------

You need to install the `sshsig` Python package.
A popular way to do this is by running:

```
pip install sshsig
```

The steps in this tutorial include lines of Python code to be run within a Python interpreter.
A popular way to run the Python interpreter is to run:
```
python3
```
from the command line.


Steps
-----

### 1. Get the message that was signed.

```python
message = "Hello World.\n"
```

The author of this tutorial signed the above message using a
[public/private key pair](https://en.wikipedia.org/wiki/Public-key_cryptography).


### 2. Get the plain-text encoded signature.

```python
signature = """
-----BEGIN SSH SIGNATURE-----
U1NIU0lHAAAAAQAAADMAAAALc3NoLWVkMjU1MTkAAAAghB1C63jrmh3eWRXJVbrTfw9wP/
BIZf/aKPdFxBlMCq0AAAADZ2l0AAAAAAAAAAZzaGE1MTIAAABTAAAAC3NzaC1lZDI1NTE5
AAAAQCK1pq8IPKuxTat0RJbZ0LjElEgXNowFdFKElzuq9gunl2b4DHrla0FTkUJKp1eOk1
+GOGN+EejKcsFnrfmZ1gE=
-----END SSH SIGNATURE-----
"""
```

The above SSH signature was generated by the `ssh-keygen` utility
using the message and the author's public/private key pair.


### 3. Check the signature with the message.

```python
import sshsig
pub_key = sshsig.check_signature(message, signature)
```

Since `check_signature` returns without raising an exception, we know that
the signature resulted from signing `message`.
The public key returned is from the public/private key pair used to sign
the message.


### 4. Print the public key used to sign.

We can also print out a textual representation of the signing public key in OpenSSH format.

```
str(pub_key)
```


### 5. Check with a different message.

The signature is specifically for `message`.
Checking a different message will fail.

```python
sshsig.check_signature("A different message", signature)
```

An `sshsig.sshsig.InvalidSignature` exception is raised,
indicating that the signature is not for a different message.


### Conclusion

You have successfully checked that the signature is for the message in this tutorial.
You have also printed the public key that was used to sign the message.
