///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/snapping/SnappingManager.h>
#include "GridSnappingProvider.h"

namespace Core {

IMPLEMENT_PLUGIN_CLASS(GridSnappingProvider, SnappingProvider)

/******************************************************************************
* Renders the marker for the given world point in the viewport.
******************************************************************************/
void GridSnappingMarker::render(Viewport& vp, const Point3& worldPoint) 
{
	CHECK_POINTER(snapViewport);

	FloatType size = snapViewport->grid().gridSpacing();
	Point3 gridPoint = snapViewport->grid().inverseGridMatrix() * worldPoint;

	Point3 markerPoints[4] = {
		Point3(gridPoint.X - size, gridPoint.Y, 0),
		Point3(gridPoint.X + size, gridPoint.Y, 0),
		Point3(gridPoint.X, gridPoint.Y - size, 0),
		Point3(gridPoint.X, gridPoint.Y + size, 0)
	};
	
	Box3 bb(Point3(gridPoint.X-size, gridPoint.Y-size, 0),
			Point3(gridPoint.X+size, gridPoint.Y+size, 0));

	vp.setWorldMatrix(snapViewport->grid().gridMatrix());
	vp.setRenderingColor(Viewport::getVPColor(Viewport::COLOR_SNAPPING_MARKER));
    vp.renderLines(4, bb, markerPoints);
}

/******************************************************************************
* This methods computes hits on the scene objects and records them using the
* given context object.
******************************************************************************/
void GridSnappingProvider::snap(SnappingContext& context) 
{
	Point3 result;
	if(!context.viewport()->grid().screenComputePlaneIntersection(context.mousePos(), result)) 
		return;

	// Snap
	FloatType gridSize = context.viewport()->grid().gridSpacing();
	if(gridSize == 0) return;
	
	result.X = floor(result.X / gridSize + 0.5) * gridSize;
	result.Y = floor(result.Y / gridSize + 0.5) * gridSize;

	// Transform result to world space.
	result = context.viewport()->grid().gridMatrix() * result;
	
	// Record hit.
	gridMarker->snapViewport = context.viewport();
	context.recordHit(result, gridMarker);
}

};
