/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isPositiveInteger = require( '@stdlib/assert/is-positive-integer' ).isPrimitive;
var incrmmean = require( './../../../incr/mmean' );


// MAIN //

/**
* Returns an accumulator function which incrementally computes a moving arithmetic mean of squared absolute values.
*
* @param {PositiveInteger} W - window size
* @throws {TypeError} must provide a positive integer
* @returns {Function} accumulator function
*
* @example
* var accumulator = incrmmeanabs2( 3 );
*
* var m = accumulator();
* // returns null
*
* m = accumulator( 2.0 );
* // returns 4.0
*
* m = accumulator( -5.0 );
* // returns 14.5
*
* m = accumulator( 3.0 );
* // returns ~12.67
*
* m = accumulator( 5.0 );
* // returns ~19.67
*
* m = accumulator();
* // returns ~19.67
*/
function incrmmeanabs2( W ) {
	var mean;
	if ( !isPositiveInteger( W ) ) {
		throw new TypeError( 'invalid argument. Must provide a positive integer. Value: `' + W + '`.' );
	}
	mean = incrmmean( W );
	return accumulator;

	/**
	* If provided a value, the accumulator function returns an updated mean. If not provided a value, the accumulator function returns the current mean.
	*
	* @private
	* @param {number} [x] - input value
	* @returns {(number|null)} mean or null
	*/
	function accumulator( x ) {
		if ( arguments.length === 0 ) {
			return mean();
		}
		return mean( x*x );
	}
}


// EXPORTS //

module.exports = incrmmeanabs2;
