/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/invgamma/cdf' );
import InvGamma = require( './../../../../../base/dists/invgamma/ctor' );
import entropy = require( './../../../../../base/dists/invgamma/entropy' );
import kurtosis = require( './../../../../../base/dists/invgamma/kurtosis' );
import logpdf = require( './../../../../../base/dists/invgamma/logpdf' );
import mean = require( './../../../../../base/dists/invgamma/mean' );
import mode = require( './../../../../../base/dists/invgamma/mode' );
import pdf = require( './../../../../../base/dists/invgamma/pdf' );
import quantile = require( './../../../../../base/dists/invgamma/quantile' );
import skewness = require( './../../../../../base/dists/invgamma/skewness' );
import stdev = require( './../../../../../base/dists/invgamma/stdev' );
import variance = require( './../../../../../base/dists/invgamma/variance' );

/**
* Interface describing the `invgamma` namespace.
*/
interface Namespace {
	/**
	* Inverse gamma distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 1.0, 8.0, 3.0 );
	* // returns ~0.988
	*
	* y = ns.cdf( 0.0, 1.0, 1.0 );
	* // returns 0.0
	*
	* var mycdf = ns.cdf.factory( 2.0, 0.5 );
	* y = mycdf( 0.5 );
	* // returns ~0.736
	*
	* y = mycdf( 2.0 );
	* // returns ~0.973
	*/
	cdf: typeof cdf;

	/**
	* Inverse gamma distribution.
	*/
	InvGamma: typeof InvGamma;

	/**
	* Returns the differential entropy of an inverse gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 1.0, 1.0 );
	* // returns ~2.154
	*
	* @example
	* var v = ns.entropy( 4.0, 12.0 );
	* // returns ~1.996
	*
	* @example
	* var v = ns.entropy( 8.0, 2.0 );
	* // returns ~-0.922
	*
	* @example
	* var v = ns.entropy( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of an inverse gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns kurtosis
	*
	* @example
	* var v = ns.kurtosis( 6.0, 1.0 );
	* // returns 19.0
	*
	* @example
	* var v = ns.kurtosis( 5.0, 12.0 );
	* // returns 42.0
	*
	* @example
	* var v = ns.kurtosis( 8.0, 2.0 );
	* // returns ~8.7
	*
	* @example
	* var v = ns.kurtosis( 5.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 6.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Inverse gamma distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 0.5, 1.0 );
	* // returns ~-2.112
	*
	* var mylogpdf = ns.logpdf.factory( 6.0, 7.0 );
	* y = mylogpdf( 2.0 );
	* // returns ~-1.464
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of an inverse gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns 4.0
	*
	* @example
	* var v = ns.mean( 8.0, 2.0 );
	* // returns ~0.286
	*
	* @example
	* var v = ns.mean( 1.0, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the mode of an inverse gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 1.0, 1.0 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mode( 4.0, 12.0 );
	* // returns 2.4
	*
	* @example
	* var v = ns.mode( 8.0, 2.0 );
	* // returns ~0.222
	*
	* @example
	* var v = ns.mode( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Inverse gamma distribution probability density function (PDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.5, 1.0 );
	* // returns ~0.121
	*
	* var myPDF = ns.pdf.factory( 6.0, 7.0 );
	* y = myPDF( 2.0 );
	* // returns ~0.231
	*/
	pdf: typeof pdf;

	/**
	* Inverse gamma distribution quantile function.
	*
	* @param p - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 1.0, 1.0 );
	* // returns ~4.481
	*
	* var myQuantile = ns.quantile.factory( 2.0, 2.0 );
	* y = myQuantile( 0.8 );
	* // returns ~2.426
	*
	* y = myQuantile( 0.4 );
	* // returns ~0.989
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of an inverse gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns ~5.657
	*
	* @example
	* var v = ns.skewness( 8.0, 2.0 );
	* // returns ~1.96
	*
	* @example
	* var v = ns.skewness( 2.0, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 4.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 4.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of an inverse gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 3.0, 5.0 );
	* // returns ~2.5
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~2.828
	*
	* @example
	* var v = ns.stdev( 8.0, 2.0 );
	* // returns ~0.117
	*
	* @example
	* var v = ns.stdev( 3.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 1.5, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 3.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of an inverse gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns variance
	*
	* @example
	* var v = ns.variance( 3.0, 5.0 );
	* // returns ~6.25
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns 8.0
	*
	* @example
	* var v = ns.variance( 8.0, 2.0 );
	* // returns ~0.014
	*
	* @example
	* var v = ns.variance( 3.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 1.5, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 3.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Inverse gamma distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
