/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/betaprime/cdf' );
import BetaPrime = require( './../../../../../base/dists/betaprime/ctor' );
import kurtosis = require( './../../../../../base/dists/betaprime/kurtosis' );
import logcdf = require( './../../../../../base/dists/betaprime/logcdf' );
import logpdf = require( './../../../../../base/dists/betaprime/logpdf' );
import mean = require( './../../../../../base/dists/betaprime/mean' );
import mode = require( './../../../../../base/dists/betaprime/mode' );
import pdf = require( './../../../../../base/dists/betaprime/pdf' );
import quantile = require( './../../../../../base/dists/betaprime/quantile' );
import skewness = require( './../../../../../base/dists/betaprime/skewness' );
import stdev = require( './../../../../../base/dists/betaprime/stdev' );
import variance = require( './../../../../../base/dists/betaprime/variance' );

/**
* Interface describing the `betaprime` namespace.
*/
interface Namespace {
	/**
	* Beta prime distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 0.5, 1.0, 1.0 );
	* // returns ~0.333
	*
	* y = ns.cdf( 0.5, 2.0, 4.0 );
	* // returns ~0.539
	*
	* var myCDF = ns.cdf.factory( 0.5, 0.5 );
	*
	* var y = myCDF( 0.8 );
	* // returns ~0.465
	*
	* y = myCDF( 0.3 );
	* // returns ~0.319
	*/
	cdf: typeof cdf;

	/**
	* Beta prime distribution.
	*/
	BetaPrime: typeof BetaPrime;

	/**
	* Returns the excess kurtosis of a beta prime distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 4`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns kurtosis
	*
	* @example
	* var v = ns.kurtosis( 2.0, 6.0 );
	* // returns ~26.143
	*
	* @example
	* var v = ns.kurtosis( 4.0, 12.0 );
	* // returns ~5.764
	*
	* @example
	* var v = ns.kurtosis( 12.0, 6.0 );
	* // returns ~19.49
	*
	* @example
	* var v = ns.kurtosis( 2.0, 4.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Beta prime distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 0.5, 1.0, 1.0 );
	* // returns ~-1.099
	*
	* y = ns.logcdf( 0.5, 2.0, 4.0 );
	* // returns ~-0.618
	*
	* var mylogcdf = ns.logcdf.factory( 0.5, 0.5 );
	*
	* var y = mylogcdf( 0.8 );
	* // returns ~-0.766
	*
	* y = mylogcdf( 0.3 );
	* // returns ~-1.142
	*/
	logcdf: typeof logcdf;

	/**
	* Beta prime distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 0.5, 1.0, 1.0 );
	* // returns ~-0.811
	*
	* y = ns.logpdf( 0.5, 2.0, 4.0 );
	* // returns ~-0.13
	*
	* var mylogpdf = ns.logpdf.factory( 0.5, 0.5 );
	*
	* var y = mylogpdf( 0.8 );
	* // returns ~-0.228
	*
	* y = mylogpdf( 0.3 );
	* // returns ~-0.364
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a beta prime distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 1`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 1.0, 2.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns ~0.364
	*
	* @example
	* var v = ns.mean( 8.0, 2.0 );
	* // returns 8.0
	*
	* @example
	* var v = ns.mean( 1.0, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the mode of a beta prime distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 1.0, 2.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( 4.0, 12.0 );
	* // returns ~0.231
	*
	* @example
	* var v = ns.mode( 8.0, 2.0 );
	* // returns ~2.333
	*
	* @example
	* var v = ns.mode( 1.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Beta prime distribution probability density function (PDF).
	*
	* @param x - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 0.5, 1.0, 1.0 );
	* // returns ~0.444
	*
	* y = ns.pdf( 0.5, 2.0, 4.0 );
	* // returns ~0.878
	*
	* var mypdf = ns.pdf.factory( 0.5, 0.5 );
	*
	* var y = mypdf( 0.8 );
	* // returns ~0.198
	*
	* y = mypdf( 0.3 );
	* // returns ~0.447
	*/
	pdf: typeof pdf;

	/**
	* Beta prime distribution quantile function.
	*
	* @param p - input value
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 2.0, 1.0 );
	* // returns ~8.472
	*
	* y = ns.quantile( 0.5, 4.0, 2.0 );
	* // returns ~2.187
	*
	* var myQuantile = ns.quantile.factory( 2.0, 2.0 );
	*
	* var y = myQuantile( 0.8 );
	* // returns ~2.482
	*
	* y = myQuantile( 0.4 );
	* // returns ~0.763
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a beta prime distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 3`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 2.0, 4.0 );
	* // returns ~6.261
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns ~1.724
	*
	* @example
	* var v = ns.skewness( 8.0, 4.0 );
	* // returns ~5.729
	*
	* @example
	* var v = ns.skewness( 2.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a beta prime distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 2`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 1.0, 3.0 );
	* // returns ~0.866
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~0.223
	*
	* @example
	* var v = ns.stdev( 8.0, 2.5 );
	* // returns ~8.219
	*
	* @example
	* var v = ns.stdev( 8.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a beta prime distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 2`, the function returns `NaN`.
	*
	* @param alpha - first shape parameter
	* @param beta - second shape parameter
	* @returns variance
	*
	* @example
	* var v = ns.variance( 1.0, 3.0 );
	* // returns ~0.75
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns ~0.05
	*
	* @example
	* var v = ns.variance( 8.0, 2.5 );
	* // returns ~67.556
	*
	* @example
	* var v = ns.variance( 8.0, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Beta prime distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
