/**
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var dtypes = require( './../../../dtypes' );
var bufferCtors = require( './../../../base/buffer-ctors' );
var isFunction = require( '@stdlib/assert/is-function' );
var dtypeEnum = require( './../lib' );


// VARIABLES //

var DTYPES = dtypes();
var TABLE = dtypes.enum();


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.strictEqual( typeof dtypeEnum, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns the data type enumeration constant for ndarray data buffers', function test( t ) {
	var ctor;
	var buf;
	var dt;
	var i;

	for ( i = 0; i < DTYPES.length; i++ ) {
		ctor = bufferCtors( DTYPES[ i ] );
		if ( DTYPES[ i ] === 'binary' && isFunction( ctor.alloc ) ) {
			buf = ctor.alloc( 10 );
		} else {
			buf = new ctor( 10 );
		}
		dt = dtypeEnum( buf );
		t.strictEqual( dt, TABLE[ DTYPES[ i ] ], 'returns expected value' );
	}
	t.end();
});

tape( 'the function supports generic objects', function test( t ) {
	var buf;
	var dt;

	buf = {
		'length': 10
	};
	dt = dtypeEnum( buf );
	t.strictEqual( dt, TABLE[ 'generic' ], 'returns expected value' );
	t.end();
});

tape( 'if provided an ndarray data buffer having an unknown/unsupported data type, the function returns `null`', function test( t ) {
	var buffers;
	var i;

	buffers = [
		'beep',
		5,
		NaN,
		null,
		void 0,
		true,
		false,
		function noop() {}
	];
	for ( i = 0; i < buffers.length; i++ ) {
		t.strictEqual( dtypeEnum( buffers[i] ), null, 'returns expected value for ' + buffers[ i ] );
	}
	t.end();
});
