/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var hasGeneratorSupport = require( './../../has-generator-support' );
var nativeClass = require( '@stdlib/utils/native-class' );


// VARIABLES //

var hasGenerators = hasGeneratorSupport();


// MAIN //

/**
* Tests if a value is a generator object.
*
* @param {*} value - value to test
* @returns {boolean} boolean indicating whether value is a generator object
*
* @example
* function* generateID() {
*     var idx = 0;
*     while ( idx < idx+1 ) {
*         yield idx;
*         idx += 1;
*     }
* }
* var bool = isGeneratorObject( generateID() );
* // returns true
*
* @example
* var bool = isGeneratorObject( {} );
* // returns false
*
* @example
* var bool = isGeneratorObject( null );
* // returns false
*/
function isGeneratorObject( value ) {
	return ( hasGenerators && nativeClass( value ) === '[object Generator]' );
}


// EXPORTS //

module.exports = isGeneratorObject;
