/*
 * NASPRO - The NASPRO Architecture for Sound PROcessing
 * Core library
 *
 * Copyright (C) 2007-2013 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

/*
   Title: Threads

   Threading.
 */

#ifndef _NASPRO_CORE_THREAD_H
#define _NASPRO_CORE_THREAD_H

#ifndef _NASPRO_CORE_LIB_H
# error Only <NASPRO/core/lib.h> can be included directly.
#endif

NACORE_BEGIN_C_DECLS

#ifdef _WIN32
typedef struct
  {
	HANDLE		handle;
	DWORD		id;
  } nacore_thread;
#else
typedef pthread_t nacore_thread;
#endif

/*
   Type: nacore_thread

   Thread identifier.

   It is unspecified whether nacore_thread is a pointer, a structure or
   whatever.

   Two thread identifiers can only be compared using <nacore_thread_equal()>.
 */

/*
   Function: nacore_thread_create()

   Creates a new thread.

   Parameters:

     thread		- Memory location where to put a thread identifier.
     size		- Stack size for the new thread or 0 to indicate the
			  default stack size.
     start_routine	- Function to be executed by the new thread.
     arg		- Argument to be passed to start_routine or NULL.

   Returns:

     0 on success, ENOMEM if there was not enough memory, EINVAL if the system
     cannot allocate the requested stack size, EAGAIN if the system lacked the
     necessary resources or EPERM if the caller does not have appropriate
     permission.
 */
_NACORE_DEF int
nacore_thread_create(nacore_thread *thread, size_t stack_size,
		     void (*start_routine)(void *arg), void *arg);

/*
   Function: nacore_thread_exit()

   Terminates the calling thread.
 */
_NACORE_DEF void
nacore_thread_exit();

/*
   Function: nacore_thread_self()

   Gets an identifier for the calling thread.

   Returns:

     Thread identifier for the calling thread.
 */
_NACORE_DEF nacore_thread
nacore_thread_self();

/*
   Function: nacore_thread_join()

   Waits for termination of the specified thread.

   Parameters:

     thread	- Thread identifier.

   Returns:

     0 on success. Depending on the platform, in case of error it may return
     EDEADLK if thread is the calling thread or if a deadlock was detected,
     EINVAL if the specified thread is not joinable or another thread is already
     waiting to join with it, ESRCH if no thead corresponding to the given
     thread identifier was found or <NACORE_EUNKNOWN> if another kind of error
     happened.
 */
_NACORE_DEF int
nacore_thread_join(nacore_thread thread);

/*
   Function: nacore_thread_equal()

   Compares two thread identifiers.

   Parameters:

     t1	- First thread identifier.
     t2	- Second thread identifier.

   Returns:

     A nonzero value if the two identifiers indicate the same thread, 0
     otherwise.
 */
_NACORE_DEF char
nacore_thread_equal(nacore_thread t1, nacore_thread t2);

NACORE_END_C_DECLS

#endif
