/* ***************************************************************************
 *
 * Pico Technology USB Device Driver
 *
 *//**
 * \file      PicoDownloader_MacOSX.cpp 
 * \brief     Background firmware downloader for Pico products: MacOSX-specific implementation
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: PicoDownloader_MacOSX.cpp,v 1.3 2007/05/21 09:10:29 douglas Exp $
 *
 *************************************************************************** */

#include <unistd.h>
#include <pthread.h>

#include <CoreFoundation/CoreFoundation.h>

#include <IOKit/IOKitLib.h>
#include <IOKit/IOCFPlugIn.h>
#include <IOKit/usb/IOUSBLib.h>

#include <mach/mach.h>
#include <sys/wait.h>

#include "IntelHexRecord.h"
#include "PicoUsbID.h"

#include "PicoDownloader_MacOSX.h"

//////////////////////////////////////////////////////////////////////////
// Set the device to configuration zero for FW download
//////////////////////////////////////////////////////////////////////////
static IOReturn ConfigureAnchorDevice(IOUSBDeviceInterface **dev)
{
    UInt8				numConf;
    IOReturn				kr;
    IOUSBConfigurationDescriptorPtr	confDesc;
    
    // get number of configurations (must be at least one)
    kr = (*dev)->GetNumberOfConfigurations(dev, &numConf);
    if (!numConf)
        return -1;
    
    // get the configuration descriptor for index 0
    kr = (*dev)->GetConfigurationDescriptorPtr(dev, 0, &confDesc);
    if (kr) {
#if DEBUG
        printf("\tunable to get config descriptor for index %d (err = %08x)\n", 0, kr);
#endif
		return -1;
    }
	
    // set device to configuration zero
    kr = (*dev)->SetConfiguration(dev, confDesc->bConfigurationValue);
    if (kr)
    {
#if DEBUG
        printf("\tunable to set configuration to value %d (err=%08x)\n", 0, kr);
#endif
        return -1;
    }
    
    return kIOReturnSuccess;
}

//////////////////////////////////////////////////////////////////////////
// Write the contents of writeBuffer to the device
//////////////////////////////////////////////////////////////////////////

IOReturn AnchorWrite(IOUSBDeviceInterface **dev, UInt16 anchorAddress, UInt16 count, UInt8 writeBuffer[])
{
    IOUSBDevRequest 		request;
    
    request.bmRequestType = USBmakebmRequestType(kUSBOut, kUSBVendor, kUSBDevice);
    request.bRequest = 0xa0;
    request.wValue = anchorAddress;
    request.wIndex = 0;
    request.wLength = count;
    request.pData = writeBuffer;
	
    return (*dev)->DeviceRequest(dev, &request);
}

//////////////////////////////////////////////////////////////////////////
// Download firmware to the specified processor type
//////////////////////////////////////////////////////////////////////////

IOReturn DownloadToAnchorDevice(IOUSBDeviceInterface **dev,PicoDownloader::UsbChip chip, INTEL_HEX_RECORD *firmware) {
    UInt8 	writeVal;
    IOReturn	kr;
	
	// Select download method based on processor type
	UInt16 usbReg=0;
	switch(chip) {
		case PicoDownloader::FX:
#if DEBUG
			printf("DownloadToAnchorDevice: Downloading to FX\n");
#endif
			usbReg=kFX_FX2_USBCS;
			break;
		case PicoDownloader::FX2:
#if DEBUG
			printf("DownloadToAnchorDevice: Downloading to FX2\n");
#endif
			usbReg=kFX_FX2_USBCS;
			break;
		case PicoDownloader::FX1:
#if DEBUG
			printf("DownloadToAnchorDevice: Downloading to FX1\n");
#endif
			usbReg=kFX1_FX2LP_USBCS;
			break;
		case PicoDownloader::FX2LP:
#if DEBUG
			printf("DownloadToAnchorDevice: Downloading to FX2LP\n");
#endif
			usbReg=kFX1_FX2LP_USBCS;
			break;
		default:
#if DEBUG
			printf("DownloadToAnchorDevice: Unknown device\n");
#endif
			return kIOReturnBadArgument;
	}
	
    // Assert reset
    writeVal = 1;
    kr = AnchorWrite(dev, usbReg, 1, &writeVal);
    if (kIOReturnSuccess != kr) {
#if DEBUG
        printf("AnchorWrite reset returned err 0x%x!\n", kr);
#endif
		//	Don't do this, the calling function does this on error.
		//        (*dev)->USBDeviceClose(dev);
		//        (*dev)->Release(dev);
        return kr;
    }
    
    
    
    // Download code using AnchorWrite
    do {
		kr=AnchorWrite(dev,firmware->Address,firmware->Length,firmware->Data);
        if (kIOReturnSuccess != kr)  {
#if DEBUG
            printf("AnchorWrite download %lx returned err 0x%x!\n", firmware->Address, kr);
#endif
			//	Don't do this, the calling function does this on error.
			//            (*dev)->USBDeviceClose(dev);
			//            (*dev)->Release(dev);
            return kr;
        }
#if NARRATEDOWNLOAD
        printf("%04lx ",firmware->Address);
#endif
		// The last record has a length of 0
    } while((++firmware)->Type==0);
#if NARRATEDOWNLOAD
    printf("\n");
#endif
	
    // De-assert reset
    writeVal = 0;
    kr = AnchorWrite(dev, usbReg, 1, &writeVal);
    if (kIOReturnSuccess != kr) {
#if DEBUG
        printf("AnchorWrite run returned err 0x%x!\n", kr);
#endif
    }
    
    return kr;
}


//////////////////////////////////////////////////////////////////////////
// This callback function is registered with the IOKIt and called whenever a 
// matching device is added.
//////////////////////////////////////////////////////////////////////////

void PicoDownloader_MacOSX::callbackProcNewDeviceAdded(void *args,io_iterator_t iterator) {
#if DEBUG
	printf("PicoDownloader::callbackProcNewDeviceAdded(%p)\n",args);
#endif
	
	if(args!=NULL) {
		((PicoDownloader_MacOSX *)args)->NewDeviceAdded(iterator);
	}
#if DEBUG
	else {
		printf("PicoDownloader::callbackProcNewDeviceAdded: args==NULL\n");
	}
#endif
}

//////////////////////////////////////////////////////////////////////////
// Create interface for a new device and download firmware
// This is called by the downloader callback when a new device is detected
//////////////////////////////////////////////////////////////////////////

void PicoDownloader_MacOSX::NewDeviceAdded(io_iterator_t iterator) {
	
    IOReturn				err;
    IOReturn				releaseDeviceError;
	io_service_t			usbDevice;
    IOCFPlugInInterface 	**plugInInterface=NULL;
    IOUSBDeviceInterface 	**dev=NULL;
	
    HRESULT 		res;
    SInt32 			score;
    UInt16			vendor;
    UInt16			deviceProduct;
    UInt16			deviceRelease;
    int exclusiveErr = 0;
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded()\n", product, release, chip);
#endif
	
	// If we haven't been passed a valid iterator, give up.
	if (!iterator){
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Invalid iterator, giving up.\n", product, release, chip);
#endif
		return;
	}
	
	while (usbDevice = IOIteratorNext(iterator)) {
		// Above while statement guarantees USB Device isn't null so don't need to check
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Got device, creating plugin\n", product, release, chip);
#endif
		// Get plugin for the device object
		err = IOCreatePlugInInterfaceForService(usbDevice, kIOUSBDeviceUserClientTypeID, kIOCFPlugInInterfaceID, &plugInInterface, &score);
		// Done with the device object now that I have the plugin
		// Delete it regardless of whether an error occurred above
		releaseDeviceError = IOObjectRelease(usbDevice);
		if (releaseDeviceError != kIOReturnSuccess) {
#if DEBUG
			// If releasing the device causes an error, log it but don't do anything
            printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: unable to release device (%08x)\n", product, release, chip, releaseDeviceError);
#endif
		}
		// Check to see whether the plugin was successfully created
        if ((kIOReturnSuccess != err) || !plugInInterface) {
#if DEBUG
            printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: unable to create a plugin (%08x)\n", product, release, chip, err);
			if (err == kIOReturnNoResources) 
				// This happens quite a bit if the device has been renumerated by another downloader in the meantime
				printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Error is kIOReturnNoResources - device has probably disconnected or renumerated\n", product, release, chip, err);
			else
				printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Unrecognised error \n", product, release, chip);			
#endif
            continue;
        }
		
        // I have the device plugin, I need the device interface
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Creating device interface\n", product, release, chip);
#endif
        res = (*plugInInterface)->QueryInterface(plugInInterface,CFUUIDGetUUIDBytes(kIOUSBDeviceInterfaceID),(LPVOID*)&dev);
		
		// Now that we have the interface we can delete the plugin
		err = IODestroyPlugInInterface(plugInInterface);			// done with this
		if (err != kIOReturnSuccess) {
#if DEBUG
			// If releasing the device causes an error, log it but don't do anything
            printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: unable to destroy interface, error (%08x)\n", product, release, chip, err);
#endif
		}
		
		
        if (res || !dev) {
#if DEBUG
            printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Couldn't create a device interface (%08x)\n", product, release, chip, (int) res);
#endif
            continue;
        }
        
		// Get PID, VID and DID for the new device
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Getting vendor\n", product, release, chip);
#endif
        err = (*dev)->GetDeviceVendor(dev, &vendor);
		if (kIOReturnSuccess != err) {
#if DEBUG
			printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Error getting vendor\n", product, release, chip);
#endif
			err = (*dev)->Release(dev);
			continue;
		}
#if DEBUG
        printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Getting product\n", product, release, chip);
#endif
		err = (*dev)->GetDeviceProduct(dev, &deviceProduct);
		if (kIOReturnSuccess != err) {
#if DEBUG
			printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Error getting product\n", product, release, chip);
#endif
			err = (*dev)->Release(dev);
			continue;
		}
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Getting release\n", product, release, chip);
#endif
        err = (*dev)->GetDeviceReleaseNumber(dev, &deviceRelease);
		if (kIOReturnSuccess != err) {
#if DEBUG
			printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Error getting release\n", product, release, chip);
#endif
			err = (*dev)->Release(dev);
			continue;
		}
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Vendor %04hx Product %04hx Release %04hx\n", product, release, chip, vendor,product,release);
#endif
		
		// Only open devices which match our PID, DID and VID
		if((vendor==VENDOR_ID_PICO_TECHNOLOGY)&&(deviceProduct==this->product)&&(deviceRelease==this->release)) {
#if DEBUG
			printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Downloading firmware\n", product, release, chip);
#endif
			
			// need to open the device in order to change its state
			do{
				err = (*dev)->USBDeviceOpen(dev);
				// Exclusive access error occurs if someone else has the device open
				// Retry a few times if this is the case
				if(kIOReturnExclusiveAccess == err)
				{
					exclusiveErr++;
#if DEBUG
					printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Exclusive access err, sleeping on it %d\n", product, release, chip, exclusiveErr);
#endif
					sleep(1);
				}
			}while( (kIOReturnExclusiveAccess == err) && (exclusiveErr < 5) );
			
			if (kIOReturnSuccess != err) {
#if DEBUG
				printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Unable to open device: %08x\n", product, release, chip, err);
#endif
				(*dev)->Release(dev);
				continue;
			}
			
			// Configure device (select config descr zero)
			err = ConfigureAnchorDevice(dev);
			if (kIOReturnSuccess != err) {
#if DEBUG
				printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Unable to configure device: %08x\n", product, release, chip, err);
#endif
				(*dev)->USBDeviceClose(dev);
				(*dev)->Release(dev);
				continue;
			}
			
			// Download the FW
			err = DownloadToAnchorDevice(dev,this->chip,this->firmware);
			if (kIOReturnSuccess != err) {
#if DEBUG
				printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Unable to download to device: %08x\n", product, release, chip, err);
#endif
				(*dev)->USBDeviceClose(dev);
				(*dev)->Release(dev);
				continue;
			}
			
			// Close the device
			err = (*dev)->USBDeviceClose(dev);
			
			
			pthread_mutex_lock(&initialDeviceMutex);
			if (!initialized) {
				initialDevices++;
			}
			pthread_mutex_unlock(&initialDeviceMutex);
			
		} else {
#if DEBUG
			printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: Not a matching device\n", product, release, chip);
#endif
		}
		
        err = (*dev)->Release(dev);
	}
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::NewDeviceAdded: End\n", product, release, chip);
#endif
}

//////////////////////////////////////////////////////////////////////////
// threadProc is started in a new thread by the PicoDownloader ctor
// and just calls the DownloadThread method 
//////////////////////////////////////////////////////////////////////////
void *PicoDownloader_MacOSX::threadProc(void *args) {
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::threadProc(%p)\n",args);
#endif
	
	if(args!=NULL) {
		((PicoDownloader_MacOSX *)args)->DownloadThread();
	}
	return NULL;
}


//////////////////////////////////////////////////////////////////////////
// Start downloader run loop
//////////////////////////////////////////////////////////////////////////
void PicoDownloader_MacOSX::DownloadThread (void) {
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::DownloadThread()\n", product, release, chip);
#endif
	
	runLoop=CFRunLoopGetCurrent();
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::DownloadThread: Trying to setup notification\n", product, release, chip);
#endif
	
	// Set up notifications for devices matching our PID and VID
	if(SetupNotification()==0) {
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::DownloadThread: Successfull setup of notification\n", product, release, chip);
#endif
		
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::DownloadThread: Before CFRunLoopRun\n", product, release, chip);
#endif
		
		// Start the run loop. Now we'll receive notifications.
		CFRunLoopRun();
		
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::DownloadThread: After CFRunLoopRun\n", product, release, chip);
#endif
	} else {
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::DownloadThread: Failed to setup notification\n", product, release, chip);
#endif
	}
	
	runLoop=0;
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::DownloadThread: End of thread\n", product, release, chip);
#endif
}


//////////////////////////////////////////////////////////////////////////
// Register with the IO Kit to get notifications when matching
// devices are connected
//////////////////////////////////////////////////////////////////////////
int PicoDownloader_MacOSX::SetupNotification () {
    mach_port_t 		masterPort;
    CFMutableDictionaryRef 	matchingDict;
    CFRunLoopSourceRef		runLoopSource;
    kern_return_t		kr;
	
    SInt32			usbVendor = VENDOR_ID_PICO_TECHNOLOGY;
    SInt32			usbProduct = this->product;
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::SetupNotification()\n", product, release, chip);
	printf("\tLooking for devices matching vendor ID=%ld, product ID=%ld and release ID=%ld\n", product, release, chip,usbVendor ,usbProduct, this->release);
#endif
	
    // first create a master_port for my task
    kr = IOMasterPort(MACH_PORT_NULL, &masterPort);
    if (kr || !masterPort) {
#if DEBUG
        printf("PicoDownloader (%04x, %d, %d)::SetupNotification: Couldn't create a master IOKit Port(%08x)\n", product, release, chip, kr);
#endif
        return -1;
    }
	
    // Set up the matching criteria for the devices we're interested in
    matchingDict = IOServiceMatching(kIOUSBDeviceClassName);	// Interested in instances of class IOUSBDevice and its subclasses
    if (!matchingDict) {
#if DEBUG
        printf("PicoDownloader (%04x, %d, %d)::SetupNotification: Can't create a USB matching dictionary\n", product, release, chip);
#endif
        mach_port_deallocate(mach_task_self(), masterPort);
        return -1;
    }
    
    // Add our vendor and product IDs to the matching criteria
    CFDictionarySetValue( 
						  matchingDict, 
						  CFSTR(kUSBVendorID), 
						  CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &usbVendor)); 
    CFDictionarySetValue( 
						  matchingDict, 
						  CFSTR(kUSBProductID), 
						  CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &usbProduct)); 
	
    // Create a notification port and add its run loop event source to our run loop
    // This is how async notifications get set up.
    this->notifyPort = IONotificationPortCreate(masterPort);
    runLoopSource = IONotificationPortGetRunLoopSource(this->notifyPort);
    
    CFRunLoopAddSource(CFRunLoopGetCurrent(), runLoopSource, kCFRunLoopDefaultMode);
    
    // Retain additional references because we use this same dictionary with four calls to 
    // IOServiceAddMatchingNotification, each of which consumes one reference.
    matchingDict = (CFMutableDictionaryRef) CFRetain( matchingDict ); 
    matchingDict = (CFMutableDictionaryRef) CFRetain( matchingDict ); 
    matchingDict = (CFMutableDictionaryRef) CFRetain( matchingDict ); 
    
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::SetupNotification: Adding notification.\n", product, release, chip);
#endif
    // Now set up notification to be called when a clean device is first matched by I/O Kit.
	kr = IOServiceAddMatchingNotification(  this->notifyPort,
                                            kIOFirstMatchNotification,
                                            matchingDict,
                                            callbackProcNewDeviceAdded,
                                            this,
                                            &(this->deviceAddedIter));
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::SetupNotification: Looking for devices.\n", product, release, chip);
#endif
    NewDeviceAdded(this->deviceAddedIter);	// Iterate once to get already-present devices and
											// arm the notification
	
	pthread_mutex_lock(&initialDeviceMutex);
	initialized = true;
	pthread_mutex_unlock(&initialDeviceMutex);
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::SetupNotification: Finished looking for devices.\n", product, release, chip);
#endif
	
	
    // Now done with the master_port
    mach_port_deallocate(mach_task_self(), masterPort);
    masterPort = 0;
	
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::SetupNotification: End.\n", product, release, chip);
#endif
	
	return 0;
}

//////////////////////////////////////////////////////////////////////////
// Returns a count of already-connected devices to which we have downloaded firmware
// or -1 if we haven't finished downloding firmware yet.
//////////////////////////////////////////////////////////////////////////
int PicoDownloader_MacOSX::InitialDeviceCount(void)
{
			int deviceCount = -1;
			pthread_mutex_lock(&initialDeviceMutex);
			if (initialized) {
				deviceCount = initialDevices;
			}
			pthread_mutex_unlock(&initialDeviceMutex);
			return deviceCount;
}


//////////////////////////////////////////////////////////////////////////
// Downloader class constructor. 
//////////////////////////////////////////////////////////////////////////
PicoDownloader_MacOSX::PicoDownloader_MacOSX(unsigned short product,unsigned short release,INTEL_HEX_RECORD *firmware,PicoDownloader::UsbChip chip) {
	int err;
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::PicoDownloader()\n", product, release, chip);
#endif
	
	// Save our parameters
	this->product=product;
	this->release=release;
	this->firmware=firmware;
	this->chip=chip;
	
	runLoop = NULL;
	initialDevices = 0;
	initialized = false;
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::PicoDownloader: Creating thread.\n", product, release, chip);
#endif
	thread=new pthread_t();
	
	// Start the polling thread
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::PicoDownloader: Starting thread...", product, release, chip);
#endif
	err=pthread_create(thread,NULL,threadProc,this);
#if DEBUG
	if(err) {
		printf("failed.\n");
	} else {
		printf("ok.\n");
	}
#endif
}

//////////////////////////////////////////////////////////////////////////
// Downloader class destructor. 
//////////////////////////////////////////////////////////////////////////
PicoDownloader_MacOSX::~PicoDownloader_MacOSX() {
	int err;
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader()\n", product, release, chip);
#endif
	
	// Stop the run loop
	if(runLoop!=NULL) {
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: Stopping RunLoop...", product, release, chip);
#endif
		CFRunLoopStop(runLoop);
#if DEBUG
		printf("ok\n");
#endif
	} else {
#if DEBUG
		printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: RunLoop already stopped.\n", product, release, chip);
#endif
	}
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: Joining thread...\n", product, release, chip);
#endif
	// Ensure that the thread exits safely
	err=pthread_join(*thread,NULL);
#if DEBUG
	if(err) {
		printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: Joining thread...error!\n", product, release, chip);
		if(err==EINVAL) {
			printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: Thread does not refer to a joinable thread.\n", product, release, chip);
		} else if(err==ESRCH) {
			printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: Thread could not be found.\n", product, release, chip);
		} else if(err==EDEADLK) {
			printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: A deadlock was detected.\n", product, release, chip);
		}
	} else {
		printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: Joining thread...ok!\n", product, release, chip);
	}
#endif
	
#if DEBUG
	printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: Detaching thread...", product, release, chip);
#endif
	err=pthread_detach(*thread);
#if DEBUG
	if(err) {
		printf("error!\n");
		if(err==EINVAL) {
			printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: Thread does not refer to a joinable thread.\n", product, release, chip);
		} else if(err==ESRCH) {
			printf("PicoDownloader (%04x, %d, %d)::~PicoDownloader: Thread could not be found.\n", product, release, chip);
		}
	} else {
		printf("ok!\n");
	}
#endif
}


