////////////////////////////////////////////////////////////////////////////////
//
//	Pico Technology USB TC08 Driver
//
///	\file     TC08Device.h
///	\brief    TC08 device class providing the C++ API to the driver
//
//	Copyright (c) 2007, Pico Technology.
//	All rights reserved.
//   
//	Redistribution and use in source and binary forms, with or without
//	modification, are permitted provided that the following conditions are met:
//		* Redistributions of source code must retain the above copyright
//		  notice, this list of conditions and the following disclaimer.
//		* Redistributions in binary form must reproduce the above copyright
//		  notice, this list of conditions and the following disclaimer in the
//		  documentation and/or other materials provided with the distribution.
//		* The name of Pico Technology may not be used to endorse or promote
//		  products derived from this software without specific prior written
//		  permission.
//
//	THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY ``AS IS'' AND ANY
//	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
//	DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//	ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//	THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//	Version $Id: TC08Device.h,v 1.8 2007/08/01 21:39:57 douglas Exp $
//
////////////////////////////////////////////////////////////////////////////////

// Avoid including this header more than once
#ifndef TC08DEVICE_H
#define TC08DEVICE_H

// TC08Api.h contains type definitions used in both the C and C++ APIs
#include "TC08Api.h"
#include "PicoStatus.h"


////////////////////////////////////////////////////////////////////////////////
// Structs used only by the C++ API
////////////////////////////////////////////////////////////////////////////////

/// Version information
typedef struct {
	unsigned char Major;
	unsigned char Minor;
	unsigned char Revision;
} USBTC08_VERSION;

/// Unit info returned by GetUnitInfo
typedef struct {
	short HardwareVersion;
	char Serial[USBTC08_MAX_SERIAL_CHARS];
	char CalDate[USBTC08_MAX_DATE_CHARS];
	USBTC08_VERSION FirmwareVersion;
} USBTC08_UNIT_INFO;



////////////////////////////////////////////////////////////////////////////////
//  TC08Device class
/// Provides the C++ API through which user code interacts with the TC08.
////////////////////////////////////////////////////////////////////////////////

class TC08Device {
	// Public methods
	public:
		// Constructor and destructor
		TC08Device(class PicoUsbDevice * device);
		~TC08Device();

		// Opening and closing the unit
		USBTC08_STATES  Open(void);
		PICO_RETURNS Close(void);

		// Unit status and information methods 
		PICO_RETURNS      Ping (void);
		USBTC08_STATES    GetDeviceState(void);
		const USBTC08_UNIT_INFO * GetUnitInfo(void);
		const char *      GetSerialString(void);
		short             GetHandle(void);
		
		// Unit setup methods
		PICO_RETURNS SetLED(USBTC08_LED_STATES ledState);
		PICO_RETURNS SetMains(USBTC08_MAINS_FREQUENCY mains_frequency);		
		PICO_RETURNS SetChannelConfig(USBTC08_CHANNELS channel,
		                              USBTC08_TCTYPES type);
		USBTC08_TCTYPES GetChannelConfig(USBTC08_CHANNELS channel);		
		PICO_RETURNS    SetChannelConfig(unsigned int channel,char type);
		char            GetChannelConfig(unsigned int channel);

		// Acquiring readings
		PICO_RETURNS  GetSingle(float *temp,unsigned short *overflow_flags,
		                        USBTC08_UNITS units);
		unsigned int  GetMinimumInterval(void);
		unsigned int  GetMaximumInterval(void);
		PICO_RETURNS  Run(unsigned int interval_ms);
		PICO_RETURNS  GetValues(float **temp_buffer,unsigned int *lengths,
		                        unsigned short *overflow,USBTC08_UNITS units,
		                        bool fill_missing);
		PICO_RETURNS  GetTimesAndValues(float **temp_buffer, long**times_buffer,
		                                unsigned int *lengths,
		                                unsigned short *overflow,
		                                USBTC08_UNITS units,
		                                bool fill_missing, bool deskew_times);
		PICO_RETURNS  GetSingleChannelValues(USBTC08_CHANNELS channel, 
		                                     float *temp_buffer,
		                                     unsigned int * length,
		                                     unsigned short *overflow,
		                                     USBTC08_UNITS units,
		                                     bool fill_missing);
		PICO_RETURNS  GetSingleChannelTimesAndValues(USBTC08_CHANNELS channel, 
		                                             float *temp_buffer,
		                                             long* times_buffer,
		                                             unsigned int *length,
		                                             unsigned short *overflow,
		                                             USBTC08_UNITS units,
		                                             bool fill_missing, 
		                                             bool deskew_times);
		PICO_RETURNS  Stop(void);

	// Internal state is stored in a private class
	// Avoids needless API changes when internal implementation is changed
	private:
		class TC08DeviceInternalData * dev;
};

#endif // not defined TC08DEVICE_H
