////////////////////////////////////////////////////////////////////////////////
//
//	Pico Technology USB TC08 Driver
//
///	\file     TC08Device.cpp
///	\brief    TC08 device class providing the C++ API to the driver
//
//	Copyright (c) 2007, Pico Technology.
//	All rights reserved.
//   
//	Redistribution and use in source and binary forms, with or without
//	modification, are permitted provided that the following conditions are met:
//		* Redistributions of source code must retain the above copyright
//		  notice, this list of conditions and the following disclaimer.
//		* Redistributions in binary form must reproduce the above copyright
//		  notice, this list of conditions and the following disclaimer in the
//		  documentation and/or other materials provided with the distribution.
//		* The name of Pico Technology may not be used to endorse or promote
//		  products derived from this software without specific prior written
//		  permission.
//
//	THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY ``AS IS'' AND ANY
//	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
//	DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//	ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//	THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//	Version $Id: TC08Device.cpp,v 1.15 2008/02/19 20:12:41 douglas Exp $
//
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//	INCLUDES
////////////////////////////////////////////////////////////////////////////////

#include <assert.h>
#include <stdio.h>
#include <ctype.h>
#include <limits>
#include <math.h>
#include <pthread.h>
#include <unistd.h>

#include <sys/times.h>

#include "../shared/PicoStatus.h"
#include "../shared/PicoPortability.h"
#include "../shared/PicoUsbID.h"
#include "../shared/PicoUsbDevice.h"

#include "TC08Device.h"
#include "TC08DeviceInternalData.h"
#include "readingbuff.h"
#include "tctables.h"


// TC08Device implementation

//////////////////////////////////////////////////////////////////////////
/// Read the unit info from the EEPROM into a struct.
/// Reads back 30 bytes from the unit over USB:
/// <ul><li>1 byte: EEPROM Read Status 0x00 = Success 0xFF = Fail
/// <li>1 bytes: EEPROM version (not used in info struct)
/// <li>2 bytes: HW version
/// <li>8 bytes: Reserved
/// <li>10 bytes: Serial number
/// <li>8 bytes: Calibration date </ul>
/// <returns> USBTC08_INFO struct containing the Hardware and Firmware
/// versions, serial number and calibration date of the TC08. Returns NULL
/// if any error occurs while getting the information</returns>
//////////////////////////////////////////////////////////////////////////
const USBTC08_UNIT_INFO *TC08Device::GetUnitInfo(void) {
	unsigned char buf[USBTC08_CMD_BUF_SIZE];
	PICO_RETURNS retval;
	
	// Check the device is open and not streaming
	assert(dev && dev->usbDevice);

#if DEBUG
	printf("TC08Device::GetUnitInfo()\n");
#endif
	
	if(GetDeviceState()!=USBTC08_STATE_IDLE) {
#if DEBUG
		printf("TC08Device::GetUnitInfo(), device not idle\n");
#endif
		return NULL;
	}

	// If we have already got the info, don't get it again
	if(dev->info==NULL) {
		// Check the device ID
		int did = dev->usbDevice->GetDID();
		if((did != DEVICE_ID_USB_TC08MK1_LOADED) && (did != DEVICE_ID_USB_TC08MK2_LOADED)) {
#if DEBUG
	printf("TC08Device::GetUnitInfo(), wrong DID (returned %d)!\n", did);
#endif
			return NULL;
		}
	
		// Issue read unit info command
		buf[0] = USBTC08_CMD_READ_UNIT_INFO;
		if ((retval = dev->WritePipe(buf,1)) != PICO_SUCCESS) {
#if DEBUG
			printf("TC08Device::GetUnitInfo(), write fail, returned %d\n", retval);
#endif
			return NULL;			
		}
		
		// Read the response from the unit
		unsigned int cnt = USBTC08_UNIT_INFO_LENGTH;
		if((retval = dev->ReadPipe(buf,&cnt)) != PICO_SUCCESS) {
#if DEBUG
			printf("TC08Device::GetUnitInfo(), read fail, returned %d, got %d bytes\n", retval, cnt);
#endif
			return NULL;
		}
		

		// Check the success byte
		if (buf[0] != 0) {
#if DEBUG
			printf("TC08Device::GetUnitInfo(), checksum fail\n");
#endif
			return NULL;
		}
		
		// Create the info struct and fill with info from unit
		dev->info=new USBTC08_UNIT_INFO();
		if (!dev->info){
#if DEBUG
			printf("TC08Device::GetUnitInfo(), couldn't create info struct\n");
#endif
			return NULL;			
		}
		
		if(did==DEVICE_ID_USB_TC08MK1_LOADED) {
			dev->info->HardwareVersion = 1;
		} else if(did==DEVICE_ID_USB_TC08MK2_LOADED) {
			dev->info->HardwareVersion = 2;
		} else {
			// This should never happen, we checked DID above
#if DEBUG
			printf("TC08Device::GetUnitInfo(), DID out of sync\n");
#endif
			assert(0);
		}
		
		memcpy(&(dev->info->Serial), buf+12, sizeof(dev->info->Serial) - 1);
		memcpy(&(dev->info->CalDate), buf+22,  sizeof(dev->info->CalDate) - 1);

		// Strings are null-terminated in EEPROM but just make sure
		dev->info->Serial[ sizeof(dev->info->Serial) - 1] = 0;
		dev->info->CalDate[ sizeof(dev->info->CalDate) - 1] = 0;
		
		// Add firmware version
		buf[0] = USBTC08_CMD_GET_FIRMWARE_VERSION;
		if ((retval = dev->WritePipe(buf,1)) != PICO_SUCCESS) {
#if DEBUG
			printf("TC08Device::GetUnitInfo(), write fail, returned %d\n", retval);
#endif
			return NULL;			
		}
		
		cnt = USBTC08_FIRMWARE_VERSION_LENGTH;
		if((retval = dev->ReadPipe(buf,&cnt)) != PICO_SUCCESS) {
#if DEBUG
			printf("TC08Device::GetUnitInfo(), read fail, returned %d, got %d bytes\n", retval, cnt);
#endif
			return NULL;
		}
		memcpy(&(dev->info->FirmwareVersion),buf,USBTC08_FIRMWARE_VERSION_LENGTH);
	}
#if DEBUG
	printf("TC08Device::GetUnitInfo(), returning OK\n");
#endif
	return dev->info;
}


//////////////////////////////////////////////////////////////////////////
/// Constructor - get pipes, init device, set initial settings. 
/// Since this needs a PicoUsbDevice object as a parameter, user code should 
/// never instantiate a TC08Device directly but should instead use TC08Device
/// objects returned by the static methods of TC08Enumerator.
/// <param name="device">Pointer to a PicoUsbDevice for the TC08</param>
//////////////////////////////////////////////////////////////////////////
TC08Device::TC08Device(PicoUsbDevice* device) {
#if DEBUG
	printf("TC08Device::TC08Device(PicoUsbDevice at %p)\n",device);
#endif
	assert(device);
	
	// Create the TC08DeviceInternalData for private data
	dev = new TC08DeviceInternalData();
	
	if (!dev)
	{
		// Failing to create the internal data class is a fatal error
#if DEBUG
		printf("Failed to create new TC08DeviceInternalData\n");
#endif
		assert(0);
	} else {
#if DEBUG
		printf("New TC08DeviceInternalData created OK at %p\n", dev);
#endif
	}
	// Initialize fields
	dev->usbDevice = device;
	
	// If the PicoUsbDevice is open try to open the TC08 too
	if(dev->usbDevice->GetDeviceState() == PICODEVICE_STATE_OPEN) {
		USBTC08_STATES retval = this->Open();
		if (retval != USBTC08_STATE_IDLE){
			// This is a non-critical error: we can always try to open
			// the device again later 
#if DEBUG
			printf("Failed to open the TC08Device, returned %d\n", retval);
#endif
		}
	}
	
	// Init all channels to disabled
	for (short iChannelIndex = 0; iChannelIndex < USBTC08_MAX_CHANNELS+1; ++iChannelIndex)
	{
		dev->Channels[iChannelIndex].fEnabled             = false;
		dev->Channels[iChannelIndex].bTCType              = ' ';
	}
	// The CJC uses 'C' to identify the thermistor scaling
	dev->Channels[USBTC08_CHANNEL_CJC].fEnabled             = false;
	dev->Channels[USBTC08_CHANNEL_CJC].bTCType              = 'C';
	
}

//////////////////////////////////////////////////////////////////////////
/// Destructor - close device and free memory.
//////////////////////////////////////////////////////////////////////////
TC08Device::~TC08Device() {
#if DEBUG
	printf("TC08Device::~TC08Device()\n");
#endif

	// All our data is in the TC08DeviceInternalData so if it's gone, give up
	if (!dev)
	{
#if DEBUG
		printf("TC08Device::~TC08Device: no TC08DeviceInternalData!\n");
#endif
		return;
	}
		
	// Free the info structure
	if(dev->info) {
		delete dev->info;
	}
	
	// Delete the USB device
	if(dev->usbDevice!=NULL) {
#if DEBUG
		//printf("TC08Device::~TC08Device: Device is not null.\n");
#endif
		
		if(dev->usbDevice->GetDeviceState()==PICODEVICE_STATE_OPEN) {
#if DEBUG
			printf("TC08Device::~TC08Device: Device is open.\n");
#endif		
			Close();
		} else {
#if DEBUG
			printf("TC08Device::~TC08Device: Device is closed.\n");
#endif
		}
#if DEBUG
		printf("TC08Device::~TC08Device: Deleting device...\n");
#endif
		delete dev->usbDevice;
		dev->usbDevice = NULL;
	} else {
#if DEBUG
		printf("TC08Device::~TC08Device: Device is already null.\n");
#endif
	}
	

	// Delete the TC08DeviceInternalData
	delete dev;
	dev = NULL;
	
}

//////////////////////////////////////////////////////////////////////////
/// Convert a single centigrade value to another unit.
/// <param name="temp">The temperature in Celsius to be converted</param>
/// <param name="unit">USBTC08_UNITS value of the temperature scale which temp 
/// should be converted to</param>
/// <returns>The value of temp on the chosen temperature scale.</returns>
//////////////////////////////////////////////////////////////////////////
float ConvertFromCentigrade(float temp,USBTC08_UNITS unit) {
	switch (unit) {
		case USBTC08_UNITS_FAHRENHEIT:
			return ConvertToFahrenheit(temp);
			
		case USBTC08_UNITS_KELVIN:
			return ConvertToKelvin(temp);
			
		case USBTC08_UNITS_RANKINE:
			return ConvertToRankine(temp);
			
		default: // Data already stored in centigrade
			return temp;
	}
}

//////////////////////////////////////////////////////////////////////////
/// Convert an array of centigrade values to another unit.
/// <param name="temp">An array of temperatures in Celsius to be converted. The
/// array is converted in place, i.e. each element is replaced by its converted
/// value</param>
/// <param name="length">The number of values in temp to be converted</param>
/// <param name="unit">USBTC08_UNITS value of the temperature scale which temp 
/// should be converted to</param>
//////////////////////////////////////////////////////////////////////////
void ConvertFromCentigradeI(float *temp,int length,USBTC08_UNITS unit) {
	// Validate parameters
	if (length < 1) // We don't have any elements to convert
		return;
	switch (unit) {
		case USBTC08_UNITS_FAHRENHEIT:
			for(int i=0;i<length;i++) {
				temp[i]=ConvertToFahrenheit(temp[i]);
			}
			break;
			
		case USBTC08_UNITS_KELVIN:
			for(int i=0;i<length;i++) {
				temp[i]=ConvertToKelvin(temp[i]);
			}
			break;
			
		case USBTC08_UNITS_RANKINE:
			for(int i=0;i<length;i++) {
				temp[i]=ConvertToRankine(temp[i]);
			}
			break;
			
		default: // Data already stored in centigrade
			break;
	}
}


//////////////////////////////////////////////////////////////////////////
/// Get the device state from the underlying PicoUsbDevice
/// <returns>Value representing the state of the USB device: <ul>
/// <li>USBTC08_STATE_DISCONNECTED if the device has been unplugged since
///  the TC08Device object was created.
/// <li>USBTC08_STATE_CLOSED if the device is either closed (available for opening)
///  or locked (another process has it open)
/// <li>USBTC08_STATE_STREAMING if the device is streaming, or
/// <li>USBTC08_STATE_IDLE if the device is open and not streaming</ul></returns>
//////////////////////////////////////////////////////////////////////////
USBTC08_STATES TC08Device::GetDeviceState() {
	
	assert(dev && dev->usbDevice);

	PICODEVICE_STATES baseState=dev->usbDevice->GetDeviceState();
#if DEBUG
	printf("TC08Device::GetDeviceState. State is %d\n", baseState);
#endif

	if(baseState==PICODEVICE_STATE_DISCONNECTED) {
		return dev->state=USBTC08_STATE_DISCONNECTED;
	} else if(baseState==PICODEVICE_STATE_LOCKED) {
		return dev->state=USBTC08_STATE_CLOSED;
	} else if(baseState==PICODEVICE_STATE_CLOSED) {
		return dev->state=USBTC08_STATE_CLOSED;
	} else {
		assert (dev->state == USBTC08_STATE_IDLE || dev->state == USBTC08_STATE_STREAMING);
		return dev->state;
	}
}

//////////////////////////////////////////////////////////////////////////
/// Set LED to red/green/off.
/// Note that other commands (for example, collecting a reading or closing
/// the unit) will reset the LED state.
/// <param name="ledState">USBTC08_LED_STATE value to set the LED to</param>
/// <returns>PICO_RETURNS value: <ul>
/// <li>PICO_UNIT_STATE_ERROR if the device is not open, or it is streaming
/// <li>PICO_ARGUMENT_ERROR if ledState was invalid,
/// <li>PICO_DEVICE_DISCONNECTED if the command could not be sent to the 
///  unit successfully (probably means the unit has been unplugged
/// <li>PICO_SUCCESS if the call succeded</ul></returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::SetLED(USBTC08_LED_STATES ledState) {
	unsigned char buf[USBTC08_CMD_BUF_SIZE];
	PICO_RETURNS retval;
	
  // Check the device is open and not streaming
	assert(dev && dev->usbDevice);
	if(GetDeviceState()!=USBTC08_STATE_IDLE) {
		return PICO_UNIT_STATE_ERROR;
	}

  // Validate arguments
	if((ledState<0)||(ledState>=USBTC08_LED_MAX)) {
		return PICO_ARGUMENT_ERROR;
	}
	
	buf[0] = USBTC08_CMD_SET_LED;
	buf[1] = (UInt8)ledState;
	
	if ((retval = dev->WritePipe(buf,2)) != PICO_SUCCESS) {
#if DEBUG
		printf("TC08Device::SetLED(), write fail, returned %d\n", retval);
#endif
		return PICO_DEVICE_DISCONNECTED;
	}
	
	return PICO_SUCCESS;
}


//////////////////////////////////////////////////////////////////////////
/// Sets the mains frequency for on-board noise rejection.
/// <param name="mains_frequency">USBTC08_MAINS_FREQUENCY value to set</param>
/// <returns>PICO_RETURNS value: <ul>
/// <li>PICO_UNIT_STATE_ERROR if the device is not open, or it is streaming
/// <li>PICON_ARGUMENT_ERROR if mains_frequency was invalid,
/// <li>PICO_DEVICE_DISCONNECTED if the command could not be sent to the 
///  unit successfully (probably means the unit has been unplugged
/// <li>PICO_SUCCESS if the call succeded</ul></returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::SetMains(USBTC08_MAINS_FREQUENCY mains_frequency){

	unsigned char buf[USBTC08_CMD_BUF_SIZE];
	PICO_RETURNS retval;
	
  // Check the device is open and not streaming
	assert(dev && dev->usbDevice);
	if(GetDeviceState()!=USBTC08_STATE_IDLE) {
		return PICO_UNIT_STATE_ERROR;
	}

  // Validate arguments
	if((mains_frequency<0)||(mains_frequency>=USBTC08_MAINS_MAX)) {
		return PICO_ARGUMENT_ERROR;
	}
	
	buf[0] = USBTC08_CMD_SET_MAINS;
	buf[1] = (UInt8)mains_frequency;
	
	if ((retval = dev->WritePipe(buf,2)) != PICO_SUCCESS) {
#if DEBUG
		printf("TC08Device::SetMAINS(), write fail, returned %d\n", retval);
#endif
		return PICO_DEVICE_DISCONNECTED;
	}
	
	return PICO_SUCCESS;
}


//////////////////////////////////////////////////////////////////////////
/// Send a command to the TC08 unit and wait for a reply
/// <returns>PICO_RETURNS value: <ul>
/// <li>PICO_UNIT_STATE_ERROR if the device is not open, or it is streaming
/// <li>PICO_DEVICE_DISCONNECTED if the command could not be sent to the 
///  unit successfully (probably means the unit has been unplugged)
/// <li>PICO_SUCCESS if the call succeded</ul></returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::Ping() {
	unsigned char buf[USBTC08_CMD_BUF_SIZE];
	unsigned int length=USBTC08_CMD_BUF_SIZE;
	PICO_RETURNS retval;	
	// Check the device is open and not streaming
	assert(dev && dev->usbDevice);
	if(GetDeviceState()!=USBTC08_STATE_IDLE) {
		return PICO_UNIT_STATE_ERROR;
	}
	
	buf[0] = USBTC08_CMD_ACKNOWLEDGE;
	
	if ((retval = dev->WritePipe(buf,1)) != PICO_SUCCESS) {
#if DEBUG
		printf("TC08Device::Ping(), write fail, returned %d\n", retval);
#endif
		return PICO_DEVICE_DISCONNECTED;
	}
	if((retval = dev->ReadPipe(buf,&length)) != PICO_SUCCESS) {
#if DEBUG
		printf("TC08Device::Ping(), read fail, returned %d, got %d bytes\n", retval, length);
#endif
		return PICO_DEVICE_DISCONNECTED;
	}
	
	#if DEBUG
	printf ("TC08Device::Ping(): Returned %d unsigned chars, first unsigned char %d\n", length, buf[0]);
	#endif
	
	return PICO_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////
/// Get serial number from the underlying PicoUsbDevice
/// <returns>Pointer to character array containing serial number</returns>
//////////////////////////////////////////////////////////////////////////
const char *TC08Device::GetSerialString(void) {
	
  // Device does not need to be open to do this
  assert(dev->usbDevice);
	
	return dev->usbDevice->GetSerialString();
}


//////////////////////////////////////////////////////////////////////////
/// Enum-typed overload for SetChannelConfig(unsigned int,char)
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::SetChannelConfig(USBTC08_CHANNELS channel,USBTC08_TCTYPES type) {
	return SetChannelConfig((unsigned int)channel,(char)type);
}

//////////////////////////////////////////////////////////////////////////
/// Sets specified channel to given thermocouple type
/// <param name="channel">Number of the single channel to be configured</param>
/// <param name="type">Character indicating the thermocouple type (or mV or off)
/// for the given channel</param>
/// <returns> PICO_RETURNS value indicating whether the call succeded</returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::SetChannelConfig(unsigned int channel,char type) {
	
  // Check the device is open and not streaming
	assert(dev->usbDevice);
	if(GetDeviceState()!=USBTC08_STATE_IDLE) {
		return PICO_UNIT_STATE_ERROR;
	}
	
	// Make sure type is upper case
	type=toupper(type);
		
	// Validate the arguments
	if ((channel < 0) || (channel > USBTC08_MAX_CHANNELS)) {
	    // User passed an invalid channel number
		return PICO_ARGUMENT_ERROR;
    }
	
	if ((strchr(USBTC08_ALL_TCTYPES,type) == NULL) && (channel != USBTC08_CHANNEL_CJC)) {
		// User passed an invalid thermocouple type
		return PICO_ARGUMENT_ERROR;
    }
	
	dev->Channels[channel].fEnabled = (type != (char)USBTC08_TCTYPE_OFF); // Type ' ' specifies "Off"
	dev->Channels[channel].bTCType = (channel == (unsigned int)USBTC08_CHANNEL_CJC) ? USBTC08_TCTYPE_CJC : type; // For the CJC any other char specifies "On"
	
	
	// Tell the unit which channels are currently setup
	dev->SetAllChannels();
	
	return PICO_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////
/// Enum-typed overload for GetChannelConfig(unsigned int)
//////////////////////////////////////////////////////////////////////////
USBTC08_TCTYPES TC08Device::GetChannelConfig(USBTC08_CHANNELS channel) {
#if DEBUG
	printf("TC08Device::GetChannelConfig(%i).\n",channel);
#endif

	return (USBTC08_TCTYPES)GetChannelConfig((unsigned int)channel);
}

//////////////////////////////////////////////////////////////////////////
/// Get the currently set thermocouple type for a given channel
/// <param name="channel">Number of the single channel to be checked</param>
/// <returns>Character indicating the thermocouple type (or mV or off)
/// for the given channel</returns>
//////////////////////////////////////////////////////////////////////////
char TC08Device::GetChannelConfig(unsigned int channel) {
#if DEBUG
	printf("TC08Device::GetChannelConfig(%i).\n",channel);
#endif

  // Check the device is open (doesn't matter if it is streaming)
	assert(dev->usbDevice);
	if(dev->usbDevice->GetDeviceState()!=PICODEVICE_STATE_OPEN) {
    // All channels are "OFF" if the unit is not open!
		return (char)USBTC08_TCTYPE_OFF;
	}
			
	// Validate the arguments
	if ((channel < 0) || (channel > USBTC08_MAX_CHANNELS)) {
	    // User passed an invalid channel number
      // Just return "OFF" in this case
		return (char)USBTC08_TCTYPE_OFF;
    }
	return dev->Channels[channel].bTCType;
}



//////////////////////////////////////////////////////////////////////////
/// Get a single reading on all enabled channels from the unit
/// <param name="temp">Pointer to an array of 9 floats where the readings will be 
/// placed. Note that the array must always be of size 9 regardless of the number 
/// of channels enabled</param>
/// <param name="overflow_flags">Pointer to an unsigned short which the driver sets 
/// as a bit-field indicating whether each of the thermocouple channels was 
/// over-range.</param>
/// <param name="units">USBTC08_UNITS value indicating the temperature unit in 
/// which to return the results</param>
/// <returns>PICO_RETURNS value indicating whether the conversion succedeed</returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::GetSingle(float * temp,unsigned short * overflow_flags,USBTC08_UNITS units) {
	bool            bConverted;
	int             iChannel;
	float           flMillivolts;
	float           flCJCMillivolts;
	float           TempValues[USBTC08_MAX_CHANNELS+1];
	long            lADCCount;
	bool            tempOverflow;
  int             iChannelIndex;
	int             iChannelsEnabled;
	unsigned char buf[USBTC08_CMD_BUF_SIZE];
	
	
  // Check the device is open and not streaming
	assert(dev->usbDevice);
	if(GetDeviceState()!=USBTC08_STATE_IDLE) {
		return PICO_UNIT_STATE_ERROR;
	}
	
	// Validate arguments
	if (units < 0 || units > USBTC08_UNITS_MAX) {
		return PICO_ARGUMENT_ERROR;
	}

  // Check there is at least one channel enabled
  for(iChannelIndex = 0, iChannelsEnabled = 0;iChannelIndex <= USBTC08_MAX_CHANNELS; iChannelIndex++) {
		if(dev->Channels[iChannelIndex].fEnabled) {
			iChannelsEnabled++;
		}
  }
	
	if (iChannelsEnabled < 1) {
		return PICO_NO_ENABLED_CHANNELS; 
  }

	// temp must not be a null ptr
	if (!temp) {
		return PICO_ARGUMENT_ERROR;
  }		
	
  // Overflow flags can be null: reset them if they aren't
	if (overflow_flags) {
		*overflow_flags = 0x0; // Default: no overflows
	}

	// Perform single conversion on all enabled channels
	buf[0] = USBTC08_CMD_READ_CHANNELS;
	
	dev->WritePipe(buf,1);
	
	// Make the function sleep for the minimum streaming
	//  interval so it doesn't ask for data begfore it is ready

	usleep(1000l * GetMinimumInterval());	

  // Read back the data from the unit
	unsigned int length = USBTC08_STREAM_PKT_SIZE;
	dev->ReadPipe(buf, &length);
	// TODO: Does this succeed?
	// CJC first
	if (buf[0]) // the CJC was converted
    {

    // Get the CJC reading into mV
		lADCCount = ConstructReading(buf[1], buf[2], buf[3]);
		Truncate20Bit(lADCCount);
		flMillivolts = AdcToMilliVolts(USBTC08_GAIN_1V25, lADCCount);
		
    // Convert mV into celsius
		bConverted = dev->TCTables.ConvertTemp(  USBTC08_TCTYPE_THERMISTOR, 
											flMillivolts,
											&TempValues[USBTC08_CHANNEL_CJC],
											NULL);
		assert(bConverted);
    }
	
  // Each of the thermocouple channels
	for(iChannel = 1; iChannel <= USBTC08_MAX_CHANNELS; ++iChannel)
    {
		if (dev->Channels[iChannel].fEnabled)
		{
			lADCCount = ConstructReading(buf[(iChannel*4)+1], buf[(iChannel*4)+2], buf[(iChannel*4)+3]); 
			
			// Limit the reading by truncating it to 20 bits,
			//  then applying constraints, report any overflows but continue
			Truncate20Bit(lADCCount);
			flMillivolts = AdcToMilliVolts(USBTC08_GAIN_THERMOCOUPLE, lADCCount);
			if (!dev->LimitReading(&flMillivolts, THERMOCOUPLE_MV_LIMIT))
			{
				if (overflow_flags)
					*overflow_flags |= (0x01 << (iChannel - 1));
			}
			
			// Apply CJC compensation to the reading
			if ((dev->Channels[iChannel].bTCType == USBTC08_TCTYPE_MV))
			{
				TempValues[iChannel] = flMillivolts; // No compensation for mV channels
			}
			else
			{
				// Convert the thermistor temperature back to millivolts using the TC's scaling
				bConverted = dev->TCTables.ConvertMillivolts ( dev->Channels[iChannel].bTCType,
														  TempValues[USBTC08_CHANNEL_CJC],
														  &flCJCMillivolts,
														  &tempOverflow);
				assert(bConverted);
				if (tempOverflow)
				{
					if (overflow_flags)
						*overflow_flags |= (0x01 << (iChannel - 1));
				}
				
				// Perform the Cold Junction Compensation by adding the CJC mV onto the TCs mV
				flMillivolts += flCJCMillivolts;
				
				// Now finally ... scale the mV reading to temperature
				bConverted = dev->TCTables.ConvertTemp(  dev->Channels[iChannel].bTCType,
													flMillivolts,
													&TempValues[iChannel],
													&tempOverflow);
				assert(bConverted);
				if (tempOverflow)
				{
					if (overflow_flags)
						*overflow_flags |= (0x01 << (iChannel - 1));
				}
			}
		}
		else // The current channel is disabled
		{
			TempValues[iChannel] = std::numeric_limits<float>::quiet_NaN();
		}
    }
	
	// Re-scale the units to fahrenheit or kelvin if required
	//
	for (iChannel = 0; iChannel <= USBTC08_MAX_CHANNELS; ++iChannel) {
		if (dev->Channels[iChannel].fEnabled && (dev->Channels[iChannel].bTCType != USBTC08_TCTYPE_MV)) {
			//TODO: if (!_isnan(TempValues[iChannel]))
			TempValues[iChannel]=ConvertFromCentigrade(TempValues[iChannel],(USBTC08_UNITS)units);
		}
	}
	
	memcpy(temp, TempValues, sizeof(TempValues));
	
	return PICO_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////
/// Get the minimum (i.e. fastest) sampling interval in streaming mode, with the
/// currently enabled set of channels
/// <returns>Minimum sampling interval in streaming mode with the
/// current channel setup, in milliseconds</returns>
//////////////////////////////////////////////////////////////////////////
unsigned int TC08Device::GetMinimumInterval(void) {
	int             iChannelIndex;
	int             iChannelsEnabled;
	
	// Check the device is open (doesn't matter if it is streaming)
	assert(dev->usbDevice);
	if(dev->usbDevice->GetDeviceState()!=PICODEVICE_STATE_OPEN) {
		return 0; // 0ms interval denotes failure
	}
	
	// Count the enabled channels
	for(iChannelIndex = 0, iChannelsEnabled = 0;iChannelIndex <= USBTC08_MAX_CHANNELS; iChannelIndex++) {
		if(dev->Channels[iChannelIndex].fEnabled) {
			iChannelsEnabled++;
		}
	}
	
	// Check at least one channel is enabled (unit will refuse to sample otherwise)
	if (iChannelsEnabled<1) {
		return 0;  // 0ms interval denotes failure
	}
	
	// Sample interval is just (enabled channels) * (interval per channel)
	return (unsigned int)iChannelsEnabled * USBTC08_MIN_INTERVAL_MS;
}

//////////////////////////////////////////////////////////////////////////
/// Get the maximum (i.e. slowest) sampling interval in streaming mode. This
/// is constant regardless of the channels enabled, this function is provided
/// for consistency with GetMinimumInterval()
/// <returns>Maximum sampling interval in streaming mode in milliseconds</returns>
//////////////////////////////////////////////////////////////////////////
unsigned int TC08Device::GetMaximumInterval(void) {
	// Doesn't matter whether the unit is open
	return (unsigned int)USBTC08_MAX_INTERVAL_MS;
}

//////////////////////////////////////////////////////////////////////////
/// Set device running in streaming mode
/// <param name="interval_ms"> Sample interval in milliseconds to get one sample 
/// on all enabled channels. Must be between the values returned by GetMinimumInterval() 
/// and GetMaximumInterval() or the call will fail</param>
/// <returns>PICO_RETURNS value indicating whether the conversion succedeed</returns>
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::Run(unsigned int interval_ms) {
	int				iChannel;
	int				iChannelsEnabled;
	ReadingBuffer	*pSampleBuffer;
	unsigned char buf[USBTC08_CMD_BUF_SIZE];

#if DEBUG
	printf("TC08Device::Run(%d)\n",interval_ms);
#endif
	
	// Check state - must be open and not streaming
	assert(dev->usbDevice);
	if(GetDeviceState()!=USBTC08_STATE_IDLE) {
		return PICO_UNIT_STATE_ERROR;
	}
	// Check argument
	if((interval_ms<GetMinimumInterval())||(interval_ms>GetMaximumInterval())) {
		return PICO_ARGUMENT_ERROR;
	}
	
	// The sampling interval must be a multiple of the unit's interrupt timer
	//  (always round down)
	
	dev->interval=interval_ms-(interval_ms%USBTC08_INTERRUPT_MS);
	
#if DEBUG
	printf("Interval is %d\n", dev->interval);
#endif
	
	dev->ResetBuffers();
	
	//////////////////////////////////////////////////////////////////////////
	// Setup the channel buffers (to hold the data)
	for (iChannel = 0, iChannelsEnabled = 0; iChannel <= USBTC08_MAX_CHANNELS; ++iChannel) {
		if(dev->Channels[iChannel].fEnabled) {
			pSampleBuffer = &(dev->Channels[iChannel].SampleBuffer);
			
			++iChannelsEnabled;
			// set the time interval
			pSampleBuffer->set_TimeInfo((long)(dev->interval));
			
			// Set the maximum buffer size, this should be at least 1 minute
			// Worst case: 1 channel sampling at USBTC08_MIN_INTERVAL_MS
			pSampleBuffer->set_ReadingBufferSize(60000L / USBTC08_MIN_INTERVAL_MS);
		} 
    }
	
	// check that at least one channel is enabled
	if (iChannelsEnabled == 0) {
		return PICO_NO_ENABLED_CHANNELS;
    }
	
	for (iChannel=0;iChannel<USBTC08_CHANNEL_COUNT;iChannel++) {
		dev->LastGoodValues[iChannel] = 0.0f;
    }
	
  // Write the "run" command to the device
	buf[0] = USBTC08_CMD_RUN_STREAMING;
	buf[1] = (unsigned char)(((dev->interval)>>8)&0xff);
	buf[2] = (unsigned char)((dev->interval)&0xff);
	
	if(dev->WritePipe(buf,3)) {
#if DEBUG
		printf("TC08Device::Run: Could not send command to unit.\n");
#endif
		return PICO_DEVICE_DISCONNECTED;
	}
	
	dev->thread=new pthread_t();
	
	// Start the polling thread
	int err;
	if(err=pthread_create(dev->thread,NULL,dev->threadProc,dev)) {
		Stop();
		return PICO_FAILURE;
	}
	
	// The unit is now streaming so we should only talk to it under 2 circumstances
	//  1. to read values
	//  2. to stop streaming
	dev->state=USBTC08_STATE_STREAMING;
	
	return PICO_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////
/// Get a unique device handle for use with the C wrapper
/// <returns>Non-negative short uniquely identifying the device</returns>
//////////////////////////////////////////////////////////////////////////
short TC08Device::GetHandle(void)
{

#if DEBUG
	//printf("TC08Device::GetHandle()\n");
#endif
	
	assert(dev && dev->usbDevice);
	return dev->usbDevice->GetHandle();
}

//////////////////////////////////////////////////////////////////////////
/// Copy values only from the driver's circ buffer to a user buffer during
/// streaming data collection
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::GetValues(float **temp_buffer,unsigned int *lengths,
                                   unsigned short *overflow,
                                   USBTC08_UNITS units,
                                   bool fill_missing) 
{
	return TC08Device::GetTimesAndValues(temp_buffer, NULL, lengths, overflow,units, fill_missing, false);
}

//////////////////////////////////////////////////////////////////////////
/// Copy values only from the driver's circ buffer to a user buffer during
/// streaming data collection for the specified channel only
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::GetSingleChannelValues(USBTC08_CHANNELS channel, 
		float *temp_buffer, unsigned int * length, unsigned short *overflow,
		USBTC08_UNITS units, bool fill_missing) 
{
	return TC08Device::GetSingleChannelTimesAndValues(channel, temp_buffer, 
	                                                  NULL, length, 
	                                                  overflow,units, 
	                                                  fill_missing, false);
}
//////////////////////////////////////////////////////////////////////////
/// Copy times and values from the driver's circ buffer to a user buffer during
/// streaming data collection for the specified channel only
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::GetSingleChannelTimesAndValues(USBTC08_CHANNELS channel, 
		float *temp_buffer,long* times_buffer,unsigned int * length,
		unsigned short *overflow,USBTC08_UNITS units,bool fill_missing, 
		bool deskew_times) 
{

	// Fake the multichannel arrays
	float * multiTempBuffer[USBTC08_CHANNEL_COUNT] = {0,0,0,0,0,0,0,0,0};
	long  * multiTimeBuffer[USBTC08_CHANNEL_COUNT] = {0,0,0,0,0,0,0,0,0};
	unsigned int multiLengthBuffer[USBTC08_CHANNEL_COUNT] = {0,0,0,0,0,0,0,0,0};
	
	// Validate the arguments
	if ((channel < 0) || (channel > USBTC08_MAX_CHANNELS) || (!dev->Channels[channel].fEnabled)) {
		// User passed an invalid channel number
		return PICO_ARGUMENT_ERROR;
	}
	
	// Put our pointer in the correct array location
	multiTempBuffer[channel] = temp_buffer;
	multiTimeBuffer[channel] = times_buffer;
	multiLengthBuffer[channel] = *length;
	
	PICO_RETURNS retVal = TC08Device::GetTimesAndValues(multiTempBuffer, 
			multiTimeBuffer, multiLengthBuffer, overflow,units, 
			fill_missing, deskew_times);
	
	*length = multiLengthBuffer[channel];
	return retVal;
	
}


//////////////////////////////////////////////////////////////////////////
/// Copy times and values from the driver's circ buffer to a user buffer during
/// streaming data collection
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::GetTimesAndValues(float **temp_buffer,
		long** times_buffer,  unsigned int * lengths, 
		unsigned short *overflow,USBTC08_UNITS units,
		bool fill_missing, bool deskew_times) 
{
	int             iReading;
	int             iFirstReading;
	unsigned int    readings_copied[USBTC08_CHANNEL_COUNT];
	bool            temp_bufferOverflow;
	long            lFirstTime_ms[USBTC08_CHANNEL_COUNT];
	long            lTimeOffset = 0;
	int             iEnabledChannelIndex = 0;
#if DEBUG
	//printf("TC08Device::GetValues()\n");
#endif
	
	assert(dev->usbDevice);
	
  // Validate the arguments
  // Device must be open and streaming
	if(GetDeviceState()!=USBTC08_STATE_STREAMING) {
		return PICO_UNIT_STATE_ERROR;
	}
	
  // Buffer to copy data into must not be null
	if(!temp_buffer) {
		return PICO_ARGUMENT_ERROR;
    }
	
	// Buffer for reading counts must not be null
	if(!readings_copied) {
		return PICO_ARGUMENT_ERROR;
	}
		
  // Reset the overflow flags
	if(overflow) {
		*overflow = 0; // Default: no overflow
	}

  // TODO: validate units parameter
	
	for(int curCh=0;curCh<USBTC08_CHANNEL_COUNT;curCh++) {
		// If user has passed a buffer for this curCh
		if((dev->Channels[curCh].fEnabled)&&(temp_buffer[curCh]!=NULL) && lengths[curCh] > 0) {
			
			// This does a memcpy to the caller's temp_buffer array
			readings_copied[curCh]=dev->Channels[curCh].SampleBuffer.get_Readings(temp_buffer[curCh],&temp_bufferOverflow,lengths[curCh],&(lFirstTime_ms[curCh]));
			lengths[curCh] = readings_copied[curCh];
			
			// Set overflow flag for this channel
			if((overflow)&&(temp_bufferOverflow)) {
				*overflow|=1<<curCh;
			}
			
			// Re-scale the units to fahrenheit or kelvin if required
			//  (not when we're measuring mV though
			if(dev->Channels[curCh].bTCType!=USBTC08_TCTYPE_MV) {
				ConvertFromCentigradeI(temp_buffer[curCh],readings_copied[curCh],(USBTC08_UNITS)units);
			}
			
			// If the user requests us to fill_missing,
			// it means we are going to replace missing readings (QNaNs)
			// with the last valid reading
			if (fill_missing) {
				// find the first non-QNaN value
				iFirstReading = 0;
				while ((iFirstReading < readings_copied[curCh]) && isnan(temp_buffer[curCh][iFirstReading])) {
					++iFirstReading;
				}
				
				// If there is only QNaNs, then use the last resort, 
				//  which is the readings from the last call to this function
				if (iFirstReading == readings_copied[curCh]) {
					for (iReading = 0; iReading < readings_copied[curCh]; iReading++) {
						temp_buffer[curCh][iReading]=dev->LastGoodValues[curCh];
					}
					
					continue;
				}
				
				// Now pad backwards to the first reading
				for (iReading = (iFirstReading - 1); iReading >= 0; --iReading) {
					temp_buffer[curCh][iReading] = temp_buffer[curCh][iFirstReading];
				}
				
				// Now pad forwards (we know that all the readings upto
				// iFirstReading are already filled)
				for (iReading = (iFirstReading + 1); iReading < readings_copied[curCh]; ++iReading) {
					if (isnan(temp_buffer[curCh][iReading])) {
						temp_buffer[curCh][iReading] = temp_buffer[curCh][iReading - 1]; // pad with the previous value
					}
				}
			}
			
			// Store the last known value to help with 
			//  padding on subsequent calls to this function
			if (readings_copied[curCh] > 0) {
				dev->LastGoodValues[curCh]=temp_buffer[curCh][readings_copied[curCh] - 1];
			}
		
			lTimeOffset = deskew_times ?
					(long)(iEnabledChannelIndex * USBTC08_CONVERSION_TIME 
					* (dev->freq == USBTC08_MAINS_60HZ ? 1.2f : 1L)) : 0;
					
			// Write sample times to buffers
			if (times_buffer && times_buffer[curCh])
			{
				for (unsigned int iTime = 0; iTime < readings_copied[curCh]; ++iTime)
				{
					times_buffer[curCh][iTime] = lFirstTime_ms[curCh] + lTimeOffset + (iTime * dev->interval);
				}
			}
			
			
		} else {
			// This buffer will hold no valid readings 
			//temp_buffer[curCh]=NULL;
			readings_copied[curCh]=0;
			lengths[curCh] = 0;
		}
		// Increment for next channel 
		if (dev->Channels[curCh].fEnabled)
			iEnabledChannelIndex++;

	}
	

	return PICO_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////
/// Stop the device when running in streaming mode
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::Stop() {
	unsigned char buf[USBTC08_CMD_BUF_SIZE];
#if DEBUG
	printf("TC08Device::Stop()\n");
#endif
	
	assert(dev->usbDevice);

	// Make sure we're actually in streaming mode
	if(GetDeviceState()==USBTC08_STATE_STREAMING) {
#if DEBUG
		printf("TC08Device::Stop: Device is in streaming mode.\n");
#endif
		// Signal the streaming thread to terminate
		if(dev->thread) {
#if DEBUG
			printf("TC08Device::Stop: Thread is not null.\n");
#endif
				
#if DEBUG
			printf("TC08Device::Stop: Locking threadLock.\n");
#endif
			pthread_mutex_lock(&(dev->threadLock));
#if DEBUG
			printf("TC08Device::Stop: Setting threadStop flag.\n");
#endif
			dev->threadStop=true;
#if DEBUG
			printf("TC08Device::Stop: Unlocking threadLock.\n");
#endif
			pthread_mutex_unlock(&(dev->threadLock));
			
#if DEBUG
			printf("TC08Device::Stop: Joining thread...\n");
#endif
			// Ensure that the thread exits safely
			int err;
			if(err=pthread_join(*(dev->thread),NULL)) {
#if DEBUG
/*				printf("TC08Device::Stop: Joining thread...error!\n");
				if(err==EINVAL) {
					printf("TC08Device::Stop: Thread does not refer to a joinable thread.\n");
				} else if(err==ESRCH) {
					printf("TC08Device::Stop: Thread could not be found.\n");
				} else if(err==EDEADLK) {
					printf("TC08Device::Stop: A deadlock was detected.\n");
				}
*/
#endif
			} else {
#if DEBUG
				printf("TC08Device::Stop: Joining thread...ok!\n");
#endif
			}
			
#if DEBUG
			printf("TC08Device::Stop: Detaching thread...");
#endif
			if(err=pthread_detach(*(dev->thread))) {
#if DEBUG
/*
				printf("error!\n");
				if(err==EINVAL) {
					printf("TC08Device::Stop: thread does not refer to a joinable thread.\n");
				} else if(err==ESRCH) {
					printf("TC08Device::Stop: thread could not be found.\n");
				}
*/
#endif
			} else {
#if DEBUG
				printf("ok!\n");
#endif
			}
			delete (dev->thread);
			
		} else {
#if DEBUG
			printf("TC08Device::Stop: Thread is null.\n");
#endif
		}
		
    // Streaming thread has been terminated.	
		// Stop the unit and
		// flush out any readings in the endpoint
		
#if DEBUG
		printf("TC08Device::Stop: Stopping unit...");
#endif		
		// Stop the unit from running
		buf[0]=USBTC08_CMD_STOP_STREAMING;
		if(dev->WritePipe(buf, 1)) {
#if DEBUG
			printf("error!\n");
#endif		
			return PICO_DEVICE_DISCONNECTED;
		} else {
#if DEBUG
			printf("ok!\n");
#endif		
		}
		
		// Read in data until we get 36 x 0xFF
		//  then we know the endpoints are clear of data
		bool success;
		do {
			success = true;
			unsigned int bytesRead=USBTC08_STREAM_PKT_SIZE;
			dev->ReadPipe(buf,&bytesRead);
			if(bytesRead==0) {
				return PICO_DEVICE_DISCONNECTED;
			} else if (bytesRead!=USBTC08_STREAM_PKT_SIZE) {
				success = false;
				continue;
			}
			
			for(int i=0;i<USBTC08_STREAM_PKT_SIZE;i++) {
				if(buf[i]!=0xff) {
					success = false;
				}
			}
		} while(!success);
		
		// we can now talk to the unit
		dev->threadStop = false;

		dev->state=USBTC08_STATE_IDLE;
		
	} else { // We weren't in streaming mode
#if DEBUG
		printf("TC08Device::Stop: Device is not in streaming mode.\n");
#endif
	return PICO_UNIT_STATE_ERROR;
	}
	return PICO_SUCCESS;
		
}


//////////////////////////////////////////////////////////////////////////
/// Open the TC08 unit
//////////////////////////////////////////////////////////////////////////
USBTC08_STATES TC08Device::Open(void) {
#if DEBUG
	printf("TC08Device::Open()\n");
#endif
	
	assert(dev->usbDevice);
	
	PICODEVICE_STATES deviceState=dev->usbDevice->GetDeviceState();
	
	// If the device isn't open for whatever reason, try to open it
	if((deviceState==PICODEVICE_STATE_CLOSED)||(deviceState==PICODEVICE_STATE_LOCKED)) {
		// Try to open the device if it is closed
		deviceState = dev->usbDevice->Open();
	} else if(deviceState==PICODEVICE_STATE_DISCONNECTED) {
		// TODO: If the device has been unplugged and reconnected, attempt
		// to find it by serial number and reopen it
		// Probably something like
		// PicoDevice::Enumerate()
		// find device with same serial
		// this->device=newDevice
	}
	
	// If the device is now open (or was already open) then init it.
	if((deviceState==PICODEVICE_STATE_OPEN)) {
		
		/*
#if DEBUG
		 int numPipes=device->GetPipes();		
		 
		 printf("TC08Device::TC08Device: GetPipes()=%i\n",numPipes);
		 
		 for(int i=1;i<=numPipes;i++) {
			 PICODEVICE_PIPEINFORMATION pipeInfo=device->GetPipeInformation(i);
			 
			 printf("TC08Device::TC08Device: Info for pipe %i\n",i);
			 printf("TC08Device::TC08Device:\t\tdirection=%i\n",pipeInfo.direction);
			 printf("TC08Device::TC08Device:\t\tnumber=%i\n",pipeInfo.number);
			 printf("TC08Device::TC08Device:\t\ttransferType=%i\n",pipeInfo.transferType);
			 printf("TC08Device::TC08Device:\t\tmaxPacketSize=%i\n",pipeInfo.maxPacketSize);
			 printf("TC08Device::TC08Device:\t\tinterval=%i\n",pipeInfo.interval);
			 printf("TC08Device::TC08Device:\t\tstatus=%i\n",pipeInfo.status);
		 }
#endif
		 */
		
#if DEBUG
	printf("TC08Device::Open(): About to get pipes\n");
#endif
		assert(dev->usbDevice->GetPipes()==2);
#if DEBUG
	printf("TC08Device::Open(): Got pipes\n");
#endif
	
		dev->ResetPipes();
		dev->state = USBTC08_STATE_IDLE;
		dev->Init();
		SetLED(USBTC08_LED_GREEN);
	}
	
	return GetDeviceState();	
}


//////////////////////////////////////////////////////////////////////////
/// Close an open device
//////////////////////////////////////////////////////////////////////////
PICO_RETURNS TC08Device::Close() {
#if DEBUG
	printf("TC08Device::Close()\n");
#endif
	
	assert(dev->usbDevice);
	// Check the device is actually open
	if(dev->usbDevice->GetDeviceState()==PICODEVICE_STATE_OPEN) {
#if DEBUG
		printf("TC08Device::Close: Device is open.\n");
#endif
		
    // Stop the device if it is streaming
		if(GetDeviceState()==USBTC08_STATE_STREAMING) {
#if DEBUG
			printf("TC08Device::Close: Device is streaming.\n");
#endif
			
#if DEBUG
			printf("TC08Device::Close: Stopping device.\n");
#endif
			Stop();
			
      // If the device has been unplugged it is already "closed"!
			if(dev->usbDevice->GetDeviceState()==PICODEVICE_STATE_DISCONNECTED) {
#if DEBUG
				printf("TC08Device::Close: Device is disconnected.\n");
#endif

				return PICO_SUCCESS;
			}
		} else {
#if DEBUG
			printf("TC08Device::Close: Device is idle.\n");
#endif
		}

		// Turn off LED
		SetLED(USBTC08_LED_OFF);
		
#if DEBUG
		printf("TC08Device::Close: Closing device.\n");
#endif		
		dev->usbDevice->Close();
	} else {
#if DEBUG
		printf("TC08Device::Close: Device is already closed.\n");
#endif
	}
	
	return PICO_SUCCESS;
}

