/*******************************************************************************/
/* Permission is hereby granted, free of charge, to any person or organization */
/* obtaining a copy of the software and accompanying documentation covered by  */
/* this license (the "Software") to use, reproduce, display, distribute,       */
/* execute, and transmit the Software, and to prepare derivative works of the  */
/* Software, and to permit third-parties to whom the Software is furnished to  */
/* do so, all subject to the following:                                        */
/*                                                                             */
/* The copyright notices in the Software and this entire statement, including  */
/* the above license grant, this restriction and the following disclaimer,     */
/* must be included in all copies of the Software, in whole or in part, and    */
/* all derivative works of the Software, unless such copies or derivative      */
/* works are solely in the form of machine-executable object code generated by */
/* a source language processor.                                                */
/*                                                                             */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR  */
/* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,    */
/* FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT   */
/* SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE   */
/* FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, */
/* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER */
/* DEALINGS IN THE SOFTWARE.                                                   */
/*******************************************************************************/

#include <lfp/syslog.h>
#include <lfp/stdlib.h>
#include <lfp/string.h>

#include <pthread.h>

static pthread_mutex_t syslog_mutex = PTHREAD_MUTEX_INITIALIZER;

static char *syslog_ident = NULL;

#define SYSLOG_IDENT_SIZE 1024

static void
_lfp_closelog(void)
{
    closelog();
    if (syslog_ident) {
        free(syslog_ident);
        syslog_ident = NULL;
    }
}

static void
copy_syslog_ident(const char *ident)
{
    if (ident) {
        syslog_ident = malloc(SYSLOG_IDENT_SIZE);
        strncpy(syslog_ident, ident, SYSLOG_IDENT_SIZE - 1);
        syslog_ident[SYSLOG_IDENT_SIZE - 1] = 0;
    }
}

DSO_PUBLIC void
lfp_openlog(const char *ident, int options, int facility)
{
    pthread_mutex_lock(&syslog_mutex);
    _lfp_closelog();
    copy_syslog_ident(ident);
    openlog(syslog_ident, options, facility);
    pthread_mutex_unlock(&syslog_mutex);
}

DSO_PUBLIC void
lfp_syslog(int priority, const char *msg, ...)
{
    va_list args;
    va_start(args, msg);
    vsyslog(priority, msg, args);
    va_end(args);
}

DSO_PUBLIC void
lfp_vsyslog(int priority, const char *msg, va_list args)
{
    vsyslog(priority, msg, args);
}

DSO_PUBLIC void
lfp_closelog(void)
{
    pthread_mutex_lock(&syslog_mutex);
    _lfp_closelog();
    pthread_mutex_unlock(&syslog_mutex);
}

DSO_PUBLIC int
lfp_setlogmask(int maskpri)
{
    return setlogmask(maskpri);
}

DSO_PUBLIC int
lfp_log_mask(int priority)
{
    return LOG_MASK(priority);
}

DSO_PUBLIC int
lfp_log_upto(int priority)
{
    return LOG_UPTO(priority);
}
