package tests::ReportParserHTMLWriterTest;

use strict;

use base qw/ Lire::Test::TestCase tests::TestStoreFixture 
             tests::ChartTypesFixture /;

use File::Basename qw/dirname/;

use Lire::ChartType;
use Lire::Config::ConfigSpec;
use Lire::DlfSchema;
use Lire::Report;
use Lire::Report::TableInfo;
use Lire::Report::Section;
use Lire::Report::Subreport;
use Lire::ReportParser::HTMLWriter;
use Lire::ReportParser::ReportBuilder;
use Lire::Utils qw/ tempdir file_content /;
use Lire::Test::Mock;
use IO::Scalar;
use File::Path qw/rmtree/;
use Time::Local;

#our @TESTS = qw//;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schema();
    $self->{'tmpdir'} = tempdir( $self->name() . "_XXXXXX" );

    $self->set_up_tz( 'UTC' );

    $self->{'writer'} =
      new_proxy Lire::Test::Mock( 'Lire::ReportParser::HTMLWriter' );
    $self->{'writer'}{'_fh'} = new IO::Scalar();
    $self->{'writer'}{'_outputdir'} = $self->{'tmpdir'};
    $self->{'writer'}{'_etag'} = '>';
    $self->{'writer'}{'_xhtml'} = '0';

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down( @_ );

    rmtree( $self->{'tmpdir'}, 0 )
      unless $self->{'keep_tmpdir'};

    no warnings 'redefine';

    *Lire::ReportParser::HTMLWriter::ac_path = $self->{'_or_ac_path'}
      if $self->{'_or_ac_path'};

    return;
}

sub set_up_report {
    my $self = $_[0];

    my $report = new Lire::Report();
    my $sect1 = new Lire::Report::Section( '<Section 1>' );
    my $sub1 = new Lire::Report::Subreport( 'test', 'top-dirs' );
    $sub1->title( '<Top Dirs>' );
    $sub1->id( 'top-dirs.0' );
    $sub1->schemas( 'test-extended' );
    $sect1->add_subreport( $sub1 );
    $report->add_section( $sect1 );
    my $sect2 = new Lire::Report::Section( 'Section 2' );
    $report->add_section( $sect2 );
    my $sub2 = new Lire::Report::Subreport( 'test', 'top-files' );
    $sub2->title( 'Top Files 1' );
    $sub2->id( 'top-files.0' );
    $sub2->schemas( 'test' );
    $sect2->add_subreport( $sub2 );
    my $sub3 = new Lire::Report::Subreport( 'test', 'top-files' );
    $sub3->title( 'Top Files 2' );
    $sub3->id( 'top-files.1' );
    $sub3->schemas( 'test' );
    $sect2->add_subreport( $sub3 );

    $self->{'report'} = $report;
    return;
}

sub set_up_subreport {
    my $self = $_[0];

    my $subreport = new Lire::Report::Subreport( 'test', 'test1' );
    my $table_info = new Lire::Report::TableInfo();
    $table_info->create_column_info( 'user', 'categorical', 'string',
                                     '<User>' );
    my $group = $table_info->create_group_info( 'group1' );
    $group->create_column_info( 'file', 'categorical', 'string', '<File>' );
    $group->create_column_info( 'count', 'numerical', 'int', 'Downloads' );
    $group->create_column_info( 'size', 'numerical', 'bytes', 'Size' );

    $table_info->compute_group_layout();
    $subreport->table_info( $table_info );
    $subreport->schemas( 'test' );
    $self->{'subreport'} = $subreport;

    return;
}

sub set_up_subreport_data {
    my $self = $_[0];

    my $data = [ [ 'flacoste', undef,  9, 53 ],
                 [ undef, 'page1.html',  5, 50 ],
                 [ undef, 'page2.html',  4,  3 ],
                 [ '<wsourdeau>', undef,  5, 10 ],
                 [ undef, 'page3.html',  3,  6  ],
                 [ undef, 'page2.html',  1,  2  ],
                 [ undef, 'page4.html',  1,  2 ] ];
    my $group;
    foreach my $r ( @$data ) {
        my ( $user, $page, $count, $size ) = @$r;
        if ( $user ) {
            my $entry = $self->{'subreport'}->create_entry();
            $entry->add_name( $user );
            $group = $entry->create_group();
            $group->set_summary_value( 'count', 'content' => $count );
            $group->set_summary_value( 'size', 'content' => $size );
        } else {
            my $entry = $group->create_entry();
            $entry->add_name( $page );
            $entry->add_value( 'content' => $count );
            $entry->add_value( 'content' => $size );
        }
    }
    $self->{'subreport'}->finalize();
    return;
}

sub set_up_chart_configs {
    my $self = $_[0];

    $self->{'cfg'}{'_lr_config_spec'} = new Lire::Config::ConfigSpec();
    $self->set_up_plugin_mgr();

    my $type = new_proxy Lire::Test::Mock( 'Lire::ChartType' );
    $type->set_result( 'name' => 'mock_chart',
                       'write_chart' => sub { return "$self->{'tmpdir'}/chart.png" } );
    Lire::PluginManager->register_plugin( $type );
    $self->{'mock_chart'} = $type;

    $self->{'chart1'} = new Lire::Report::ChartConfig();
    $self->{'chart1'}->type( $type );
    $self->{'chart1'}->title( '<A chart>' );
    $self->{'chart2'} = new Lire::Report::ChartConfig();
    $self->{'chart2'}->type( $type );

    return;
}

sub test_write_report {
    my $self = $_[0];

    $self->set_up_report();

    my $output_dir = "$self->{'tmpdir'}/wawa";
    my $writer = new_proxy Lire::Test::Mock( 'Lire::ReportParser::HTMLWriter');
    $writer->set_result( 'start_page', '' );
    $writer->set_result( 'end_page', '' );
    $writer->set_result( 'write_toc', '' );
    $writer->set_result( 'write_section', '' );
    $writer->set_result( 'write_schema', '' );
    $writer->set_result( 'copy_html_files', '' );

    $writer->write_report( $self->{'report'}, $output_dir );
    $self->assert_str_equals( $self->{'report'}, $writer->{'_report'} );
    $self->assert_str_equals( $output_dir, $writer->{'_outputdir'} );
    $self->assert( -d $writer->{'_outputdir'}, 
                   "directory '$writer->{'_outputdir'}' wasn't created" );
    $self->assert_str_equals( ' />', $writer->{'_etag'} );
    $self->assert_num_equals( 0, $writer->{'_one_page'} );
    $self->assert_num_equals( 0, $writer->invocation_count( 'start_page' ) );
    $self->assert_num_equals( 0, $writer->invocation_count( 'end_page' ) );
    $self->assert_num_equals( 1, $writer->invocation_count( 'write_toc' ) );
    $self->assert_num_equals( 2, $writer->invocation_count( 'write_section' ) );
    $self->assert_num_equals( 2, $writer->invocation_count( 'write_schema' ) );
    $self->assert_num_equals( 1, $writer->invocation_count( 'copy_html_files' ) );

    $writer->write_report( $self->{'report'}, $output_dir,
                                     'one_page' => 1, 'xhtml' => 0 );
    $self->assert_num_equals( 1, $writer->invocation_count( 'start_page' ) );
    $self->assert_num_equals( 1, $writer->invocation_count( 'end_page' ) );
    $self->assert_str_equals( '>', $writer->{'_etag'} );
    $self->assert_num_equals( 1, $writer->{'_one_page'} );

    return;
}

sub create_test_schema {
    my $self = $_[0];

    my $test_schema = new Lire::DlfSchema( 'superservice' => 'test',
                                           'timestamp' => 'time' );
    $test_schema->title( '<A title>' );
    $test_schema->description( '<para>A description.</para>' );
    $test_schema->add_field( new Lire::Field( 'name' => 'time',
                                              'type' => 'timestamp',
                                              'label' => '<Time>',
                                              'description' => '<para>The time field.</para>' ) );
    $test_schema->add_field( new Lire::Field( 'name' => 'field_1',
                                              'type' => 'string',
                                              'label' => 'Field 1',
                                              'description' => '<para>The first field.</para>' ) );

    return $test_schema;
}

sub test_write_schema {
    my $self = $_[0];

    my $test_schema = $self->create_test_schema();
    local $Lire::DlfSchema::SCHEMA_CACHE{'test'} = $test_schema;

    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}{'_one_page'} = 1;
    $self->{'writer'}->set_result( 'start_page', '' );
    $self->{'writer'}->set_result( 'end_page', '' );
    $self->{'writer'}->write_schema( 'test' );
    $self->assert_num_equals( 0, $self->{'writer'}->invocation_count( 'start_page' ) );
    $self->assert_num_equals( 0, $self->{'writer'}->invocation_count( 'end_page' ) );
    $self->assert_str_equals( <<SCHEMA, $$buf );
    <div class="section">
      <h2><a name="schema:test">&lt;A title></a></h2>
      <dl>
        <dt><a name="schema:test:time">&lt;Time></a></dt>
        <dd>
            <div class="para">The time field.</div>

        </dd>
        <dt><a name="schema:test:field_1">Field 1</a></dt>
        <dd>
            <div class="para">The first field.</div>

        </dd>
      </dl>
    </div>
SCHEMA

    $$buf = '';
    $self->{'writer'}{'_one_page'} = 0;
    $self->{'writer'}->write_schema( 'test');
    $self->assert_num_equals( 1, $self->{'writer'}->invocation_count( 'start_page' ) );
    $self->assert_num_equals( 1, $self->{'writer'}->invocation_count( 'end_page' ) );
    $self->assert_str_equals( 'test.html', $self->{'writer'}{'_prev'} );

    $self->assert_str_equals( <<SCHEMA, $$buf );
    <div class="section">
      <dl>
        <dt><a name="time">&lt;Time></a></dt>
        <dd>
            <div class="para">The time field.</div>

        </dd>
        <dt><a name="field_1">Field 1</a></dt>
        <dd>
            <div class="para">The first field.</div>

        </dd>
      </dl>
    </div>
SCHEMA

}

sub test_write_section {
    my $self = $_[0];

    $self->set_up_report();
    my $section = new Lire::Report::Section( 'Empty section' );
    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}{'_one_page'} = 1;
    $self->{'writer'}->set_result( 'start_page', '' );
    $self->{'writer'}->set_result( 'end_page', '' );
    $self->{'writer'}->set_result( 'write_subreport', '' );
    $self->{'writer'}->write_section( $section, 1 );
    $self->assert_num_equals( 0, $self->{'writer'}->invocation_count( 'start_page' ) );
    $self->assert_num_equals( 0, $self->{'writer'}->invocation_count( 'end_page' ) );
    $self->assert_str_equals( <<SECTION, $$buf );
    <div class="section">
      <h2><a name="section-1">Empty section</a></h2>
      <p class="no-content">No subreports were generated for this section.</p>
    </div>
SECTION

    $$buf = '';
    $self->{'writer'}{'_one_page'} = 0;
    $section = ($self->{'report'}->sections())[1];
    $section->description( '<para>Section 2 description.</para>' );
    $self->{'writer'}->write_section( $section, 2 );
    $self->assert_num_equals( 1, $self->{'writer'}->invocation_count( 'start_page' ) );
    $self->assert_num_equals( 1, $self->{'writer'}->invocation_count( 'end_page' ) );
    $self->assert_num_equals( 2, $self->{'writer'}->invocation_count( 'write_subreport' ) );
    $self->assert_str_equals( 'section-2.html', $self->{'writer'}->{'_prev'} );
    $self->assert_str_equals( <<SECTION, $$buf );
    <div class="section">
      <h2>Section 2</h2>
      <div class="description">
          <div class="para">Section 2 description.</div>

      </div>
    </div>
SECTION
}

sub test_write_subreport {
    my $self = $_[0];

    $self->set_up_chart_configs();
    my $subreport = new Lire::Report::Subreport( 'test', 'top-files' );
    $subreport->id( 'top-files' );
    $subreport->title( '<Top Files>' );
    $subreport->description( '<para>Subreport description.</para>' );
    $subreport->add_chart_config( $self->{'chart1'} );
    $subreport->add_chart_config( $self->{'chart2'} );

    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}->set_result( 'write_chart', '' );
    $self->{'writer'}->set_result( 'write_table_header', '' );
    $self->{'writer'}->set_result( 'write_table_entries', '' );
    $self->{'writer'}->set_result( 'write_table_footer', '' );
    $self->{'writer'}->write_subreport( $subreport );
    $self->assert_num_equals( 2, $self->{'writer'}->invocation_count( 'write_chart' ) );
    $self->assert_num_equals( 1, $self->{'writer'}->invocation_count( 'write_table_header' ) );
    $self->assert_num_equals( 1, $self->{'writer'}->invocation_count( 'write_table_entries' ) );
    $self->assert_num_equals( 1, $self->{'writer'}->invocation_count( 'write_table_footer' ) );
    $self->assert_str_equals( <<SUBREPORT, $$buf );
      <div class="subreport">
        <h3><a name="top-files">&lt;Top Files></a></h3>
        <div class="description">
            <div class="para">Subreport description.</div>

        </div>
      </div>
SUBREPORT
}

sub test_write_subreport_missing {
    my $self = $_[0];

    my $sub = new_missing Lire::Report::Subreport( 'test', 'test1', '<No good reason>' );
    $sub->id( 'test1' );
    $sub->title( '<My subreport>' );
    $sub->description( '<para>Description.</para>' );
    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}->write_subreport( $sub );
    $self->assert_str_equals( <<EOF, $$buf );
      <div class="subreport">
        <h3><a name="test1">&lt;My subreport></a></h3>
        <div class="description">
            <div class="para">Description.</div>

        </div>
        <div class="warning">
          This report is missing: &lt;No good reason>
        </div>
      </div>
EOF
}

sub test_write_toc_one_page {
    my $self = $_[0];

    $self->set_up_report();
    $self->{'writer'}{'_report'} = $self->{'report'};
    $self->{'writer'}{'_one_page'} = 1;
    $self->{'writer'}->set_result( 'start_page', '' );
    $self->{'writer'}->set_result( 'end_page', '' );

    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}->write_toc();
    $self->assert_num_equals( 0, $self->{'writer'}->invocation_count( 'start_page' ) );
    $self->assert_num_equals( 0, $self->{'writer'}->invocation_count( 'end_page' ) );
    $self->assert_null( $self->{'writer'}{'_toc'}, '_toc != undef' );
    $self->assert_null( $self->{'writer'}{'_prev'}, '_next != undef' );
    $self->assert_null( $self->{'writer'}{'_next'}, '_next != undef' );
    $self->assert_str_equals( <<TOC, $$buf );
    <div class="section">
      <h2>Table Of Contents</h2>
      <ol class="toc">
        <li>
          <div class="toc-section">
            <a href="#section-1" class="link toc-entry">&lt;Section 1></a>
            <ol class="toc-section">
              <li><a href="#top-dirs.0" class="link toc-entry">&lt;Top Dirs></a></li>
            </ol>
          </div>
        </li>
        <li>
          <div class="toc-section">
            <a href="#section-2" class="link toc-entry">Section 2</a>
            <ol class="toc-section">
              <li><a href="#top-files.0" class="link toc-entry">Top Files 1</a></li>
              <li><a href="#top-files.1" class="link toc-entry">Top Files 2</a></li>
            </ol>
          </div>
        </li>
        <li><a href="#schema:test" class="link toc-entry">Simple Schema Definition for Test</a></li>
        <li><a href="#schema:test-extended" class="link toc-entry">Simple Extended Schema for Tests</a></li>
      </ol>
    </div>
TOC
}

sub test_write_toc {
    my $self = $_[0];

    $self->set_up_report();
    $self->{'writer'}{'_report'} = $self->{'report'};
    $self->{'writer'}{'_one_page'} = 0;
    $self->{'writer'}->set_result( 'start_page', '' );
    $self->{'writer'}->set_result( 'end_page', '' );

    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}->write_toc();
    $self->assert_num_equals( 1, $self->{'writer'}->invocation_count( 'start_page' ) );
    $self->assert_num_equals( 1, $self->{'writer'}->invocation_count( 'end_page' ) );
    $self->assert_str_equals( 'index.html', $self->{'writer'}{'_toc'} );
    $self->assert_str_equals('index.html',  $self->{'writer'}{'_prev'} );
    $self->assert_str_equals('section-1.html',  $self->{'writer'}{'_next'} );
    $self->assert_str_equals( <<TOC, $$buf );
    <div class="section">
      <ol class="toc">
        <li>
          <div class="toc-section">
            <a href="section-1.html" class="link toc-entry">&lt;Section 1></a>
            <ol class="toc-section">
              <li><a href="section-1.html#top-dirs.0" class="link toc-entry">&lt;Top Dirs></a></li>
            </ol>
          </div>
        </li>
        <li>
          <div class="toc-section">
            <a href="section-2.html" class="link toc-entry">Section 2</a>
            <ol class="toc-section">
              <li><a href="section-2.html#top-files.0" class="link toc-entry">Top Files 1</a></li>
              <li><a href="section-2.html#top-files.1" class="link toc-entry">Top Files 2</a></li>
            </ol>
          </div>
        </li>
        <li><a href="test.html" class="link toc-entry">Simple Schema Definition for Test</a></li>
        <li><a href="test-extended.html" class="link toc-entry">Simple Extended Schema for Tests</a></li>
      </ol>
    </div>
TOC

}

sub test_write_chart {
    my $self = $_[0];

    $self->set_up_chart_configs();
    my $subreport = new Lire::Report::Subreport( 'test', 'test1' );
    my $buf = $self->{'writer'}{'_fh'}->sref();

    $self->{'writer'}->write_chart( $subreport, $self->{'chart1'} );
    $self->assert_str_equals( <<EOF, $$buf );
        <div class="chart">
          <img src="chart.png" alt="&lt;A chart>" title="&lt;A chart>">
        </div>
EOF
    $self->assert_deep_equals( [ $self->{'mock_chart'},
                                 $self->{'chart1'}, $subreport,
                                 'outputdir', $self->{'tmpdir'},
                                 'format', 'png' ],
                               $self->{'mock_chart'}->get_invocation( 'write_chart' ) );

    $self->{'mock_chart'}->set_result( 'write_chart',
                                       sub { die "An error occured\n" } );
    $$buf = '';
    $self->{'writer'}->write_chart( $subreport, $self->{'chart1'} );
    $self->assert_str_equals( <<EOF, $$buf );
        <div class="warning">
          An error occured while generating the chart: An error occured
        </div>
EOF

    $self->{'mock_chart'}->set_result( 'write_chart', sub { return undef } );
    $$buf = '';
    $self->{'writer'}->write_chart( $subreport, $self->{'chart1'} );
    $self->assert_str_equals( '', $$buf );
}

sub test_start_page {
    my $self = $_[0];

    $self->{'writer'}{'_xhtml'} = 1;
    $self->{'writer'}{'_etag'} = ' />';
    $self->{'writer'}{'_fh'} = undef;
    $self->{'writer'}{'_report'} = new Lire::Report();
    $self->{'writer'}->set_result( 'write_navbar', sub { print {$_[0]{'_fh'}} "[write_navbar]\n" } );
    $self->{'writer'}->set_result( 'write_report_infos', sub { print {$_[0]{'_fh'}} "[write_report_infos]\n" } );
    $self->{'writer'}->set_result( 'write_logo', sub { print {$_[0]{'_fh'}} "[write_logo]\n" } );
    $self->{'writer'}->set_result( 'write_links', sub { print {$_[0]{'_fh'}} "[write_links]\n" } );
    $self->{'writer'}->start_page( 'index.html', '<A report>' );

    $self->assert_not_null( $self->{'writer'}{'_fh'}, 'missing _fh attr' );
    $self->assert( -f "$self->{'tmpdir'}/index.html" );
    $self->{'writer'}{'_fh'}->close();
    $self->assert_str_equals( <<EOD, file_content( "$self->{'tmpdir'}/index.html" ) );
<?xml version="1.0"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">
<html>
  <head>
    <title>Lire Report : &lt;A report></title>
    <meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
[write_links]
  </head>
  <body>
    <div class="title">
[write_logo]
[write_navbar]
      <h2>&lt;A report></h2>
[write_report_infos]
    </div>
EOD

    $self->{'writer'}{'_report'}->title( 'My title' );
    $self->{'writer'}{'_xhtml'} = 0;
    $self->{'writer'}{'_etag'} = '>';
    $self->{'writer'}{'_fh'} = undef;
    $self->{'writer'}->start_page( 'page2.html' );
    $self->assert_not_null( $self->{'writer'}{'_fh'}, 'missing _fh attr' );
    $self->assert( -f "$self->{'tmpdir'}/page2.html" );
    $self->{'writer'}{'_fh'}->close();
    $self->assert_str_equals( <<EOD, file_content( "$self->{'tmpdir'}/page2.html" ) );
<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html40/strict.dtd">
<html>
  <head>
    <title>My title</title>
    <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
[write_links]
  </head>
  <body>
    <div class="title">
[write_logo]
[write_navbar]
      <h2>My title</h2>
[write_report_infos]
    </div>
EOD
}

sub test_end_page {
    my $self = $_[0];

    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}->end_page();
    $self->assert_null( $self->{'writer'}{'_fh'}, '_fh != undef' );
    $self->assert_str_equals( "  </body>\n</html>\n", $$buf );
    return;
}

sub test_html_value {
    my $self = $_[0];

    my $writer = $self->{'writer'};
    my $nolink_string = '<ordinary string>';
    my $http_string = 'http://www.logreport.org/';
    my $email_string = 'Logreport Wawa Team <logrep#r$_wawa-t34m@logreport.org>';

    $self->assert_str_equals( '&lt;ordinary string>',
                              $writer->html_value( $nolink_string ) );
    $self->assert_str_equals( '<a href="http://www.logreport.org/" class="link">http://www.logreport.org/</a>',
                              $writer->html_value( $http_string ) );
    $self->assert_str_equals( '<a href="mailto:logrep#r$_wawa-t34m@logreport.org" class="link">Logreport Wawa Team &lt;logrep#r$_wawa-t34m@logreport.org></a>',
                              $writer->html_value( $email_string ) );

    $self->assert_str_equals( '<a href="http://www.logreport.org/" class="link">htt...org/</a>',
                              $writer->html_value( $http_string, 10 ) );
    $self->assert_str_equals( 'long...text',
                              $writer->html_value( 'long useless text', 11 ) );

    return;
}

sub test_write_table_header {
    my $self = $_[0];

    $self->set_up_subreport();
    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}{'_one_page'} = 0;
    $self->{'writer'}->write_table_header( $self->{'subreport'} );
    $self->assert_str_equals( <<EOF, $$buf );
        <table>
          <thead>
            <tr>
              <th colspan="2"><a href="test.html#user">&lt;User></a></th>
              <th>Downloads</th>
              <th>Size</th>
            </tr>
            <tr>
              <th class="empty"></th>
              <th><a href="test.html#file">&lt;File></a></th>
              <th class="empty"></th>
              <th class="empty"></th>
            </tr>
          </thead>
EOF

    return;
}

sub test_write_table_footer {
    my $self = $_[0];

    $self->set_up_subreport();
    $self->{'subreport'}->nrecords( 100 );
    $self->{'subreport'}->set_summary_value( 'count', 'content' => '356' );
    $self->{'subreport'}->set_summary_value( 'size', 'value' => 1024 * 1024,
                                             'content' => '1 &' );
    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}->write_table_footer( $self->{'subreport'} );
    $self->assert_str_equals( <<'EOF', $$buf );
          <tfoot>
            <tr>
              <td colspan="2">Total for 100 records</td>
              <td class="int">356</td>
              <td class="bytes">1 &amp;</td>
            </tr>
          </tfoot>
EOF

}

sub test_write_table_entries {
    my $self = $_[0];

    $self->set_up_subreport();
    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}->write_table_entries( $self->{'subreport'} );
    $self->assert_str_equals( <<EOF, $$buf );
          <tbody>
            <tr>
              <td colspan="4" class="empty">There is no entries in this table.</td>
            </tr>
          </tbody>
        </table>
EOF

    $self->set_up_subreport_data();
    $$buf = '';
    $self->{'writer'}->write_table_entries( $self->{'subreport'} );
    $self->assert_str_equals( <<EOF, $$buf );
          <tbody>
            <tr>
              <td colspan="2" class="string">flacoste</td>
              <td class="int">9</td>
              <td class="bytes">53</td>
            </tr>
            <tr>
              <td class="empty"></td>
              <td class="string">page1.html</td>
              <td class="int">5</td>
              <td class="bytes">50</td>
            </tr>
            <tr>
              <td class="empty"></td>
              <td class="string">page2.html</td>
              <td class="int">4</td>
              <td class="bytes">3</td>
            </tr>
            <tr>
              <td colspan="2" class="string">&lt;wsourdeau></td>
              <td class="int">5</td>
              <td class="bytes">10</td>
            </tr>
            <tr>
              <td class="empty"></td>
              <td class="string">page3.html</td>
              <td class="int">3</td>
              <td class="bytes">6</td>
            </tr>
            <tr>
              <td class="empty"></td>
              <td class="string">page2.html</td>
              <td class="int">1</td>
              <td class="bytes">2</td>
            </tr>
            <tr>
              <td class="empty"></td>
              <td class="string">page4.html</td>
              <td class="int">1</td>
              <td class="bytes">2</td>
            </tr>
          </tbody>
        </table>
EOF
}

sub test_write_navbar {
    my $self = $_[0];

    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}{'_one_page'} = 1;
    $self->{'writer'}->write_navbar();
    $self->assert_str_equals( '', $$buf );

    $self->{'writer'}{'_one_page'} = 0;
    $self->{'writer'}{'_next'} = 'section-1.html';
    $self->{'writer'}->write_navbar();
    $self->assert_str_equals( <<EOD, $$buf );
    <div class="navigation">
      <a href="section-1.html" class="navigation"><img src="next.png" alt="Next" title="Next" id="next-button"></a>
    </div>
EOD
    $$buf = '';
    $self->{'writer'}{'_toc'} = 'index.html';
    $self->{'writer'}{'_prev'} = 'section-1.html';
    $self->{'writer'}{'_next'} = undef;
    $self->{'writer'}->write_navbar();
    $self->assert_str_equals( <<EOD, $$buf );
    <div class="navigation">
      <a href="index.html" class="navigation"><img src="toc.png" alt="Table Of Contents" title="Table Of Contents" id="toc-button"></a>
      <a href="section-1.html" class="navigation"><img src="prev.png" alt="Previous" title="Previous" id="prev-button"></a>
    </div>
EOD

    return;
}

sub test_write_report_infos {
    my $self = $_[0];

    my $report = $self->{'writer'}{'_report'} = new Lire::Report();
    $report->date( timelocal( 13, 17, 23, 28, 03, 2004 ) );
    $report->timespan_start( timelocal( 14, 15, 18, 28, 03, 2004 ) );
    $report->timespan_end( timelocal( 59, 19, 18, 28, 03, 2004 ) );

    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}->write_report_infos();
    $self->assert_str_equals( <<_EOD, $$buf );
    <div class="report-infos">
      Report generated on: 2004-04-28 23:17:13 UTC<br>
      Reporting on period: 2004-04-28 18:15:14 UTC - 2004-04-28 18:19:59 UTC<br>
    </div>
_EOD

    $$buf = '';
    $report->timespan_start( undef );
    $self->{'writer'}->write_report_infos();
    $self->assert_str_equals( <<_EOD, $$buf );
    <div class="report-infos">
      Report generated on: 2004-04-28 23:17:13 UTC<br>
      Reporting on period: Unknown Period<br>
    </div>
_EOD

    return;
}

sub test_write_links {
    my $self = $_[0];

    my $report = $self->{'writer'}{'_report'} = new Lire::Report();
    $report->add_section( new Lire::Report::Section( 'Section 1' ) );
    $report->add_section( new Lire::Report::Section( 'Section 2' ) );
    $report->add_section( new Lire::Report::Section( 'Section 3' ) );

    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}{'_one_page'} = 1;
    $self->{'cfg'}{'lr_html_css'} = '/usr/share/lire/html/lire-html.css';
    $self->{'writer'}->write_links();

    $self->assert_str_equals( <<_EOD, $$buf );
    <link rel="stylesheet" href="lire-html.css" type="text/css">
    <link rel="author" href="http://www.logreport.org/" title="The LogReport Foundation">
    <link rel="help" href="http://logreport.org/doc/lire/" title="Lire documentation">
_EOD

    $$buf = '';
    $self->{'writer'}{'_one_page'} = 0;
    $self->{'writer'}{'_toc'} = 'index.html';
    $self->{'writer'}{'_next'} = 'section-2.html';
    $self->{'writer'}{'_prev'} = 'index.html';
    $self->{'cfg'}{'lr_html_css'} = 'http://www.logreport.org/lire.css';
    $self->{'writer'}->write_links();

    $self->assert_str_equals( <<_EOD, $$buf );
    <link rel="stylesheet" href="http://www.logreport.org/lire.css" type="text/css">
    <link rel="author" href="http://www.logreport.org/" title="The LogReport Foundation">
    <link rel="help" href="http://logreport.org/doc/lire/" title="Lire documentation">
    <link rel="start" href="index.html" title="Table Of Contents">
    <link rel="first" href="section-1.html" title="First Section">
    <link rel="last" href="section-3.html" title="Last Section">
    <link rel="prev" href="index.html" title="Previous Page">
    <link rel="next" href="section-2.html" title="Next Page">
_EOD
}

sub test_write_logo {
    my $self = $_[0];

    $self->{'cfg'}{'lr_logo'} = undef;

    my $buf = $self->{'writer'}{'_fh'}->sref();
    $self->{'writer'}->write_logo();
    $self->assert_str_equals( '', $$buf );

    $self->{'cfg'}{'lr_logo'} = '/usr/share/lire/html/lire.png';
    $self->{'cfg'}{'lr_logo_url'} = 'http://www.logreport.org/';

    $self->{'writer'}->write_logo();
    $self->assert_str_equals( <<EOD, $$buf );
    <div class="logo">
      <a href="http://www.logreport.org/" class="logo"><img src="lire.png" class="logo"></a>
    </div>
EOD

    $$buf = '';
    $self->{'cfg'}{'lr_logo'} = 'http://www.logreport.org/lire.png';
    $self->{'cfg'}{'lr_logo_url'} = '';
    $self->{'writer'}->write_logo();
    $self->assert_str_equals( <<EOD, $$buf );
    <div class="logo">
      <img src="http://www.logreport.org/lire.png" class="logo">
    </div>
EOD

}

sub set_up_write_report {
    my $self = $_[0];

    $self->set_up_chart_types();
    $self->{'keep_tmpdir'} = $ENV{'KEEP_HTML'};

    my $parser = new Lire::ReportParser::ReportBuilder();
    $self->{'report'} = $parser->parsefile( "$self->{'testdir'}/data/TestReport_daily_jan25_2003.xml" );

    $self->{'writer'} = new Lire::ReportParser::HTMLWriter();
    $self->{'cfg'}{'lr_logo'} = "$self->{'testdir'}/../html/lire.png";
    $self->{'cfg'}{'lr_logo_url'} = "http://www.logreport.org/";
    $self->{'cfg'}{'lr_html_css'} = "$self->{'testdir'}/../html/lire-html.css";

    no warnings 'redefine';
    $self->{'_or_ac_path'} = \&Lire::ReportParser::HTMLWriter::ac_path;
    *Lire::ReportParser::HTMLWriter::ac_path = sub { return "$self->{'testdir'}/../" };

    return;
}

sub test_write_report_multi_page {
    my $self = $_[0];

    $self->set_up_write_report();
    $self->{'writer'}->write_report( $self->{'report'}, $self->{'tmpdir'},
                                     'xhtml' => 1, 'one_page' => 0 );
    foreach my $file ( ( 'index.html', 'section-1.html', 'toc.png',
                         'section-2.html', 'section-3.html', 'test.html',
                         'test-extended.html', 'test-derived.html',
                         'lire.png', 'prev.png', 'next.png', 
                         'lire-html.css' ) ) 
    {
        $self->assert( -s "$self->{'tmpdir'}/$file",
                       "missing '$self->{'tmpdir'}/$file'" );
    }

    print "\nMulti-pages HTML report generated in $self->{'tmpdir'}/index.html\n"
      if $self->{'keep_tmpdir'};
}

sub test_write_report_one_page {
    my $self = $_[0];

    $self->set_up_write_report();
    $self->{'writer'}->write_report( $self->{'report'}, $self->{'tmpdir'},
                                     'xhtml' => 0, 'one_page' => 1 );

    foreach my $file ( ( 'index.html', 'lire.png', 'lire-html.css' ) ) {
        $self->assert( -s "$self->{'tmpdir'}/$file",
                       "missing '$self->{'tmpdir'}/$file'" );
    }
        $self->assert( ! -f "$self->{'tmpdir'}/toc.png",
                       "file '$self->{'tmpdir'}/toc.png' exists" );

    print "\nOne page HTML report generated in $self->{'tmpdir'}/index.html\n"
      if $self->{'keep_tmpdir'};
}

1;
