#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include "LoggerMethods.h"

#include <Exception.h>

#include <stdio.h>
#include <syslog.h>
#include <errno.h>

//#include <unistd.h>	// isatty

using namespace Log;
using namespace std;

//
// FileMethod
//

FileMethod::FileMethod(const string& file) throw (FileException) : out(0), fname(file)
{
	out = fopen(fname.c_str(), "at");
	if (!out)
		throw FileException(errno, "opening logfile " + fname);
}

FileMethod::~FileMethod() throw ()
{
	if (out)
		fclose((FILE*)out);
}
	
void FileMethod::output(LogLevel level, const char* tag, const char* func,
		const string& msg) throw ()
{
	time_t now = time(NULL);
	struct tm pnow;
	localtime_r(&now, &pnow);
	char timebuf[20];
	/*
	 * Strftime specifiers used here:
	 *	%b      The abbreviated month name according to the current locale.
	 *	%d      The day of the month as a decimal number (range 01 to 31).
	 *	%e      Like %d, the day of the month as a decimal number, but a
	 *			leading zero is replaced by a space. (SU)
	 *	%T      The time in 24-hour notation (%H:%M:%S). (SU)
	 */
	strftime(timebuf, 20, "%b %e %T", &pnow);
	fprintf((FILE*)out, "%s %s %s: %.*s\n", timebuf, tag, func, PFSTR(msg));
	if (level >= WARN)
		fflush((FILE*)out);
}

//
// StderrMethod
//

void StderrMethod::output(LogLevel level, const char* tag, const char* func,
		const string& msg) throw ()
{
	fprintf(stderr, "%s %s: %.*s\n", tag, func, PFSTR(msg));
}

//
// SyslogMethod
//

SyslogMethod::SyslogMethod(const string& ident, int facility) throw ()
	: ident(ident), facility(facility)
{
	openlog(ident.c_str(), LOG_PID, facility);
}

SyslogMethod::~SyslogMethod() throw ()
{
	closelog();
}

void SyslogMethod::output(LogLevel level, const char* tag, const char* func,
		const string& msg) throw ()
{
	int prio;
	switch (level)
	{
		case DEBUG: prio = LOG_DEBUG; break;
		case INFO: prio = LOG_INFO; break;
		case UNUSUAL: prio = LOG_NOTICE; break;
		case WARN: prio = LOG_WARNING; break;
		case ERR: prio = LOG_ERR; break;
		case CRIT: prio = LOG_CRIT; break;
		//LOG_ALERT, LOG_EMERG
		default: prio = LOG_WARNING; break;
	}
	syslog(prio, "%s %s: %.*s", tag, func, PFSTR(msg));
}

void SyslogMethod::setupForkedChild() throw ()
{
	closelog();
	openlog(ident.c_str(), LOG_PID, facility);
}


/*

static enum stderr_state_type stderr_state = Yes;
static bool debug_output = false;

void set_debug(bool val)
{
	debug_output = val;
}

void set_stderr(enum stderr_state_type val)
{
	switch (val)
	{
		case Yes:
			call_closelog(); break;
		case No:
			call_openlog(); break;
		case Restricted:
			call_openlog(); break;
	}
	stderr_state = val;
}

void log_debug(const char* fmt, ...) ATTR_PRINTF(1, 2)
{
	if (debug_output)
	{
		va_list ap;
		va_start(ap, fmt);
		switch (stderr_state)
		{
			case Yes:
				//if (isatty(fileno(stderr)))
				//{
					vfprintf(stderr, fmt, ap);
					fputc('\n', stderr);
				//}
				break;
			case No:
			case Restricted:
				vsyslog(LOG_DEBUG, fmt, ap);
				break;
		}
		va_end(ap);
	}
}

void log_info(const char* fmt, ...) ATTR_PRINTF(1, 2)
{
	va_list ap;
	va_start(ap, fmt);
	switch (stderr_state)
	{
		case Yes:
			//if (isatty(fileno(stderr)))
			//{
				vfprintf(stderr, fmt, ap);
				fputc('\n', stderr);
			//}
			break;
		case No:
		case Restricted:
			vsyslog(LOG_INFO, fmt, ap);
			break;
	}
	va_end(ap);
}

void log_warn(const char* fmt, ...) ATTR_PRINTF(1, 2)
{
	va_list ap;
	va_start(ap, fmt);
	switch (stderr_state)
	{
		case Yes:
		case Restricted:
			vfprintf(stderr, fmt, ap);
			fputc('\n', stderr);
			break;
		case No:
			vsyslog(LOG_WARNING, fmt, ap);
			break;
	}
	va_end(ap);
}

void log_error(const char* fmt, ...) ATTR_PRINTF(1, 2)
{
	va_list ap;
	va_start(ap, fmt);
	switch (stderr_state)
	{
		case Yes:
		case Restricted:
			vfprintf(stderr, fmt, ap);
			fputc('\n', stderr);
			break;
		case No:
			vsyslog(LOG_ERR, fmt, ap);
			break;
	}
	va_end(ap);
}

void log_crit(const char* fmt, ...) ATTR_PRINTF(1, 2)
{
	va_list ap;
	va_start(ap, fmt);
	switch (stderr_state)
	{
		case Yes:
		case Restricted:
			vfprintf(stderr, fmt, ap);
			fputc('\n', stderr);
			break;
		case No:
			vsyslog(LOG_CRIT, fmt, ap);
			break;
	}
	va_end(ap);
}

*/

// vim:set ts=4 sw=4:
