package latexDraw.ui.dialog;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;

import javax.swing.*;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;

import latexDraw.figures.Arc;
import latexDraw.figures.Figure;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.DrawPanel;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines the frame which allows to  change the parameters of arcs and wedges.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class ParametersArcFrame extends AbstractParametersFrame
{
	private static final long serialVersionUID = 1L;

	/** Allows to change the x-coordinates of the centre of the figure */
	protected JSpinner centerXField;
	
	/** Allows to change the y-coordinates of the centre of the figure */
	protected JSpinner centerYField;
	
	/** Allows to change the radius of the circle */
	protected JSpinner radiusField;
	
	/** Allows to set the start angle of the arc */
	protected JSpinner startAngle;
	
	/** Allows to set the end angle of the arc */
	protected JSpinner endAngle;
	
	/** This choice allows to choice the kind of the arc */
	protected JComboBox arcChoice;
	
	/** Allows to display the points of the arc */
	protected JCheckBox showPointCB;
	
	/** The label of the start angle field */
	public static final String LABEL_START_ANGLE = LaTeXDrawLang.getDialogFrameString("ParametersArcFrame.0"); //$NON-NLS-1$
	
	/** The label of the end angle field */
	public static final String LABEL_END_ANGLE = LaTeXDrawLang.getDialogFrameString("ParametersArcFrame.1"); //$NON-NLS-1$
	
	public static final String NAME_TABBEDPANE_GENERAL = LABEL_GENERAL;
	
	
	
	/**
	 * The constructor by default
	 * @param p The panel containing the drawing
	 */
	public ParametersArcFrame(Frame parent, DrawPanel p, boolean attachToMainFrame) 
	{
		super(parent, p, attachToMainFrame);
		try
		{
	  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
	  		JTabbedPane tabbedPane = new JTabbedPane();
	  		JPanel pButton  = createButtonsPanel(this);
	  		JPanel pGeneral = new JPanel();
	  		pGeneral.setLayout(new BoxLayout(pGeneral, BoxLayout.Y_AXIS));
	  		JPanel coordP = new JPanel(new GridBagLayout());
	  		GridBagConstraints constraint = new GridBagConstraints();
	    	constraint.gridx = 0;
	     	constraint.gridy = 0;
	     	constraint.gridwidth = 1;
	     	constraint.gridheight = 1;
	     	constraint.weightx = 0.1;
	     	constraint.weighty = 0.1;
	     	constraint.fill = GridBagConstraints.NONE;
	     	constraint.anchor = GridBagConstraints.CENTER;
	     	coordP.add(new JLabel(LABEL_CENTER_X), constraint);
	  		
	     	SpinnerModel model = new SpinnerNumberModel(0,0,10000,0.1);
	     	centerXField = new JSpinner(model);
			centerXField.setEditor(new JSpinner.NumberEditor(centerXField, "0.0"));//$NON-NLS-1$
			centerXField.addChangeListener(this);
	     	constraint.gridx = 1;
	     	centerXField.setName(LABEL_CENTER_X);
	     	coordP.add(centerXField, constraint);

	    	constraint.gridx = 3;
	     	coordP.add(new JLabel(LABEL_CENTER_Y), constraint);
	  		
	     	model = new SpinnerNumberModel(0,0,10000,0.1);
	     	centerYField = new JSpinner(model);
	     	centerYField.setEditor(new JSpinner.NumberEditor(centerYField, "0.0"));//$NON-NLS-1$
			centerYField.addChangeListener(this);
	     	constraint.gridx = 4;	
	     	centerYField.setName(LABEL_CENTER_Y);
	     	coordP.add(centerYField, constraint);
	     	
	    	constraint.gridx = 0;	constraint.gridy++;
	     	coordP.add(new JLabel(LABEL_RADIUS), constraint);
	  		
	     	model = new SpinnerNumberModel(0.1,0.1,10000,0.1);
	     	radiusField = new JSpinner(model);
	     	radiusField.setEditor(new JSpinner.NumberEditor(radiusField, "0.0"));//$NON-NLS-1$
	     	radiusField.addChangeListener(this);
	     	constraint.gridx = 1;	
	     	radiusField.setName(LABEL_RADIUS);
	     	coordP.add(radiusField, constraint);
	     	
	     	constraint.gridx = 0;	constraint.gridy++;
	     	coordP.add(new JLabel(LABEL_ROTATION), constraint);
	     	constraint.gridx++;
	     	coordP.add(rotationField, constraint);
	     	
	     	coordP.setBorder(new CompoundBorder(new TitledBorder(null, LABEL_POSITION_DIMENSIONS, 
							  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	
	     	JPanel arcP = new JPanel(new GridBagLayout());
	     	constraint.gridx=0;	constraint.gridy=0;
	     	arcP.add(new JLabel(LaTeXDrawLang.getDialogFrameString("ParametersArcFrame.2")), constraint); //$NON-NLS-1$
	     	
	     	constraint.gridx++;
	     	arcChoice = Arc.createArcChoice(this);
	     	arcP.add(arcChoice, constraint);
	     	
	     	constraint.gridy++; 	constraint.gridx = 0;
	     	arcP.add(new JLabel(LABEL_START_ANGLE), constraint);
	     	
	     	constraint.gridx++;
	     	model = new SpinnerNumberModel(0,-360,360,0.1);
	     	startAngle = new JSpinner(model);
	     	startAngle.setEditor(new JSpinner.NumberEditor(startAngle, "0.0"));//$NON-NLS-1$
	     	startAngle.addChangeListener(this);
	     	startAngle.setName(LABEL_START_ANGLE);
	     	arcP.add(startAngle, constraint);
	     	
	     	constraint.gridx++;	
	     	arcP.add(new JLabel(LABEL_END_ANGLE), constraint);
	     	
	     	constraint.gridx++;
	     	model = new SpinnerNumberModel(0,-360,360,0.1);
	     	endAngle = new JSpinner(model);
	     	endAngle.setEditor(new JSpinner.NumberEditor(endAngle, "0.0"));//$NON-NLS-1$
	     	endAngle.addChangeListener(this);
	     	endAngle.setName(LABEL_END_ANGLE);
	     	arcP.add(endAngle, constraint);
	     	
	     	arcP.setBorder(new CompoundBorder(new TitledBorder(null, LaTeXDrawLang.getOthersString("Arc.arc"),  //$NON-NLS-1$
					  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	
	     	pGeneral.add(coordP);
	     	pGeneral.add(createBordersPanel(false));
	     	pGeneral.add(createFillPanel());
	     	pGeneral.add(arcP);
	     	
	     	JPanel showP = new JPanel(new BorderLayout());
	     	showPointCB = new JCheckBox(LABEL_SHOWPOINTCB);
	     	showPointCB.addActionListener(this);
	     	showPointCB.setActionCommand(LABEL_SHOWPOINTCB);
	     	showP.add(showPointCB);
	     	
	     	showP.setBorder(new CompoundBorder(new TitledBorder(null, TITLE_TABBED_PANE_OTHERS, 
					  TitledBorder.LEFT, TitledBorder.TOP), new EmptyBorder(0,0,0,0)));
	     	
	     	JPanel pOther = new JPanel();
	     	pOther.setLayout(new BoxLayout(pOther, BoxLayout.Y_AXIS));
	     	pOther.add(createShadowPanel());
	     	pOther.add(createGradientPanel());
	     	pOther.add(createHatchingPanel());
	     	pOther.add(showP);
	     	
	     	tabbedPane.addTab(TITLE_TABBED_PANE_GENERAL, pGeneral);
	     	tabbedPane.addTab(NAME_TABBEDPANE_ARROW, createArrowPanel(true, false));
	     	tabbedPane.addTab(TITLE_TABBED_PANE_OTHERS, pOther);
	     	BoxLayout bl = new BoxLayout(getContentPane(), BoxLayout.Y_AXIS);
	     	getContentPane().setLayout(bl);
	     	getContentPane().add(glimpsePanelZoom);
	     	getContentPane().add(tabbedPane);
			getContentPane().add(pButton);
	  		pack();
	  		setLocation(dim.width/2-getWidth()/2, dim.height/2-getHeight()/2);
			setVisible(false);
		}
		catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}

	
	
	@Override
	public void actionPerformed(ActionEvent e) 
	{
		Object o = e.getSource();
		
		try
		{
			if(o instanceof JCheckBox)
			{
				String label = ((JCheckBox)o).getActionCommand();
				
				if(label.equals(LABEL_SHOWPOINTCB))
				{
					setModified(true);
					((Arc)glimpsePanel.glimpseFigure).setShowPoints(showPointCB.isSelected());
					glimpsePanel.repaint();
					return ;
				}
			}
		}catch(Exception ex) 
		{ 
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		
		super.actionPerformed(e);
	}
	
	
	

	@Override
	public void saveParameters()
	{
		super.saveParameters();
		
		LaTeXDrawPoint2D p1, p2, p3, p4;
		Arc a = (Arc)figure;
		p1 = a.getBordersPoint(0); 	p2 = a.getBordersPoint(1);
		p3 = a.getBordersPoint(2); 	p4 = a.getBordersPoint(3);
		
		float radius = Float.valueOf(radiusField.getValue().toString());
		float x = Float.valueOf(centerXField.getValue().toString());
		float y = Float.valueOf(centerYField.getValue().toString());
		float start = Float.valueOf(startAngle.getValue().toString());
		float end = Float.valueOf(endAngle.getValue().toString());
		String type = (String)arcChoice.getSelectedItem();
		
		p1.x = x - radius;
		p2.x = x + radius;
		p3.x = x - radius;
		p4.x = x + radius;

		p1.y = y - radius;
		p2.y = y - radius;
		p3.y = y + radius;
		p4.y = y + radius;

		((Arc)figure).setStartAngle(Math.toRadians(start));
		((Arc)figure).setEndAngle(Math.toRadians(end));		
		((Arc)figure).setType(type);		
		((Arc)figure).updateCenterStartEndDelimitors();
		((Arc)figure).setShowPoints(showPointCB.isSelected());
		figure.updateShape();
		drawPanel.getDraw().updateBorders();
		drawPanel.updateDraw(true);
	}
	
	
	

	@Override
	public void setFigureFrameField()
	{	
		LaTeXDrawPoint2D NW, SE;
		NW = figure.getTheNWPoint();
		SE = figure.getTheSEPoint();
		
		radiusField.setValue((SE.x-NW.x)/2.);
		centerXField.setValue((NW.x+SE.x)/2.);
		centerYField.setValue((NW.y+SE.y)/2.);		
		startAngle.setValue(Math.toDegrees(((Arc)figure).getStartAngle()));
		endAngle.setValue(Math.toDegrees(((Arc)figure).getEndAngle()));		
		arcChoice.setSelectedItem(((Arc)figure).getTypeName());
		showPointCB.setSelected(((Arc)figure).isShowPoints());
	}



	
	@Override
	public void setVisible(boolean visible, Figure selected, boolean deleteOnCancel, boolean isFramedBox) 
	{
		try
		{
			if(!(selected instanceof Arc))
				throw new ClassCastException();

			super.setVisible(visible, selected, deleteOnCancel, isFramedBox);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}	
	}


	
	@Override
	public void setDefaultsValues()
	{
		super.setDefaultsValues();
		
		try
		{
			Arc a = (Arc)glimpsePanel.getGlimpseFigure();
			startAngle.setValue(Math.toDegrees(Arc.DEFAULT_STARTANGLE));
			endAngle.setValue(Math.toDegrees(Arc.DEFAULT_ENDANGLE));
			arcChoice.setSelectedItem(Arc.DEFAULT_TYPE);
			a.setEndAngle(Arc.DEFAULT_ENDANGLE);
			a.setStartAngle(Arc.DEFAULT_STARTANGLE);
			a.setShowPoints(Arc.DEFAULT_SHOWPOINTS);
			showPointCB.setSelected(Arc.DEFAULT_SHOWPOINTS);
			a.setShowPoints(Arc.DEFAULT_SHOWPOINTS);
			
			glimpsePanel.repaint();
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	

	
	
	@Override
	public void stateChanged(ChangeEvent e) 
	{
		try
		{
			Object o = e.getSource();
			
			Arc f = (Arc) glimpsePanel.getGlimpseFigure();
			
			if(o instanceof JSpinner)
			{
				String name = ((JSpinner)o).getName();
				double v = Double.valueOf(((JSpinner)o).getValue().toString()).doubleValue();
				
				if(name.equals(LABEL_START_ANGLE))
				{
					setModified(true);
					f.setStartAngle(Math.toRadians(v));
					glimpsePanel.repaint();
					
					return ;
				}
				
				if(name.equals(LABEL_END_ANGLE))
				{
					setModified(true);
					f.setEndAngle(Math.toRadians(v));
					glimpsePanel.repaint();
					
					return ;
				}
				
				if(name.equals(LABEL_CENTER_X))
				{
					setModified(true);
					LaTeXDrawPoint2D p1, p2, p3, p4;
					p1 = f.getBordersPoint(0); 	p2 = f.getBordersPoint(1);
					p3 = f.getBordersPoint(2); 	p4 = f.getBordersPoint(3);
					double radius = Double.valueOf(radiusField.getValue().toString()).doubleValue();
					
					p1.x = v - radius;
					p2.x = v + radius;
					p3.x = v - radius;
					p4.x = v + radius;
					
					f.updateGravityCenter();
					f.updateShape();
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_CENTER_Y))
				{
					setModified(true);
					LaTeXDrawPoint2D p1, p2, p3, p4;
					p1 = f.getBordersPoint(0); 	p2 = f.getBordersPoint(1);
					p3 = f.getBordersPoint(2); 	p4 = f.getBordersPoint(3);
					double radius = Double.valueOf(radiusField.getValue().toString()).doubleValue();
					
					p1.y = v - radius;
					p2.y = v - radius;
					p3.y = v + radius;
					p4.y = v + radius;
					
					f.updateGravityCenter();
					f.updateShape();
					glimpsePanel.repaint();
					return ;
				}
				
				if(name.equals(LABEL_RADIUS))
				{
					setModified(true);
					LaTeXDrawPoint2D p1, p2, p3, p4;
					p1 = f.getBordersPoint(0); 	p2 = f.getBordersPoint(1);
					p3 = f.getBordersPoint(2); 	p4 = f.getBordersPoint(3);
					double y = Double.valueOf(centerYField.getValue().toString()).doubleValue();
					double x = Double.valueOf(centerXField.getValue().toString()).doubleValue();
					
					p1.x = x - v;
					p2.x = x + v;
					p3.x = x - v;
					p4.x = x + v;
					p1.y = y - v;
					p2.y = y - v;
					p3.y = y + v;
					p4.y = y + v;
					
					glimpsePanel.repaint();					
					return ;
				}
			}
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		
		super.stateChanged(e);
	}
	
	


	@Override
	public void itemStateChanged(ItemEvent e) 
	{
		try
		{
			Arc f = (Arc) glimpsePanel.getGlimpseFigure();
			Object o = e.getSource();
			
			if(o instanceof JComboBox)
			{
				String name = ((JComboBox)o).getName();
				
				if(name.equals(Arc.LABEL_ARCCHOICE))
				{
					if(((String)((JComboBox)o).getSelectedItem()).equals(Arc.TYPE_ARC))
					{
						arrow1Choice.setEnabled(true);
						arrow2Choice.setEnabled(true);
					}
					else
					{
						arrow1Choice.setEnabled(false);
						arrow2Choice.setEnabled(false);
					}
					
					updateEnableParamsArrow();
					
					setModified(true);
					f.setType(((JComboBox)o).getSelectedItem().toString());
					glimpsePanel.repaint();
					
					return ;
				}
			}
			
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
		super.itemStateChanged(e);
	}
}
