//
// ========================================================================
// Copyright (c) 1995 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
// which is available at https://www.apache.org/licenses/LICENSE-2.0.
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.http3.generator;

import java.nio.ByteBuffer;
import java.util.function.Consumer;

import org.eclipse.jetty.http3.frames.DataFrame;
import org.eclipse.jetty.http3.frames.Frame;
import org.eclipse.jetty.http3.frames.FrameType;
import org.eclipse.jetty.http3.internal.VarLenInt;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.util.BufferUtil;

public class DataGenerator extends FrameGenerator
{
    private final boolean useDirectByteBuffers;

    public DataGenerator(ByteBufferPool bufferPool, boolean useDirectByteBuffers)
    {
        super(bufferPool);
        this.useDirectByteBuffers = useDirectByteBuffers;
    }

    @Override
    public int generate(ByteBufferPool.Accumulator accumulator, long streamId, Frame frame, Consumer<Throwable> fail)
    {
        DataFrame dataFrame = (DataFrame)frame;
        return generateDataFrame(accumulator, dataFrame);
    }

    private int generateDataFrame(ByteBufferPool.Accumulator accumulator, DataFrame frame)
    {
        ByteBuffer data = frame.getByteBuffer();
        int dataLength = data.remaining();
        int headerLength = VarLenInt.length(FrameType.DATA.type()) + VarLenInt.length(dataLength);
        RetainableByteBuffer header = getByteBufferPool().acquire(headerLength, useDirectByteBuffers);
        ByteBuffer byteBuffer = header.getByteBuffer();
        BufferUtil.clearToFill(byteBuffer);
        VarLenInt.encode(byteBuffer, FrameType.DATA.type());
        VarLenInt.encode(byteBuffer, dataLength);
        byteBuffer.flip();
        accumulator.append(header);
        accumulator.append(RetainableByteBuffer.wrap(data));
        return headerLength + dataLength;
    }
}
