/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 * 
 * Contributions from:
 *     Ryan Wagoner <ryan@wgnrs.dynu.com>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#define PLUGIN_NAME "ICQ-AIM IMSpector protocol plugin"
#define PROTOCOL_NAME "ICQ-AIM"
#define PROTOCOL_PORT 5190

#define COOKIE_SOCKET "/tmp/.imspectoricqcookie"

extern "C"
{
	bool initprotocolplugin(struct protocolplugininfo &pprotocolplugininfo,
		class Options &options, bool debugmode);
	void closeprotocolplugin(void);
	int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer, 
		int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress);
};

#define GET_CALL_ARGS p, startp, lengthp
#define GET_ARGS char **p, char *startp, int lengthp
#define GET_CHECK_P(size) if (*p > startp + lengthp - size) return 0;
#define GET_TYPE(type) memcpy(rc, *p, sizeof(type)); *p += sizeof(type);

#pragma pack(2)

struct flap
{
	uint8_t header;
	uint8_t channel;
	uint16_t sequence;
	uint16_t datalength;
};

struct snac
{
	uint16_t family;
	uint16_t subtype;
	uint16_t flags;
	uint32_t requestid;
};

#pragma pack()

int loginpacket(GET_ARGS, bool outgoing, bool md5login, std::string &clientaddress);
int servercookiepacket(GET_ARGS, bool outgoing, std::string &clientaddress);
int snacpacket(GET_ARGS, bool outgoing, std::vector<struct imevent> &imevents, std::string &clientaddress);
void snacpacketunknown(struct snac &mysnac);
int getmessage(GET_ARGS, std::string &message, int &mestart, int &melength);
int getrtfmessage(GET_ARGS, std::string &message, int &mestart, int &melength, bool oldstyle);
void logmessage(bool outgoing, int type, std::string message, std::vector<struct imevent> &imevents,
	std::string clientaddress, int mestart, int melength);
	
bool getsnac(GET_ARGS, struct snac *rc);
bool getbyte(GET_ARGS, uint8_t *rc);
bool getbytes(GET_ARGS, char *bytes, int length);
bool getword(GET_ARGS, uint16_t *rc);
bool getwordle(GET_ARGS, uint16_t *rc);
bool getlong(GET_ARGS, uint32_t *rc);
bool getlengthbytes(GET_ARGS, char *string);
bool getwordlelengthbytes(GET_ARGS, char *string);
bool gettlv(GET_ARGS, uint16_t *tag, uint16_t *length, char *value);
bool gettlvptr(GET_ARGS, uint16_t *tag, uint16_t *length, char **value);
 
void cookiemonster(void);
bool setcookieuin(std::string cookie, std::string uin);
std::string getcookieuin(std::string cookie);
std::string cookietohex (int length, char *cookie);

std::string localid = "Unknown";
std::string remoteid = "Unknown";
int packetcount = 0;
bool tracing = false;
bool tracingerror = false;
bool localdebugmode = false;
iconv_t iconv_utf16be_utf8 = (iconv_t) -1;

bool initprotocolplugin(struct protocolplugininfo &protocolplugininfo,
	class Options &options, bool debugmode)
{
	if (options["icq_protocol"] != "on") return false;

	localdebugmode = debugmode;

	protocolplugininfo.pluginname = PLUGIN_NAME;
	protocolplugininfo.protocolname = PROTOCOL_NAME;
#ifdef HAVE_SSL
	if (options["icq_ssl"] == "on")
	{
		syslog(LOG_INFO, PROTOCOL_NAME ": Monitoring SSL");
		
		protocolplugininfo.sslport = htons(PROTOCOL_PORT);
	}
	else
		protocolplugininfo.port = htons(PROTOCOL_PORT);
#else
	protocolplugininfo.port = htons(PROTOCOL_PORT);
#endif

	/* Initialize character set conversion descriptor. */
	iconv_utf16be_utf8 = iconv_open("UTF-8", "UTF-16BE");
	if (iconv_utf16be_utf8 == (iconv_t) -1)
	{
		syslog(LOG_ERR, PROTOCOL_NAME ": Error: iconv_open failed: %s", strerror(errno));
		return false;
	}

	/* Fork off the cookier server process. */
	switch (fork())
	{
		/* An error occured. */
		case -1:
			syslog(LOG_ERR, PROTOCOL_NAME ": Error: Fork failed: %s", strerror(errno));
			return false;
		
		/* In the child. */
		case 0:
			cookiemonster();
			debugprint(localdebugmode,  PROTOCOL_NAME ": Error: We should not come here");
			exit(0);
	
		/* In the parent. */
		default:
			break;
	}		
	
	if (options["icq_trace"] == "on") tracing = true;
	if (options["icq_trace_error"] == "on") tracingerror = true;
	
	return true;
}

void closeprotocolplugin(void)
{
	iconv_close(iconv_utf16be_utf8);
	return;
}

/* The main plugin function. See protocolplugin.cpp. */
int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer, 
	int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress)
{
	struct flap myflap;
	memset(&myflap, 0, sizeof(struct flap));

	/* get the fix sized flap and ensure endianess. */	
	if (!incomingsock.recvalldata((char *) &myflap, sizeof(struct flap))) return 1;

	memcpy(replybuffer, &myflap, sizeof(struct flap));
	*replybufferlength = sizeof(struct flap);

	myflap.sequence = ntohs(myflap.sequence);
	myflap.datalength = ntohs(myflap.datalength);

	char buffer[BUFFER_SIZE];
	memset(buffer, 0, BUFFER_SIZE);

	/* get the following data assuming the flap indicates there is some. */
	if(myflap.datalength)
	{
		if (!incomingsock.recvalldata((char *) buffer, myflap.datalength)) return 1;
		
		memcpy(replybuffer + sizeof(struct flap), buffer, myflap.datalength);
		*replybufferlength += myflap.datalength;
	}	

	char *startp = replybuffer; char *p = startp + sizeof(struct flap);
	int lengthp = *replybufferlength;

	int retsnac = 0;

	if (myflap.header == 0x2a)
	{
		if (myflap.channel == 1)
			loginpacket(&p, startp, lengthp, outgoing, false, clientaddress);
		if (myflap.channel == 4)
			servercookiepacket(&p, startp, lengthp, outgoing, clientaddress);
		if (myflap.channel == 2)
		{
			if ((retsnac = snacpacket(&p, startp, lengthp, outgoing, imevents, clientaddress) == 1))
			{
				syslog(LOG_ERR, PROTOCOL_NAME ": Error: Unable to parse snac packet, icq.%d.%d", 
					getpid(), packetcount);
			}
		}
	}

	if (tracing || (tracingerror && retsnac == 1))
	{
		tracepacket("icq", packetcount, replybuffer, *replybufferlength);

		/* to convert to pcap
		 * /usr/bin/od -Ax -tx1 icq.%d.%d > icq.%d.%d.hex
		 * /usr/sbin/text2pcap -T %d,5190 icq.%d.%d.hex icq.%d.%d.cap */
	}
	
	packetcount++;
	
	return 0;
}

int loginpacket(GET_ARGS, bool outgoing, bool md5login, std::string &clientaddress)
{
	char uin[BUFFER_SIZE];
	char password[BUFFER_SIZE];
	char clientid[BUFFER_SIZE];
	char cookie[BUFFER_SIZE];
	uint16_t mytag; uint16_t mylength;
	char myvalue[BUFFER_SIZE];
	int cookielen = 0;
	
	memset(uin, 0, BUFFER_SIZE);
	memset(password, 0, BUFFER_SIZE);
	memset(clientid, 0, BUFFER_SIZE);
	memset(cookie, 0, BUFFER_SIZE);
	memset(myvalue, 0, BUFFER_SIZE);

	std::string roasted;	

	char xordata[] =
	{ 
		0xF3, 0x26, 0x81, 0xC4, 0x39, 0x86, 0xDB, 0x92,
		0x71, 0xA3, 0xB9, 0xE6, 0x53, 0x7A, 0x95, 0x7C,
		'\0'
	};	

	if (!md5login)
	{
		uint32_t dummy;
		if (!getlong(GET_CALL_ARGS, &dummy)) return 1;
	}
	
	while (gettlv(GET_CALL_ARGS, &mytag, &mylength, myvalue))
	{
		if (mytag == 1)	memcpy(uin, myvalue, mylength);
		if (mytag == 2) 
		{
			memcpy(password, myvalue, mylength);
			if (localdebugmode)
				for (int i = 0; i < mylength; i++)
					roasted.push_back((char)(password[i] xor xordata[i%16]));
		}
		if (mytag == 3) memcpy(clientid, myvalue, mylength);
		if (mytag == 6)
		{
			memcpy(cookie, myvalue, mylength);
			cookielen = mylength;
			if (tracing)
			{
				char filename[STRING_SIZE];
				memset(filename, 0, STRING_SIZE);
				snprintf(filename, STRING_SIZE - 1, TRACE_DIR "/clientcookie.%d.%d", getpid(), packetcount);
				int fd = -1;
				if ((fd = creat(filename, 0600)) > 0)
				{
					write(fd, cookie, mylength);
					close(fd);
				}
			}
		}
	}

	/* first connect: clients sends uin and pass */
	if (strlen(uin))
	{
		localid = uin;
			
		if (!roasted.empty())
		{
			debugprint(localdebugmode, PROTOCOL_NAME ": Login request, uin: %s, pass: %s", 
				uin, roasted.c_str());
		}
		else
			debugprint(localdebugmode, PROTOCOL_NAME ": Login request, uin: %s", uin);
	}

	/* second connect: client sends cookie */
	if (strlen(cookie))
	{
		std::string tmpuin;
		tmpuin = getcookieuin(cookietohex(cookielen, cookie));
		if (!tmpuin.empty())
			localid = tmpuin;
	}

	return 0;
}	

int servercookiepacket(GET_ARGS, bool outgoing, std::string &clientaddress)
{
	char uin[BUFFER_SIZE];
	char bos[BUFFER_SIZE];
	char cookie[BUFFER_SIZE];
	uint16_t mytag; uint16_t mylength;
	char myvalue[BUFFER_SIZE];
	int cookielen = 0;
	
	memset(uin, 0, BUFFER_SIZE);
	memset(bos, 0, BUFFER_SIZE);
	memset(cookie, 0, BUFFER_SIZE);
	memset(myvalue, 0, BUFFER_SIZE);
	
	while (gettlv(GET_CALL_ARGS, &mytag, &mylength, myvalue))
	{
		if (mytag == 1)	memcpy(uin, myvalue, mylength);
		if (mytag == 5) memcpy(bos, myvalue, mylength);
		if (mytag == 6)
		{
			memcpy(cookie, myvalue, mylength);
			cookielen = mylength;
			if (tracing)
			{
				char filename[STRING_SIZE];
				memset(filename, 0, STRING_SIZE);
				snprintf(filename, STRING_SIZE - 1, TRACE_DIR "/servercookie.%d.%d", getpid(), packetcount);
				int fd = -1;
				if ((fd = creat(filename, 0600)) > 0)
				{
					write(fd, cookie, mylength);
					close(fd);
				}
			}
		}
	}
	
	/* first connect: server responds with uin and cookie */
	if (strlen(uin))
	{	
		localid = uin;
		
		debugprint(localdebugmode, PROTOCOL_NAME ": Login response, uin: %s", uin);

		if (strlen(cookie))
			setcookieuin(cookietohex(cookielen, cookie), uin);		
	}
	
	return 0;
}

int snacpacket(GET_ARGS, bool outgoing, std::vector<struct imevent> &imevents, std::string &clientaddress)
{
	struct snac mysnac;
	if (!getsnac(p, startp, lengthp, &mysnac)) return 1;

	switch (mysnac.family)
	{
		/* message */
		case 4:	
			/* not all subtypes of family 4 have the following 
			 * we don't support these so return with unknown */
			if ((mysnac.subtype >= 1 && mysnac.subtype <= 5) || (mysnac.subtype >= 0x0008 && mysnac.subtype <= 0x000a))
			{	
				snacpacketunknown(mysnac);
				return 2;
			}

			uint32_t msgid1; uint32_t msgid2;
			if (!getlong(GET_CALL_ARGS, &msgid1)) return 1;
			if (!getlong(GET_CALL_ARGS, &msgid2)) return 1;

			uint16_t msgchannel;
			if (!getword(GET_CALL_ARGS, &msgchannel)) return 1;

			char mystring[BUFFER_SIZE];
			if (!getlengthbytes(GET_CALL_ARGS, mystring)) return 1;
			remoteid = mystring;
			
			switch (mysnac.subtype)
			{
				/* outgoing message */
				case 6:
					debugprint(localdebugmode, PROTOCOL_NAME ": Outgoing message, uin: %s remoteid: %s",
						localid.c_str(), remoteid.c_str());
					break;
					
				/* incoming message */
				case 7:
					uint16_t msgwarnlvl, msgtlv;
					if (!getword(GET_CALL_ARGS, &msgwarnlvl)) return 1;
					if (!getword(GET_CALL_ARGS, &msgtlv)) return 1;
					debugprint(localdebugmode, PROTOCOL_NAME ": Incoming message, uin: %s remoteid: %s",
						localid.c_str(), remoteid.c_str());
					break;

				/* typing notification */
				case 0x0014:
					uint16_t type;
					if (!getword(GET_CALL_ARGS, &type)) return 1;
					
					switch (type)
					{
						/* typing finished */
						case 0:
							debugprint(localdebugmode, PROTOCOL_NAME ": Typing finished, uin: %s remoteid: %s",
								localid.c_str(), remoteid.c_str());							
							return 0;
							
						/* text typed */
						case 1:
							debugprint(localdebugmode, PROTOCOL_NAME ": Text typed, uin: %s remoteid: %s",
								localid.c_str(), remoteid.c_str());
							return 0;
							
						/* typing begun */
						case 2:
							debugprint(localdebugmode, PROTOCOL_NAME ": Typing begun, uin: %s remoteid: %s",
								localid.c_str(), remoteid.c_str());
							logmessage(outgoing, TYPE_TYPING, "", imevents, clientaddress, 0, 0);
							return 0;

						/* not sure how to process */
						default:
							snacpacketunknown(mysnac);
							return 2;
					}
					
					/* should not come here */
					snacpacketunknown(mysnac);
					return 2;
					
				/* not sure how to process */
				default:
					snacpacketunknown(mysnac);
					return 2;
			}

			uint16_t mytag; uint16_t mylength;
			char *myvalue;
			uint16_t submytag; uint16_t submylength;
			char *submyvalue;

			/* message channel */
			switch (msgchannel)
			{
				/* plaintext format */
				case 1:
					while (gettlvptr(GET_CALL_ARGS, &mytag, &mylength, &myvalue))
					{
						if (mytag == 2)
						{
							debugprint(localdebugmode, PROTOCOL_NAME ": Plain-text message tag 2 found, len: %d",
								mylength);

							std::string message;
							int mestart, melength;
									
							if (getmessage(&myvalue, startp, lengthp, message, mestart, melength)) return 1;
							
							logmessage(outgoing, TYPE_MSG, message, imevents, clientaddress, mestart, melength);

							break;
						}
					}
					break;
					
				/* rtf format */	
				case 2:
					while (gettlvptr(GET_CALL_ARGS, &mytag, &mylength, &myvalue))
					{
						if (mytag == 5 && mylength > 4)
						{		
							debugprint(localdebugmode, PROTOCOL_NAME ": Rendezvous message data tag 5 found, len: %d", 
								mylength);
								
							uint16_t msgtype;
							if (!getword(&myvalue, startp, lengthp, &msgtype)) return 1;
						
							/* same msgid as above so we can overwrite them */
							if (!getlong(&myvalue, startp, lengthp, &msgid1)) return 1;
							if (!getlong(&myvalue, startp, lengthp, &msgid2)) return 1;

							char msgguid[16];
							if (!getbytes(&myvalue, startp, lengthp, msgguid, 16)) return 1;
						
							/* return if not a request */
							switch (msgtype)
							{
								/* request */
								case 0:
									debugprint(localdebugmode, PROTOCOL_NAME ": Request, rendezvous message, {%s}",
										cookietohex(16,msgguid).c_str());
									break;
									
								/* cancel */
								case 1:
									debugprint(localdebugmode, PROTOCOL_NAME ": Cancel, rendezvous message, {%s}",
										cookietohex(16,msgguid).c_str());
									return 2;
								
								/* accept */
								case 2:
									debugprint(localdebugmode, PROTOCOL_NAME ": Accept, rendezvous message, {%s}",
										cookietohex(16,msgguid).c_str());
									return 2;
									
								/* should not come here */
								default:
									return 1;
							}	

							char guid094613494C7F11D18222444553540000[] =
							{
								0x09, 0x46, 0x13, 0x49, 0x4C, 0x7F, 0x11, 0xD1, 
								0x82, 0x22, 0x44, 0x45, 0x53, 0x54, 0x00, 0x00
							};

							/* the capability guid determines extention data format */
							if (memcmp(msgguid, guid094613494C7F11D18222444553540000, 16) != 0)
							{	
								debugprint(localdebugmode, PROTOCOL_NAME ": Unknown rendezvous message capability");
								return 2;
							}
							
							while (gettlvptr(&myvalue, startp, lengthp, &submytag, &submylength, &submyvalue))
							{												
								if (submytag == 0x2711)
								{
									debugprint(localdebugmode, PROTOCOL_NAME ": Extention data tag 0x2711 found, len: %d", 
										submylength);

									uint16_t msgextentionlength;
									if (!getwordle(&submyvalue, startp, lengthp, &msgextentionlength)) return 1;
									
									/* check inside message extention for zeroed plugin *
									 * if plugin is 0 this is a message otherwise return */
									uint16_t msgprotocol;
									if (!getword(&submyvalue, startp, lengthp, &msgprotocol)) return 1;
									
									char msgplugin[16];
									if (!getbytes(&submyvalue, startp, lengthp, msgplugin, 16)) return 1;

									/* grab the rest of the message extention */
									char dummy[BUFFER_SIZE];
									memset(dummy, 0, BUFFER_SIZE);
									if (!getbytes(&submyvalue, startp, lengthp, dummy, msgextentionlength - 18)) return 1;	

									char pluginzeroed[16];
									memset(pluginzeroed, 0, 16);
									
									/* compare the memory for zeroed plugin */
									if (memcmp(msgplugin, pluginzeroed, 16) != 0)
									{
										debugprint(localdebugmode, PROTOCOL_NAME ": Unknown extention data plugin, {%s}",
											cookietohex(16,msgplugin).c_str());
										return 2;
									}
									
									char msgextention [BUFFER_SIZE];
									if (!getwordlelengthbytes(&submyvalue, startp, lengthp, msgextention)) return 1;

									std::string message;
									int mestart, melength;
									
									if (getrtfmessage(&submyvalue, startp, lengthp, message, mestart, melength, false)) return 1;

									logmessage(outgoing, TYPE_MSG, message, imevents, clientaddress, mestart, melength);
								
									break;
								}
							}
							break;
						}
					}		
					break;
				
				/* old-style format */
				case 4:
					while (gettlvptr(GET_CALL_ARGS, &mytag, &mylength, &myvalue))
					{
						if (mytag == 5)
						{
							debugprint(localdebugmode, PROTOCOL_NAME ": Old-style message tag 5 found, len: %d", 
								mylength);

							uint32_t dummy;
							if (!getlong(&myvalue, startp, lengthp, &dummy)) return 1;

							std::string message;
							int mestart, melength;
									
							if (getrtfmessage(&myvalue, startp, lengthp, message, mestart, melength, true)) return 1;

							logmessage(outgoing, TYPE_MSG, message, imevents, clientaddress, mestart, melength);

							break;							
						}
					}
					break;
					
				/* not sure how to process */
				default:
					snacpacketunknown(mysnac);
					return 2;
			}
			break;

		/* md5 login */
		case 0x0017:
			switch (mysnac.subtype)
			{
				/* client request */
				case 2:
					loginpacket(GET_CALL_ARGS, outgoing, true, clientaddress);
					break;
					
				/* server response */
				case 3:
					servercookiepacket(GET_CALL_ARGS, outgoing, clientaddress);
					break;
					
				/* not sure how to process */
				default:
					snacpacketunknown(mysnac);
					return 2;
			}
			break;

		/* not sure how to process */
		default:
			snacpacketunknown(mysnac);
			return 2;
	}

	return 0;
}

void snacpacketunknown(struct snac &mysnac)
{
	debugprint(localdebugmode, PROTOCOL_NAME ": uin: %s, unknown family: %04x subtype: %04x",
		localid.c_str(), mysnac.family, mysnac.subtype);	
}

int getmessage(GET_ARGS, std::string &message, int &mestart, int &melength)
{
	uint16_t mytag; uint16_t mylength;
	char *myvalue;
	
	while (gettlvptr(GET_CALL_ARGS, &mytag, &mylength, &myvalue))
	{
		if (mytag == 0x0101)
		{	
			debugprint(localdebugmode, PROTOCOL_NAME ": Message string tag 0x0101 found, len: %d", mylength);
				
			uint16_t charset;
			uint16_t charsubset;
			if (!getword(&myvalue, startp, lengthp, &charset)) return 1;					
			if (!getword(&myvalue, startp, lengthp, &charsubset)) return 1;					
			debugprint(localdebugmode, PROTOCOL_NAME ": Character set: %04x.%04x", charset, charsubset);

			mestart = myvalue - startp; melength = mylength - 4;
			
			char string[BUFFER_SIZE];
			memset(string, 0, BUFFER_SIZE);	
			if (!getbytes(&myvalue, startp, lengthp, string, mylength - 4)) return 1;

			/* Convert character encoding, if the message is Unicode */
			switch (charset)
			{
				case 2: /* UTF-16BE */
				{
					char converted_string[BUFFER_SIZE];
					memset(converted_string, 0, BUFFER_SIZE);
					
					char *inbuf = string;
					char *outbuf = converted_string;
					size_t inbytesleft = mylength - 4;
					size_t outbytesleft = BUFFER_SIZE - 1; /* Trailing \0 */
					size_t result = iconv(iconv_utf16be_utf8,
						&inbuf, &inbytesleft, &outbuf, &outbytesleft);

					if (result == (size_t) -1)
					{
						/* Ignore conversion errors. They imply incorrectly
						 * formatted input or insufficient buffer space,
						 * but the message is truncated appropriately in
						 * each case. */
						;
					}
					
					message = converted_string;
					break;
				}
				
				default:
					message = string;
					break;
			}
			
			return 0;
		}
	}
	
	debugprint(localdebugmode, PLUGIN_NAME ": Warning, message string tag 0x0101 not found");
	
	return 2;
}

int getrtfmessage(GET_ARGS, std::string &message, int &mestart, int &melength, bool oldstyle)
{
	uint8_t msgtype; uint8_t msgflags;
	if (!getbyte(GET_CALL_ARGS, &msgtype)) return 1;
	if (!getbyte(GET_CALL_ARGS, &msgflags)) return 1;
	
	if (msgtype == 1)
	{	
		debugprint(localdebugmode, PROTOCOL_NAME ": Message string type 1 found");
				
		if (oldstyle == false)
		{
			uint32_t dummy;
			if (!getlong(GET_CALL_ARGS, &dummy)) return 1;
		}

		uint16_t msglength;	
		if (!getwordle(GET_CALL_ARGS, &msglength)) return 1;

		mestart = *p - startp; melength = msglength;
			
		char string[BUFFER_SIZE];
		memset(string, 0, BUFFER_SIZE);
		if (!getbytes(GET_CALL_ARGS, string, msglength)) return 1;

		message = string;
			
		return 0;
	}
	
	debugprint(localdebugmode, PLUGIN_NAME ": Warning, unknown message string type: %d", msgtype);
	
	return 2;
}

void logmessage(bool outgoing, int type, std::string message, std::vector<struct imevent> &imevents,
	std::string clientaddress, int start, int length)
{
	struct imevent imevent;
									
	imevent.timestamp = time(NULL);
	imevent.clientaddress = clientaddress;
	imevent.protocolname = PROTOCOL_NAME;
	imevent.outgoing = outgoing;
	imevent.type = type;
	imevent.localid = localid;
	imevent.remoteid = remoteid;
	imevent.filtered = false;
	imevent.eventdata = message;
	imevent.messageextent.start = start;
	imevent.messageextent.length = length;
	
	std::transform(imevent.localid.begin(), imevent.localid.end(), imevent.localid.begin(), tolower);
	std::transform(imevent.remoteid.begin(), imevent.remoteid.end(), imevent.remoteid.begin(), tolower);
		
	imevents.push_back(imevent);
}							

bool getsnac(GET_ARGS, struct snac *rc)
{
	GET_CHECK_P(sizeof(struct snac))
	GET_TYPE(struct snac)
	
	rc->family = ntohs(rc->family);
	rc->subtype = ntohs(rc->subtype);
	rc->flags = ntohs(rc->flags);
	rc->requestid = ntohl(rc->requestid);

	return true;
}

bool getbyte(GET_ARGS, uint8_t *rc)
{
	GET_CHECK_P(sizeof(uint8_t))
	GET_TYPE(uint8_t)
	
	return true;
}

bool getbytes(GET_ARGS, char *bytes, int length)
{
	GET_CHECK_P(length)
	
	memcpy(bytes, *p, length);
	bytes[length] = '\0';
	*p += length;
	
	return true;
}

bool getword(GET_ARGS, uint16_t *rc)
{
	GET_CHECK_P(sizeof(uint16_t))
	GET_TYPE(uint16_t)
	
	*rc = ntohs(*rc);
	
	return true;
}

bool getwordle(GET_ARGS, uint16_t *rc)
{
	GET_CHECK_P(sizeof(uint16_t))
	GET_TYPE(uint16_t)
	
	return true;
}

bool getlong(GET_ARGS, uint32_t *rc)
{
	GET_CHECK_P(sizeof(uint32_t))
	GET_TYPE(uint32_t)
	
	*rc = ntohl(*rc);
	
	return true;
}

bool getlengthbytes(GET_ARGS, char *string)
{
	uint8_t length;
	
	if (!getbyte(GET_CALL_ARGS, &length)) return false;
	if (!getbytes(GET_CALL_ARGS, string, length)) return false;
	
	return true;
}

bool getwordlelengthbytes(GET_ARGS, char *string)
{
	uint16_t length;
	
	if (!getwordle(GET_CALL_ARGS, &length)) return false;
	if (!getbytes(GET_CALL_ARGS, string, length)) return false;
	
	return true;
}

bool gettlv(GET_ARGS, uint16_t *tag, uint16_t *length, char *value)
{
	if (!getword(GET_CALL_ARGS, tag)) return false;
	if (!getword(GET_CALL_ARGS, length)) return false;
	
	if (value && length)
		if (!getbytes(GET_CALL_ARGS, value, *length)) return false;
		
	return true;
}

bool gettlvptr(GET_ARGS, uint16_t *tag, uint16_t *length, char **value)
{
	if (!getword(GET_CALL_ARGS, tag)) return false;
	if (!getword(GET_CALL_ARGS, length)) return false;
	
	if (length)
	{
		*value = *p;
		*p += *length;
	}
	
	return true;
}

void cookiemonster(void)
{
	std::map<std::string, std::string> uinmap;
	class Socket cookiesock(AF_UNIX, SOCK_STREAM);
		
	if (!cookiesock.listensocket(COOKIE_SOCKET))
	{
		syslog(LOG_ERR, "Error: Couldn't bind to icq cookie socket");
	}	

	while (true)
	{
		std::string clientaddress;
		std::string doaction;
		std::string cookie;
		class Socket clientsock(AF_UNIX, SOCK_STREAM);
		char buffer[BUFFER_SIZE];
		
		if (!cookiesock.awaitconnection(clientsock, clientaddress)) continue;

		memset(buffer, 0, BUFFER_SIZE);
		if (clientsock.recvline(buffer, BUFFER_SIZE) < 0)
		{
			syslog(LOG_ERR, PROTOCOL_NAME ": Couldn't get command line from cookiemonster client");
			continue;
		}
		
		stripnewline(buffer);
		doaction = buffer;	

		memset(buffer, 0, BUFFER_SIZE);
		if (clientsock.recvline(buffer, BUFFER_SIZE) < 0)
		{
			syslog(LOG_ERR, PROTOCOL_NAME ": Couldn't get cookie line from cookiemonster client");
			continue;
		}
			
		stripnewline(buffer);
		cookie = buffer;
		
		/* recv: set\n
		 * recv: cookie\n 
		 * recv: uin\n */
		if (doaction == "set")
		{
			std::string uin;
			
			memset(buffer, 0, BUFFER_SIZE);
			if (clientsock.recvline(buffer, BUFFER_SIZE) < 0)
			{
				syslog(LOG_ERR, PROTOCOL_NAME ": Couldn't get UIN line from cookiemonster client");
				continue;
			}
			
			stripnewline(buffer);
			uin = buffer;

			uinmap[cookie] = buffer;

			debugprint(localdebugmode, PROTOCOL_NAME ": Stored cookie, uin: %s", uin.c_str());
		}
		
		/* recv: get\n
		 * recv: cookie\n
		 * send: uin\n */
		if (doaction == "get")
		{		
			std::string uin = "Unknown";

			if (!uinmap[cookie].empty())
			{
				uin = uinmap[cookie];
				debugprint(localdebugmode, PROTOCOL_NAME ": Found cookie, uin: %s", uin.c_str());
			}
			
			memset(buffer, 0, BUFFER_SIZE);
			snprintf(buffer, BUFFER_SIZE - 1, "%s\n", uin.c_str());
			
			if (!clientsock.sendalldata(buffer, strlen(buffer)))
			{
				syslog(LOG_ERR, "Couldn't send UIN back to cookiemonster client");
				continue;
			}
		}
	}
}

bool setcookieuin(std::string cookie, std::string uin)
{
	char buffer[BUFFER_SIZE];
	class Socket cookiesock(AF_UNIX, SOCK_STREAM);
					
	if (!cookiesock.connectsocket(COOKIE_SOCKET,""))
	{
		syslog(LOG_ERR, PROTOCOL_NAME ": Couldn't connect to cookie socket");
		return false;
	}

	memset(buffer, 0, BUFFER_SIZE);
	snprintf(buffer, BUFFER_SIZE - 1, "set\n%s\n%s\n", cookie.c_str(), uin.c_str());
	
	if (!cookiesock.sendalldata(buffer, strlen(buffer)))
	{
		syslog(LOG_ERR, PROTOCOL_NAME ": Couldn't send cookie set request");
		cookiesock.closesocket();
		return false;
	}
	
	cookiesock.closesocket();
	
	return true;
}

std::string getcookieuin(std::string cookie)
{
	char buffer[BUFFER_SIZE];
	class Socket cookiesock(AF_UNIX, SOCK_STREAM);

	memset(buffer, 0, BUFFER_SIZE);

	if (!cookiesock.connectsocket(COOKIE_SOCKET,""))
	{
		syslog(LOG_ERR, PROTOCOL_NAME ": Couldn't connect to cookie socket");
		return "";	
	}

	memset(buffer, 0, BUFFER_SIZE);
	snprintf(buffer, BUFFER_SIZE - 1, "get\n%s\n", cookie.c_str());

	if (!cookiesock.sendalldata(buffer, strlen(buffer)))
	{
		syslog(LOG_ERR, PROTOCOL_NAME ": Couldn't send cookie get request");
		cookiesock.closesocket();
		return "";
	}

	memset(buffer, 0, BUFFER_SIZE);
	if (!cookiesock.recvline(buffer, BUFFER_SIZE))
	{
		syslog(LOG_ERR, PROTOCOL_NAME ": Didn't get a response from cookiemonster");
		cookiesock.closesocket();
		return "";
	}
	
	stripnewline(buffer);
	
	std::string uin;
	if (strlen(buffer)) uin = buffer;
	
	cookiesock.closesocket();

	return uin;
}

std::string cookietohex (int length, char *cookie)
{
	char hexchar[STRING_SIZE];
	std::string hexcookie;
	
	for (int i = 0; i < length; i++)
	{
		sprintf(hexchar, "%02X", cookie[i]);
		hexcookie.push_back(hexchar[strlen(hexchar)-2]);
		hexcookie.push_back(hexchar[strlen(hexchar)-1]);
	}

	return hexcookie;
}
