/* itip_support.h -- Routines for dealing with iTIP
 *
 * Copyright (c) 1994-2021 Carnegie Mellon University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The name "Carnegie Mellon University" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For permission or any legal
 *    details, please contact
 *      Carnegie Mellon University
 *      Center for Technology Transfer and Enterprise Creation
 *      4615 Forbes Avenue
 *      Suite 302
 *      Pittsburgh, PA  15213
 *      (412) 268-7393, fax: (412) 268-7395
 *      innovation@andrew.cmu.edu
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Computing Services
 *     at Carnegie Mellon University (http://www.cmu.edu/computing/)."
 *
 * CARNEGIE MELLON UNIVERSITY DISCLAIMS ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL CARNEGIE MELLON UNIVERSITY BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
 * OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef ITIP_SUPPORT_H
#define ITIP_SUPPORT_H

#include <config.h>

#include <libical/ical.h>

#include "acl.h"
#include "parseaddr.h"
#include "strarray.h"


#define SCHED_INBOX     "Inbox/"
#define SCHED_OUTBOX    "Outbox/"
#define SCHED_DEFAULT   "Default/"

#define REQSTAT_PENDING         "1.0;Pending"
#define REQSTAT_SENT            "1.1;Sent"
#define REQSTAT_DELIVERED       "1.2;Delivered"
#define REQSTAT_SUCCESS         "2.0;Success"
#define REQSTAT_NOUSER          "3.7;Invalid calendar user"
#define REQSTAT_NOPRIVS         "3.8;Noauthority"
#define REQSTAT_TEMPFAIL        "5.1;Service unavailable"
#define REQSTAT_PERMFAIL        "5.2;Invalid calendar service"
#define REQSTAT_REJECTED        "5.3;No scheduling support for user"

struct sched_data {
    unsigned flags;
    icalcomponent *itip;
    icalcomponent *oldical;
    icalcomponent *newical;
    icalparameter_scheduleforcesend force_send;
    const strarray_t *schedule_addresses;
    const char *calendarid;
    const char *status;
};

#define SCHEDFLAG_ISCHEDULE        (1<<0)
#define SCHEDFLAG_IS_REPLY         (1<<1)
#define SCHEDFLAG_IS_UPDATE        (1<<2)
#define SCHEDFLAG_INVITES_ONLY     (1<<3)
#define SCHEDFLAG_UPDATES_ONLY     (1<<4)
#define SCHEDFLAG_DELETE_CANCELED  (1<<5)

#define SCHED_ISCHEDULE(sched)        (sched->flags & SCHEDFLAG_ISCHEDULE)
#define SCHED_IS_REPLY(sched)         (sched->flags & SCHEDFLAG_IS_REPLY)
#define SCHED_IS_UPDATE(sched)        (sched->flags & SCHEDFLAG_IS_UPDATE)
#define SCHED_INVITES_ONLY(sched)     (sched->flags & SCHEDFLAG_INVITES_ONLY)
#define SCHED_UPDATES_ONLY(sched)     (sched->flags & SCHEDFLAG_UPDATES_ONLY)
#define SCHED_DELETE_CANCELED(sched)  (sched->flags & SCHEDFLAG_DELETE_CANCELED)

#define SCHED_STATUS(sched, isched, ical) \
    (sched->status = SCHED_ISCHEDULE(sched) ? isched : ical)

#define SCHEDSTAT_PENDING       "1.0"
#define SCHEDSTAT_SENT          "1.1"
#define SCHEDSTAT_DELIVERED     "1.2"
#define SCHEDSTAT_SUCCESS       "2.0"
#define SCHEDSTAT_PARAM         "2.3"
#define SCHEDSTAT_NOUSER        "3.7"
#define SCHEDSTAT_NOPRIVS       "3.8"
#define SCHEDSTAT_TEMPFAIL      "5.1"
#define SCHEDSTAT_PERMFAIL      "5.2"
#define SCHEDSTAT_REJECTED      "5.3"

/* Scheduling protocol flags */
#define SCHEDTYPE_REMOTE        (1<<0)
#define SCHEDTYPE_ISCHEDULE     (1<<1)
#define SCHEDTYPE_SSL           (1<<2)

enum sched_deliver_outcome {
    SCHED_DELIVER_ERROR    = -1,
    SCHED_DELIVER_NOACTION = 0,
    SCHED_DELIVER_ADDED    = 1,
    SCHED_DELIVER_UPDATED  = 2,
    SCHED_DELIVER_DELETED  = 3
};

extern unsigned config_allowsched;
extern struct strlist *cua_domains;
extern icaltimezone *utc_zone;

struct proplist {
    icalproperty *prop;
    struct proplist *next;
};

/* Each calendar user address has the following scheduling protocol params */
/* All memory must be freed with sched_param_fini. */
struct caldav_sched_param {
    char *userid;       /* Userid corresponding to calendar address */
    char *server;       /* Remote server user lives on */
    unsigned port;      /* Remote server port, default = 80 */
    unsigned flags;     /* Flags dictating protocol to use for scheduling */
    unsigned isyou;     /* true if the user is the same as the authenticated user */
    struct proplist *props; /* List of attendee iCal properties */
};

extern void sched_param_fini(struct caldav_sched_param *sparam);

/* Returns the calendar collection name to use as scheduling default.
 * This is just the *last* part of the complete path without trailing
 * path separator, e.g. 'Default'. If no default calendar is set,
 * then optionally return a fallback calendar where to store scheduling
 * messages.
 * Returns NULL, if no calendar is set and none could be guessed. */
extern char *caldav_scheddefault(const char *userid, int fallback);

extern icalproperty *find_attendee(icalcomponent *comp, const char *match);
extern const char *get_organizer(icalcomponent *comp);
extern int partstat_changed(icalcomponent *oldcomp,
                            icalcomponent *newcomp, const char *attendee);
extern int organizer_changed(icalcomponent *oldcomp, icalcomponent *newcomp);

extern icalcomponent *master_to_recurrence(icalcomponent *master,
                                           icalproperty *recurid);

extern void itip_strip_personal_data(icalcomponent *comp);

extern enum sched_deliver_outcome sched_deliver_local(const char *userid,
                                                      const char *sender,
                                                      const char *recipient,
                                                      struct address *mailfrom,
                                                      struct caldav_sched_param *sparam,
                                                      struct sched_data *sched_data,
                                                      struct auth_state *authstate,
                                                      const char **attendeep,
                                                      icalcomponent **icalp);

#endif /* ITIP_SUPPORT_H */
