/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


/**
   \file cdw_button.c

   Implementation of "button" widget that can be embedded e.g.
   in dialog windows.
*/

#include <string.h>
#include <stdlib.h>

#include "main.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_config.h"
#include "cdw_debug.h"
#include "cdw_string.h"
#include "cdw_button.h"
#include "cdw_debug.h"







/**
   \brief Create new CDW_BUTTON widget

   \date Function's top-level comment reviewed on 2016-02-16
   \date Function's body reviewed on 2016-02-16

   Remember to assign value to CDW_BUTTON->on_click_callback

   \param parent - parent window, in which the button will be displayed
   \param begin_y - number of row of parent window, in which the button will be placed
   \param begin_x - number of column of parent window, in which the button will be placed
   \param label - string that will be displayed as button label
   \param colors - color scheme of button

   \return pointer to new button widget on success
   \return NULL pointer if creating widget fails
*/
CDW_BUTTON *cdw_button_new(WINDOW *parent, int begin_y, int begin_x, const char *label, cdw_colors_t colors)
{
	cdw_assert (parent, "ERROR: parent window is null\n");
	/* Don't let this widget begin outside of its parent
	   window. Button is one-line widget, so we don't have to
	   check its lower border - conditions are the same as for
	   upper border.. */
	cdw_assert (begin_x < getmaxx(parent), "ERROR: begin_x (%d) >= getmaxx(parent) (%d)\n", begin_x, getmaxx(parent));
	cdw_assert (begin_y < getmaxy(parent), "ERROR: begin_y (%d) >= getmaxy(parent) (%d)\n", begin_y, getmaxy(parent));

	CDW_BUTTON *button = (CDW_BUTTON *) malloc(sizeof(CDW_BUTTON));
	if (!button) {
		cdw_vdm ("ERROR: malloc()\n");
		return (CDW_BUTTON *) NULL;
	}

	cdw_widget_init(&button->widget);
	cdw_widget_add_return_keys(&button->widget, CDW_KEY_ESCAPE, CDW_KEY_TAB, CDW_KEY_BTAB, KEY_UP, KEY_DOWN, 0);
	button->widget.add_return_key = cdw_widget_add_return_key; /* Default function for adding return keys by parent cdw_form. */
	button->widget.type_id = CDW_WIDGET_ID_BUTTON;

	button->strip = (WINDOW *) NULL;
	button->label = (char *) NULL;
	button->on_click_key = 0;

	button->label = strdup(label);
	if (!(button->label)) {
		free(button);
		button = (CDW_BUTTON *) NULL;

		cdw_vdm ("ERROR: strdup()\n");
		return (CDW_BUTTON *) NULL;
	}

	button->begin_x = begin_x;
	button->begin_y = begin_y;
	button->colors = colors;
	button->on_click_callback = (cdw_form_widget_function_t) NULL;
	button->parent = parent;

	int label_len = (int) strlen(button->label);
	/* "[ label ]" = 2 + len + 2 */
	button->strip = derwin(parent, 1, 2 + label_len + 2,
			       button->begin_y, button->begin_x);
	if (!button->strip) {
		cdw_vdm ("ERROR: can't create new button: derwin()\n");
		cdw_vdm ("ERROR: nlines = %d, ncols = %d, begin_y = %d, begin_x = %d, getmaxx(parent) = %d, getmaxy(parent) = %d\n",
			 1, 2 + label_len + 2, button->begin_y, button->begin_x, getmaxx(parent), getmaxy(parent));

		cdw_button_delete(&button);
		return (CDW_BUTTON *) NULL;
	} else {
		(void) wattrset(button->strip, COLOR_PAIR(button->colors));

		mvwprintw(button->strip, 0, 0, "[ %s ]", button->label);
		wrefresh(button->strip);

		return button;
	}
}





/**
   \brief Destroy CDW_BUTTON widget

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Deallocate all resources used by CDW_BUTTON widget. The function
   accepts NULL pointer. \p button is set to NULL on return.

   \param button - button to destroy
*/
void cdw_button_delete(CDW_BUTTON **button)
{
	cdw_assert (button, "ERROR: passed to the function a NULL pointer to the widget\n");

	if (!*button) {
		cdw_vdm ("WARNING: NULL button\n");
		return;
	}

	cdw_button_free(*button);
	*button = (CDW_BUTTON *) NULL;

	return;
}





/**
   \brief Free CDW_BUTTON widget

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Deallocate all resources used by CDW_BUTTON widget. The function
   accepts NULL pointer, but does not set its argument to NULL.

   \param button - button to free
*/
void cdw_button_free(CDW_BUTTON *button)
{
	if (!button) {
		cdw_vdm ("WARNING: NULL button\n");
		return;
	}

	if (button->strip) {
		delwin(button->strip);
		button->strip = (WINDOW *) NULL;
	}

	if (button->label) {
		free(button->label);
		button->label = (char *) NULL;
	}

	free(button);

	return;
}





/**
   \brief Highlight given button

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Redraw given button using its reversed 'colors' scheme. It will look as
   if the button has changed its state to 'has focus'.

   \param button - button to highlight
*/
void cdw_button_focus(CDW_BUTTON *button)
{
	cdw_assert (button, "ERROR: cannot focus NULL button\n");

	/* wattrset(button->strip, COLOR_PAIR(button->focus_color)); */
	(void) wattrset(button->strip, COLOR_PAIR(button->colors) | A_REVERSE);

	mvwprintw(button->strip, 0, 0, "[ %s ]", button->label);
	wrefresh(button->strip);

	return;
}





/**
   \brief Display given button as it is in its normal state

   \date Function's top-level comment reviewed on 2012-01-09
   \date Function's body reviewed on 2012-01-09

   Redraw given button using its normal color scheme. It will look
   as if the button has changed its state to 'has no focus'.

   \param button - button to unfocus
*/
void cdw_button_unfocus(CDW_BUTTON *button)
{
	cdw_assert (button, "ERROR: cannot unfocus NULL button\n");

	(void) wattrset(button->strip, COLOR_PAIR(button->colors));

	mvwprintw(button->strip, 0, 0, "[ %s ]", button->label);
	wrefresh(button->strip);

	return;
}





int cdw_button_driver(CDW_BUTTON *b, void *form)
{
	cdw_assert (b, "ERROR: NULL pointer to button\n");
	cdw_assert (b->widget.type_id == CDW_WIDGET_ID_BUTTON, "ERROR: this is not a button widget\n");
	cdw_assert (b->parent, "ERROR: can't control button without parent window\n");

	int key = 'a'; /* Safe initial value. */
	while (!cdw_widget_is_return_key(&b->widget, key)) {
		key = wgetch(b->parent);

		if (key == CDW_KEY_ENTER) {
			/* On enter we can either return control, or execute a
			   callback defined by client code. */
			if (b->on_click_callback) {
				cdw_vdm ("INFO: Calling \"on click callback\"\n");
				b->on_click_callback((cdw_form_t *) form, NULL);
			} else {
				cdw_vdm ("INFO: Returning key %s\n", cdw_ncurses_key_label(b->on_click_key));
				return b->on_click_key;
			}
		} else {
			/* Either widget's return key that will be
			   caught by loop condition and handled by
			   widget's parent, or a completely
			   meaningless key that should be ignored. */
			;
		}
	}

	if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {
		key = CDW_KEY_ESCAPE;
	}

	return key;
}
