/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/

#ifndef _PROJOUTFLOWBC_H_
#define _PROJOUTFLOWBC_H_

//
// $Id: ProjOutFlowBC.H,v 1.9 2003/02/21 22:49:10 car Exp $
//

#include "FArrayBox.H"
#include "Geometry.H"
#include "Orientation.H"
#include "OutFlowBC.H"

//
// This class solves the BL_SPACEDIM-1 dimensional problem for the holy-grail
// projection at the outflow face.  It first extrapolates the interior values
// to the outflow edge.  Then, it solves using either back substitution or
// a multigrid solver.  If a multigrid solver is used, the problem is shifted
// so that the last dimension is 1.
//

class ProjOutFlowBC 
    :
    public OutFlowBC
{
public:

    ProjOutFlowBC ();

    void computeBC (FArrayBox        velFab[][2*BL_SPACEDIM],
                    FArrayBox        divuFab[2*BL_SPACEDIM],
                    FArrayBox        rhoFab[2*BL_SPACEDIM],
                    FArrayBox        phiFab[2*BL_SPACEDIM],
                    const Geometry&   geom,
                    Orientation*      outFaces,
                    int               numOutFlowFaces,
                    Real              gravity = 0.0);

    void computeRhoG (FArrayBox*          rhoFab,
                      FArrayBox*          phiFab,
                      const Geometry&    geom,
                      Orientation*       outFaces,
                      int                numOutFlowFaces,
                      Real               gravity);

protected:

#if (BL_SPACEDIM == 3)
    void computeCoefficients (FArrayBox&   rhs,
                              FArrayBox&   beta,
                              FArrayBox&   ccExt,
                              Box&         faceBox,
                              Real*        dxFiltered,
                              int*         isPeriodic);
    //
    // Static data.
    //
    static Real tol;
    static Real abs_tol;
#endif
};

#if (BL_SPACEDIM == 3)
//
// This class is a BL_SPACEDIM-1 multigrid solver for the holy grail 
// projection.
// 

class ProjOutFlowBC_MG 
    :
    public OutFlowBC_MG
{
public:

    ProjOutFlowBC_MG (const Box& Domain,
                      FArrayBox* Phi,
                      FArrayBox* Rhs,
                      FArrayBox* Resid,
                      FArrayBox* Beta,
                      Real*      H,
                      int*       isPeriodic);

    ~ProjOutFlowBC_MG ();

    static int Verbose () { return verbose; }

    static int MaxIters () { return maxIters; }

protected:

    void Restrict ();
    void interpolate ();
    Real residual ();
    void step (int nSteps);
    void gsrb (int nSteps);
    //
    // Static data.
    //
    static int  verbose;
    static bool useCGbottomSolver;
    static Real cg_tol;
    static Real cg_abs_tol;
    static Real cg_max_jump;
    static int  cg_maxiter;
    static int  maxIters;
};
#endif

#endif


