//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/IO/IOFactory.h
//! @brief     Declares data import/export functions.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_DEVICE_IO_IOFACTORY_H
#define BORNAGAIN_DEVICE_IO_IOFACTORY_H

#include <string>

class Datafield;
struct ImportSettings1D;

namespace IO {

enum Filetype1D {
    unknown1D,
    csv1D,
    csv1D_2cols,
    csv1D_3cols,
    csv1D_4cols,
    csv1D_5cols,
    bornagain1D,
    mft
};
enum Filetype2D { unknown2D, csv2D, bornagain2D, tiff, nicos2D, refsans2D };

#ifndef SWIG
bool isCompressed(const std::string& name);
Filetype1D filename2type1D(const std::string& filename);
Filetype2D filename2type2D(const std::string& filename);
#endif

//! Reads 1D data file and returns newly created Datafield object.
//! If selector is 'unknown1D', then the file extension will determine the loader to be used.
//! For any other selector, the file extension is ignored.
//! If file name ends with "*.gz" or "*.bz2" the file will be unzipped on the fly.
//! May throw, but will never return nullptr.
Datafield* readData1D(const std::string& file_name, Filetype1D ftype = unknown1D,
                      const ImportSettings1D* importSettings = nullptr);

//! Reads 2D data file and returns newly created Datafield object.
//! If selector is 'unknown2D', then the file extension will determine the loader to be used.
//! For any other selector, the file extension is ignored.
//! If file name ends with "*.gz" or "*.bz2" the file will be unzipped on the fly.
//! May throw, but will never return nullptr.
Datafield* readData2D(const std::string& file_name, Filetype2D ftype = unknown2D);

//! Writes Datafield to file.
//! The format is determined by the file extension.
//! If file name ends with "*.gz" or "*.bz2" the file will be zipped on the fly.
void writeDatafield(const Datafield& data, const std::string& file_name);

} // namespace IO

namespace Util::RW {

//! Returns true if data matches reference file
bool dataMatchesFile(const Datafield& data, const std::string& refFileName, double tol);

} // namespace Util::RW

#endif // BORNAGAIN_DEVICE_IO_IOFACTORY_H
