// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/List.h"
#include <assert.h>

namespace Puma {


List::List (const List &list) {
  _first = _last = (ListElement*)0;
  *this = list;
}


List::~List() {
  clear();
}     


List &List::operator = (const List &list) {
  clear ();
  return *this += list;
}

   
List &List::operator += (const List &list) {
  ListElement *curr;

  curr = list._first;
  while (curr) {
    append (*curr->duplicate ());
    curr = curr->_next;
  }
  return *this;
}

   
List List::operator + (const List &list) {
  List result;

  result += *this;
  result += list;
  return result;
} 


void List::clear () {
  ListElement *curr;
  ListElement *next;

  curr = _first;
  while (curr) {
    next = curr->_next;
    curr->_prev = 0;
    curr->_next = 0;
    assert (curr->_belonging_to == this);
    curr->_belonging_to = 0;
    delete curr;
    curr = next;
  }
  _first = _last = (ListElement*)0;
}


List *List::copy (ListElement *from, ListElement *to) {
  ListElement *curr = from ? from : _first;
  ListElement *the_end = to ? to->_next : 0;
  List *list = new List;

  while (curr && curr != the_end) {
    list->append (*curr->duplicate ());
    curr = curr->_next;
  }
  return list;
}
   

void List::cut (List &out, ListElement *from, ListElement *to) {
  ListElement *next, *curr = from ? from : _first;
  ListElement *the_end = to ? to->_next : from ? from->_next : 0;

  while (curr && curr != the_end) {
    next = curr->_next;
    remove (curr);
    out.append (*curr);
    curr = next;
  }
}

   
void List::paste (ListElement *at, const List &l) {
  ListElement *curr = l._first;
  ListElement *inspos = at ? at : _last;

  while (curr) {
    if (! inspos) {
      append (*(curr->duplicate ()));
    } else {
      insert (inspos, *(curr->duplicate ()));
      inspos = inspos->_next;
    }
    curr = curr->_next;
  }
}


void List::paste_before (ListElement *at, const List &l) {
  if (at && at->_prev) {
    paste (at->_prev, l);
  } 
  else {
    ListElement *curr = l._last;

    while (curr) {
      prepend (*(curr->duplicate ()));
      curr = curr->_prev;
    }
  }
}

   
void List::move (ListElement *at, List &l) {
  ListElement *curr = l._first;
  ListElement *inspos = at ? at : _last;

  while (curr) {
    l.remove (curr);
    if (! inspos) {
      append (*curr);
    } else {
      insert (inspos, *curr);
      inspos = inspos->_next;
    }
    curr = l._first;
  }
}


void List::move_before (ListElement *at, List &l) {
  if (at && at->_prev) {
    move (at->_prev,l);
  } 
  else {
    ListElement *curr = l._last;

    while (curr) {
      l.remove (curr);
      prepend (*curr);
      curr = l._last;
    }
  }
}
   
   
void List::append (ListElement &element) {
  assert (! element._belonging_to);
  element._next = (ListElement*)0;
  element._prev = _last;
  element._belonging_to = this;
  if (empty ())
    _first = &element;
  else
    _last->_next = &element;
  _last = &element;
}


void List::prepend (ListElement &element) {
  assert (! element._belonging_to);
  element._prev = (ListElement*)0;
  element._next = _first;
  element._belonging_to = this;
  if (empty ())
    _last = &element;
  else
    _first->_prev = &element;
  _first = &element;
}


void List::insert (ListElement *at, ListElement &element) {
  assert (! element._belonging_to);
  element._prev = at;
  element._next = at->_next;
  element._belonging_to = this;
  if (at && at->_next)
    at->_next->_prev = &element;
  else
    _last = &element;
  if (at)
    at->_next = &element;
}


void List::remove (ListElement *element) {
  if (! element)
    return;
    
  assert (element->_belonging_to == this);
  element->_belonging_to = (List*) 0;
   
  if (! element->_next)
    _last = element->_prev;
  else
    element->_next->_prev = element->_prev;

  if (! element->_prev)
    _first = element->_next;
  else
    element->_prev->_next = element->_next;

  element->_prev = element->_next = (ListElement*)0;
}


} // namespace Puma
