# --
# Copyright (C) 2001-2021 OTRS AG, https://otrs.com/
# Copyright (C) 2021 Znuny GmbH, https://znuny.org/
# --
# This software comes with ABSOLUTELY NO WARRANTY. For details, see
# the enclosed file COPYING for license information (GPL). If you
# did not receive this file, see https://www.gnu.org/licenses/gpl-3.0.txt.
# --
## nofilter(TidyAll::Plugin::OTRS::Perl::Pod::NamePod)

package scripts::Migration::Base::InvalidSettingsCheck;    ## no critic

use strict;
use warnings;

use IO::Interactive qw(is_interactive);

use parent qw(scripts::Migration::Base);

our @ObjectDependencies = (
    'Kernel::System::Console::Command::Admin::Config::FixInvalid',
    'Kernel::System::SysConfig',
);

=head1 SYNOPSIS

Checks for invalid configuration settings.

=cut

sub Run {
    my ( $Self, %Param ) = @_;

    my $Verbose = $Param{CommandlineOptions}->{Verbose} || 0;

    my $SysConfigObject = $Kernel::OM->Get('Kernel::System::SysConfig');

    my @InvalidSettings = $SysConfigObject->ConfigurationInvalidList(
        Undeployed => 1,    # Check undeployed settings as well.
        NoCache    => 1,
    );

    return 1 if !scalar @InvalidSettings;

    my $CommandObject = $Kernel::OM->Get('Kernel::System::Console::Command::Admin::Config::FixInvalid');

    # Run in no-ANSI mode for consistent output. Skip missing settings (they might get fixed later).
    my @CommandArgs = ( '--no-ansi', '--skip-missing' );

    # Capture the output if we are in non-interactive mode.
    if ( $Param{CommandlineOptions}->{NonInteractive} || !is_interactive() ) {

        # Try to fix invalid settings automatically.
        push @CommandArgs, '--non-interactive';

        my $CommandOutput;

        {
            local *STDOUT;
            open STDOUT, '>:utf8', \$CommandOutput;    ## no critic
            $CommandObject->Execute(@CommandArgs);
        }

        # Show command output in following cases:
        #   - verbose mode is active
        #   - warnings were raised for settings
        #   - settings were auto-corrected
        #
        #   Any errors printed to STDERR will show up regardless.
        if (
            $CommandOutput
            && (
                $Verbose
                || $CommandOutput =~ m{Warning:}
                || $CommandOutput =~ m{Auto-corrected setting:}
            )
            )
        {
            print $CommandOutput;
        }
    }

    # Otherwise, just execute the command and show complete output as-is.
    else {
        $CommandObject->Execute(@CommandArgs);
    }

    # Always return success, as warnings generated by this module are not considered fatal.
    return 1;
}

1;
