/**
 * @brief Class Config - controls loading and saving of configuartion options.
 *
 * This class contains the interface to access the global configuration
 * file. This file uses the data format of freedesktop.org as described
 * in folowing file: http://freedesktop.org/Standards/desktop-entry-spec.
 *
 * The file is loaded into memory at pbbuttonsd startup and any module
 * can then ask for configuration options and change them if needed. At
 * the end or on special request the configuration file will be written
 * back to hard disk.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/class_config.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <pbb.h>
#include <glib.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "class_config.h"

/**
 * @brief private data of class config
 */
struct class_config {
	GKeyFile *config;
	GString *configfile;
} modbase_config;

/**
 * @brief Constructor of class config
 *
 * This function initializes a new config file and loads its
 * contents into memory. If the configfile does not exists,
 * it creates a new one. The configfile will be held in memory
 * only until it is written to hard disk with the function
 * config_save().
 *
 * When the config file is not longer in use, free its ressources
 * with config_destroy().
 *
 * @see config_save(), config_destroy()
 *
 * @param configfile  a full pathname of the configuration file
 * @return TRUE, if the configuration file was loaded successfully
 *         or FALSE if something went wrong. If a new configuration
 *         file should be created, the result is always FALSE.
 */
gboolean
config_new (char *configfile)
{
	struct class_config *base = &modbase_config;
	GError *error = NULL;
	gboolean success;

	base->configfile = g_string_new (configfile);

	base->config = g_key_file_new ();
	g_key_file_set_list_separator (base->config, ',');
	success = g_key_file_load_from_file (base->config,
			        configfile,
                    G_KEY_FILE_KEEP_COMMENTS,
					&error);

	if (success) return TRUE;
	
	g_key_file_set_comment (base->config, NULL, NULL,
			_(" Configuration file for pbbuttonsd >= version 0.8.0\n"
			  " For complete list of options please see pbbuttonsd.cnf man-page.\n"
			  " For description of the file format please see\n"
			  "    http://freedesktop.org/Standards/desktop-entry-spec.\n"),
			NULL);

	print_msg (PBB_WARN, _("Can't load config file: %s, using defaults.\n"), error->message);
	g_error_free (error);
	return FALSE;
}

/**
 * @brief Frees all allocated ressources and destroys the g_key_file object
 *
 * This function frees all ressources that are used by this module.
 * Finally the g_key_file object is destroyed. All changed to the
 * configuration file, that are not saved until calling this function,
 * are lost.
 */
void
config_destroy ()
{
	struct class_config *base = &modbase_config;

	if (base->configfile) g_string_free (base->configfile, TRUE);
	if (base->config)     g_key_file_free (base->config);
}

/**
 * @brief Save the configuration file to disk
 *
 * This function writes the configuration array to disk. During program
 * execution any module was able to read, change and create configuration
 * options. All this options will be collected in the g_key_file object
 * in memory until the user request it to write it back to disk. 
 *
 * Each time the configuration file should be written, this function
 * checks if it is athorized to write the cconfiguration file to disk.
 * Various security checks are done to protect the configuration file
 * already on disk against corruption. See also config_iswritable().
 *
 * @see config_iswritable()
 *
 * @return  TRUE, if the configuration file was sucessfully written or
 *          FALSE, if something went wrong.
 */
gboolean
config_save ()
{
	struct class_config *base = &modbase_config;
	gchar *stream;
	gsize length;
	FILE *fd;
	gboolean rc = FALSE;

	if (config_iswritable()) {
		stream = g_key_file_to_data (base->config, &length, NULL);
		if ((fd = fopen(base->configfile->str, "w"))) {
			fwrite (stream, length, 1, fd);
			fclose (fd);
			print_msg (PBB_INFO, _("Configuration saved to %s.\n"), base->configfile->str);
			rc = TRUE;
		} else
			print_msg (PBB_ERR, _("Can't write config file %s\n"), base->configfile->str);

		g_free (stream);
	}
	return rc;
}

/**
 * @brief Check if the configuration file can be written
 *
 * This function checks, if the configuration file can be written.
 * Multiple security checks are done on the file to be safe that
 * no harm could happen to the configuration.
 *
 * The configuration file can be written, if
 * @li the process owner of pbbuttonsd is also owner of the
 *     configuration file
 * @li only the owner has write permissions to the configuration
 *     file
 * @li the file is not write protected
 *
 * If the configuration file is not writable, the reason for that
 * will be written into the log file.
 *
 * @return TRUE, if the file could be written and FALSE if not.
 */
gboolean
config_iswritable ()
{
	struct class_config *base = &modbase_config;
	unsigned int perms, owner;

	owner = get_owner(base->configfile->str);
	perms = get_permissions(base->configfile->str);
	if (owner != getuid() || (perms & 022))
		print_msg (PBB_WARN, _("Configuration file [%s] is insecure, saving denied.\n"), base->configfile->str);
	else if (owner == getuid() && !(perms & 0200))
		print_msg (PBB_WARN, _("Configuration file [%s] is not writable, saving not possible.\n"), base->configfile->str);
	else
		return TRUE; 

	return FALSE;
}

/**
 * @brief Checks if a section/key is present in the configuration file
 *
 * This function checks if a specific option inside a specific section
 * is present or not. The return value is a boolean.
 * 
 * @return TRUE if the option is present, FALSE if not.
 */
gboolean
config_has_key (const char *group, const char *key)
{
	struct class_config *base = &modbase_config;
	GError *error = NULL;
	gboolean rc;

	rc = g_key_file_has_key (base->config, group, key, &error);
	if (error) {
		g_error_free (error);
		return FALSE;
	}
	return rc;
}

/**
 * @brief Read an integer from the configuration file
 *
 * This function checks if the given 'key' exists in section 'group'
 * and returns the value of the key as integer value. If the key
 * does not exist the default_value will be returned.
 *
 * @see config_set_int()
 *
 * @param  *group  section name to look at
 * @param  *key    key name
 * @param  default_value  return value, if the key does not exist
 * @return the key value as integer or the default_value
 */
int
config_get_int (const char *group, const char *key, int default_value)
{
	struct class_config *base = &modbase_config;
	GError *error = NULL;
	int value;

	value = g_key_file_get_integer (base->config, group, key, &error);
	if (error) {
		g_error_free (error);
		return default_value;
	}
	return value;
}

/**
 * @brief Write an integer to the configuration file
 *
 * This function writes an integer under the key 'key' into the
 * section 'group'. If the key or the group does not exist it will
 * be created
 *
 * @see config_get_int()
 *
 * @param  *group  section name to place key in
 * @param  *key    key name
 * @param  value   new value for 'key'
 */
void
config_set_int (const char *group, const char *key, int value)
{
	struct class_config *base = &modbase_config;

	g_key_file_set_integer (base->config, group, key, value);
}

/**
 * @brief read a configuration key and return its value as percentage
 *
 * This function reads the value of a given key from the configuration
 * file and treat it as percentage value. Only the read value will be
 * checked for percentage bounds and clipped as necessary. The default
 * value will be returned as is. This allows to pass invalid values to
 * check for errors.
 *
 * If the key doesn't exists in the configuration file the given default
 * value will be returned.
 *
 * @see config_set_percent()
 *
 * @param  group          Configuration section where should be looked
 *                        for 'key'
 * @param  key            The key which value should be read
 * @param  default_value  The value that should be returned if 'key' can't
 *                        be read for any reason.
 * @return The value for 'key' or the default_value on error, clipped
 *         to percentage bounds if necessary.
 */
int
config_get_percent (const char *group, const char *key, int default_value)
{
	struct class_config *base = &modbase_config;
	GError *error = NULL;
	int value;

	value = g_key_file_get_integer (base->config, group, key, &error);
	if (error) {
		g_error_free (error);
		return default_value;
	}
	return value > 100 ? 100 : value < 0 ? 0 : value;
}

/**
 * @brief write a percentage value to the configuration file
 *
 * This function writes a new value to a given key in the configuration
 * file. The given value will be checked for percentage bounds and clipped
 * as necessary.
 *
 * If the key doesn't exists it will be created.
 *
 * @see config_get_percent()
 *
 * @param  group  Configuration section where the key/value pair should
 *                be changed/created.
 * @param  key    The key which value should be changed
 * @param  value  The new value for 'key'
 */
void
config_set_percent (const char *group, const char *key, int value)
{
	struct class_config *base = &modbase_config;

	value = value > 100 ? 100 : value < 0 ? 0 : value;
	g_key_file_set_integer (base->config, group, key, value);
}

/**
 * @brief Read a string value from the configuration file
 *
 * This function reads a string value of a given key from the
 * configuration file. If the key doesn't exists in the configuration
 * file the given default value will be returned.
 *
 * The string from the configuration file or the default value
 * will be copied to a newly allocated memory. This memory should
 * be freed with g_free() after usage.
 *
 * @see config_set_string()
 *
 * @param  group          Configuration section where should be looked
 *                        for 'key'
 * @param  key            The key which value should be read
 * @param  default_value  The value that should be returned if 'key' can't
 *                        be read for any reason.
 * @return Pointer to a allocated string.
 */
char *
config_get_string (const char *group, const char *key, const char *default_value)
{
	struct class_config *base = &modbase_config;
	GString *arg2;
	gchar *arg1;

	arg1 = g_key_file_get_string (base->config, group, key, NULL);

	if (arg1 == NULL) {
		arg2 = g_string_new (default_value);
		arg1 = arg2->str;
		g_string_free (arg2, FALSE);
	}
	return arg1;
}

/**
 * @brief Write a string value to the configuration file
 *
 * This function writes a string value of a given key to the
 * configuration file. If the key doesn't exists it will be
 * created.
 *
 * @see config_get_string()
 *
 * @param  group  Configuration section where the new 'key'
 *                should be placed.
 * @param  key    The name of the 'key'
 * @param  value  The new value to set
 */ 
void
config_set_string (const char *group, const char *key, const char *value)
{
	struct class_config *base = &modbase_config;

	g_key_file_set_string (base->config, group, key, value);
}

/**
 * @brief Read a keycode from the configuration file
 *
 * Keyboard definitions consits always of the keycode and none or
 * more modifier keys, that must pressed together to trigger the
 * keyboard action.
 *
 * This function reads the keycode portion from a keyboard key definition.
 * If the 'key' doesn't exists in the configuration file the given
 * default value will be returned.
 *
 * @see config_get_modmask(), config_set_keymod()
 *
 * @param  group          Configuration section where should be looked
 *                        for 'key'
 * @param  key            The key which value should be read
 * @param  default_value  The value that should be returned if 'key' can't
 *                        be read for any reason.
 * @return The keycode.
 */
int
config_get_keycode (const char *group, const char *key, int default_value)
{
	struct class_config *base = &modbase_config;
	gchar *arg;
	int keycode, modmask;

	if ((arg = g_key_file_get_string (base->config, group, key, NULL))) {
		decode_key_and_modifier (arg, &keycode, &modmask);
		g_free (arg);
	} else
		keycode = default_value;

	return keycode;
}

/**
 * @brief Read a modifier mask from the configuration file
 *
 * Keyboard definitions consits always of the keycode and none or
 * more modifier keys, that must pressed together to trigger the
 * keyboard action.
 *
 * This function reads the modifier portion from a keyboard key definition.
 * If the 'key' doesn't exists in the configuration file the given
 * default value will be returned.
 *
 * Supported modifier keys are Shift, Ctrl and Alt. Each modifier key
 * is represented by a bit in the mask. Use the defined values MOD_SHIFT,
 * MOD_ALT and MOD_CTRL to test for a specific modifier key.
 *
 * @see config_get_keycode(), config_set_keymod()
 *
 * @param  group          Configuration section where should be looked
 *                        for 'key'
 * @param  key            The key which value should be read
 * @param  default_value  The value that should be returned if 'key' can't
 *                        be read for any reason.
 * @return The modifier mask.
 */
int
config_get_modmask (const char *group, const char *key, int default_value)
{
	struct class_config *base = &modbase_config;
	gchar *arg;
	int keycode, modmask;

	if ((arg = g_key_file_get_string (base->config, group, key, NULL))) {
		decode_key_and_modifier (arg, &keycode, &modmask);
		g_free (arg);
	} else
		modmask = default_value;

	return modmask;
}

/**
 * @brief Writes a keycode and a modifier mask to the configuration file
 *
 * Keyboard definitions consits always of the keycode and none or
 * more modifier keys, that must pressed together to trigger the
 * keyboard action.
 *
 * This function take a keycode and a modifier mask and writes it as a
 * valid keyboard definition to the configuration file.
 *
 * @see config_get_keycode(), config_get_modmask()
 *
 * @param  group    Configuration section where the keycode should be
 *                  placed.
 * @param  key      Name of the key which keycode should be changed
 * @param  keycode  The new keycode (from linux/input.h)
 * @param  modmask  New Mask of the modifier keys
 */ 
void
config_set_keymod (const char *group, const char *key, int keycode, int modmask)
{
	struct class_config *base = &modbase_config;
	GString *arg;

	/* TODO: following check necessary ? */
	if (keycode == -1) keycode = 0;
	if (modmask == -1) modmask = 0;

	arg = g_string_new (NULL);
	g_string_printf (arg, "%d", (keycode & 255));

	if (modmask & MOD_SHIFT)
		arg = g_string_append (arg, " + shift");
	if (modmask & MOD_ALT)
		arg = g_string_append (arg, " + alt");
	if (modmask & MOD_CTRL)
		arg = g_string_append (arg, " + ctrl");
	
	g_key_file_set_string (base->config, group, key, arg->str);
	g_string_free (arg, TRUE);
}

/**
 * @brief Read a boolean value from the configuration file
 *
 * This function reads a boolean value of a given key from the
 * configuration file. If the key doesn't exists in the configuration
 * file the given default value will be returned.
 *
 * A boolean value is expressed by the strings "yes" for true and
 * "no" for false.
 *
 * @see config_set_boolean()
 *
 * @param  group          Configuration section where should be looked
 *                        for 'key'
 * @param  key            The key which value should be read
 * @param  default_value  The value that should be returned if 'key' can't
 *                        be read for any reason.
 * @return boolean value of 'key' or the default value
 */	
gboolean
config_get_boolean (const char *group, const char *key, gboolean default_value)
{
	struct class_config *base = &modbase_config;
	GError *error = NULL;
	gboolean value;

	value = g_key_file_get_boolean (base->config, group, key, &error);
	if (error) {
		g_error_free (error);
		return default_value;
	}
	return value;
}

/**
 * @brief Write a boolean value to the configuration file
 *
 * This function writes a boolean value under the key 'key' into
 * the section 'group'. If the key or the group does not exist
 * it will be created
 *
 * @see config_get_boolean()
 *
 * @param  *group  section name to place key in
 * @param  *key    key name
 * @param  value   new value for 'key'
 */
void
config_set_boolean (const char *group, const char *key, gboolean value)
{
	struct class_config *base = &modbase_config;

	g_key_file_set_boolean (base->config, group, key, value);
}

/**
 * @brief Read a string list from the configuration file
 *
 * This function reads a string list from a given key in the
 * configuration file. If the key doesn't exists in the configuration
 * file the given default value will be returned. The default
 * value will be converted in a string list. If multiple values
 * are given as default value, they must be seperated by a comma.
 *
 * The strings must be seperated with commas. No other delimiter
 * is allowed.
 *
 * The string list from the configuration file or the default value
 * will be copied to a newly allocated memory. This memory should
 * be freed with g_strfreev() after usage.
 *
 * @see config_set_strlist()
 *
 * @param  group          Configuration section where should be looked
 *                        for 'key'
 * @param  key            The key which value should be read
 * @param  default_values The values that should be returned if 'key' can't
 *                        be read for any reason.
 * @return Pointer to a allocated string list.
 */
gchar **
config_get_strlist (const char *group, const char *key, char *default_values)
{
	struct class_config *base = &modbase_config;
	GError *error = NULL;
	gchar **values, **tmp;

	values = g_key_file_get_string_list (base->config, group, key, NULL, &error);
	if (error) {
		g_error_free (error);
		values = g_strsplit (default_values, ",", 0);
	}

	tmp = values;
	while (*tmp) {
		g_strstrip (*tmp); /* remove leading and trailing whitespaces */
		tmp++;
	}

	return values;
}

/**
 * @brief Write a string list to the configuration file
 *
 * This function writes a string list to the configuration
 * file. If the key doesn't exists it will be created.
 *
 * @see config_set_strlist()
 *
 * @param  group  Configuration section where the new 'key'
 *                should be placed.
 * @param  key    The name of the 'key'
 * @param  values The new string list to set
 */ 
void
config_set_strlist (const char *group, const char *key, const char *values[])
{
	struct class_config *base = &modbase_config;

	g_key_file_set_string_list (base->config, group, key,
			values, g_strv_length ((char **) values));
}

/**
 * @brief Read a integer list from the configuration file
 *
 * This function reads list of integers from a given key in the
 * configuration file. This function will fail under following 
 * circumstances:
 *   @li  the key doesn't exists in the configuration file
 *   @li  the key has less integers as requested
 *   @li  one ore more list elements can't be trensformed to intergers
 *
 * In all those cases the default value will be returned. A single
 * default value will be converted to an integer list. If multiple
 * values are given as default value, they must be seperated by a
 * comma.
 *
 * The integers must be seperated with commas. No other delimiter
 * is allowed.
 *
 * The integer list from the configuration file or the default value
 * will be copied to a newly allocated memory. This memory should
 * be freed with g_free() after usage.
 *
 * @param  group   Configuration section where should be looked
 *                 for 'key'
 * @param  key     The key which value should be read
 * @param  count   How many integers should be read from the list
 * @param  ...     The values that should be returned if 'key' can't
 *                 be read for any reason. Exactly 'count' integers
 *                 must be given here.
 * @return Pointer to a allocated integer list.
 */
gint *
config_get_intlist (const char *group, const char *key, unsigned int count, ...)
{
	struct class_config *base = &modbase_config;
	unsigned int n;
	va_list list;
	GError *error = NULL;
	gsize num_ints;
	gint *values;

	values = g_key_file_get_integer_list (base->config, group, key, &num_ints, &error);
	if (error || num_ints != count) {
		if (error) {
			if (g_error_matches (error, G_KEY_FILE_ERROR, G_KEY_FILE_ERROR_INVALID_VALUE))
				print_msg (PBB_WARN, _("config error: %s/%s contains non-integers - using defaults.\n"), group, key);
			g_error_free (error);
		} else if (num_ints != count) {
			print_msg (PBB_WARN, _("config error: %s/%s needs exactly %d arguments - using defaults.\n"), group, key, count);
			g_free (values);
		}

		values = g_new0 (gint, count);
		va_start (list, count);
		for (n=0; n < count; n++)
			values[n] = va_arg (list, gint);
		va_end (list);
	}

	return values;
}

/**
 * @brief Write a list of integers to the configuration file
 *
 * This function writes a list of integers to the configuration
 * file. If the key doesn't exists it will be created.
 *
 * @param  group  Configuration section where the new 'key'
 *                should be placed.
 * @param  key    The name of the 'key'
 * @param  count  count of integers to be written
 * @param  ...    integers for the list. Exactly 'count' integers
 *                must be given.
 */ 
void
config_set_intlist (const char *group, const char *key, unsigned int count, ...)
{
	struct class_config *base = &modbase_config;
	unsigned int n;
	va_list list;
	gint *values;

	values = g_new0 (gint, count);
	va_start (list, count);
	for (n=0; n < count; n++)
		values[n] = va_arg (list, gint);
	va_end (list);
	
	g_key_file_set_integer_list (base->config, group, key,
			values, count);

	g_free (values);
}

/**
 * @brief read a string from the configuration file and return an appropriate enum value
 *
 * This function reads the value of a given key from the configuration
 * file and treat it as percentage value. Only the read value will be
 * checked for percentage bounds and clipped as necessary. The default
 * value will be returned as is. This allows to pass invalid values to
 * check for errors.
 *
 * If the key doesn't exists in the configuration file the given default
 * value will be returned.
 *
 * @param  group          Configuration section where should be looked
 *                        for 'key'
 * @param  key            The key which value should be read
 * @param  optionlist     Null terminated array of allowed options. The 
 *                        position of an option string in this array defines
 *                        its return value.
 * @param  default_value  The value that should be returned if 'key' can't
 *                        be read for any reason.
 * @return The value for 'key' or the default_value on error, clipped
 *         to percentage bounds if necessary.
 */ 
int
config_get_option (const char *group, const char *key, const char *optionlist[], int default_value)
{
	struct class_config *base = &modbase_config;
	gchar *arg;
	int n, rc = default_value;

	arg = g_key_file_get_string (base->config, group, key, NULL);

	if (arg) {
		for (n=0; optionlist[n] != NULL; n++) {
			if (strcasecmp (arg, optionlist[n]) == 0) {
				rc = n;
				break;
			}
		}
		g_free (arg);
	}
	return rc;
}

/**
 * @brief transform an enum to a string and write it to the configuration file
 *
 * This function gets a enum value and a null terminated array of options  writes a new value to a given key in the configuration
 * file. The given value will be checked for percentage bounds and clipped
 * as necessary.
 *
 * If the key doesn't exists it will be created.
 *
 * @param  group       Configuration section where the key/value pair
 *                     should be changed/created.
 * @param  key         The key which value should be changed
 * @param  optionlist  Null-terminated array of options
 * @param  value       The enum value which string of optionlist should be
 *                     written to the configuration file
 */ 
void
config_set_option (const char *group, const char *key, const char *optionlist[], int value)
{
	struct class_config *base = &modbase_config;
	int n;

	for (n=0; optionlist[n] != NULL; n++);

	if (value <= n)
		g_key_file_set_string (base->config, group, key, optionlist[value]);
}

/**
 * @brief parse a string and extract keycode and modifier mask
 *
 * This function parses a string and extract a keycode and a modifier
 * mask from it.
 *
 * The keycode is an integer in the range of 0..255. All values in
 * linux/input.h are valid. The keycode will be found at any position
 * in the string.
 *
 * Valid modifier keys are 'shift', 'ctrl' and 'alt'. None or more
 * modifiers could be attached to a keycode. The glue is a '+' sign.
 *
 * @param  *arg  String to be parsed
 * @param  *key  Pointer to an integer where the keycode should be
 *               written to
 * @param  *mod  Pointer to an integer where the modifier mask
 *               should be written to
 */
void
decode_key_and_modifier(char *arg, int *key, int *mod)
{
	char *tag;
	int k;

	*mod = 0; *key = 0;
	cleanup_buffer(arg); /* remove all whitespaces */

	tag = strtok(arg, "+\n");
	while (tag) {
		k = atoi (tag);
		if (k == 0) {
			if (!strncmp("shift", tag, 5))
				*mod |= MOD_SHIFT;
			else if (!strncmp("alt", tag, 3))
				*mod |= MOD_ALT;
			else if (!strncmp("ctrl", tag, 4))
				*mod |= MOD_CTRL;
		} else
			*key = k;
		tag = strtok(NULL, "+\n");
	}
}

