---
myst:
  html_meta:
    keywords: optique, schéma, schema, tikz, TikZ, optique géométrique
---
# Comment réaliser des schémas d’optique géométrique avec Ti*k*Z ?

Pour tracer des schémas, on peut utiliser le package <ctanpkg:tikz>.

:::{note}
Pour construire le schéma, en particulier les rayons lumineux,  on utilise les
bibliothèques Ti*k*Z `decorations.markings` et `arrows.meta`.

Leur fonctionnement et leur intérêt pour ces schémas sont détaillés dans la question
[](/3_composition/illustrations/tracer_ligne_fleche_au_milieu.md).

Pour bien comprendre cette réponse, il est nécessaire d’être familier avec les bases de
<ctanpkg:tikz>, en particulier la gestion des coordonnées (polaires et cartésiennes) ainsi
que la gestion des coordonnées relatives. Si ce n’est pas le cas, on pourra se reporter à sa
[documentation](https://texlive.mycozy.space/graphics/pgf/base/doc/pgfmanual.pdf), ou à
l’excellent [TikZ pour l'impatient](http://math.et.info.free.fr/TikZ/index.html).
:::

:::{tip}
Il existe une bibliothèque Ti*k*Z spécialement pour les schémas
d’optique qui semble fonctionner: <ctanpkg:circuitikz>. Mais elle n’a pas été mise à jour
depuis 2017 et n’est pas référencée dans la documentation officielle de
<ctanpkg:tikz>. Elle ne sera donc pas abordée dans cette réponse.
:::

## Construction d’une image réelle par une lentille mince convergente
### Le code

```
\documentclass[french]{article}
\usepackage[T1]{fontenc}
\usepackage[utf8]{inputenc}
\usepackage[a4paper]{geometry}
\usepackage{tikz}
\usetikzlibrary{decorations.markings, arrows.meta}
\usepackage{babel}

\begin{document}

\begin{tikzpicture}
	% Définitions des styles
	\tikzset{simple/.style={postaction={decorate,decoration={
		markings, mark= at position .5with{\arrow{Straight Barb[] }}}}}}
	\tikzset{double/.style={postaction={decorate,decoration={
        markings,mark=at position .5 with{\arrow{Straight Barb[] Straight Barb[]}}}}}}
	\tikzset{triple/.style={postaction={decorate,decoration={
        markings,mark=at position .5 with{\arrow{Straight Barb[] Straight Barb[] Straight Barb[]}}}}}}
	
	% Figure
	%% paramètres modifiables
	\newcommand \tailleL {3.5} % taille de la lentille
	\newcommand \tailleOb {2.8} % hauteur de l'objet
	\newcommand \xA {-5} % position de l'objet
	\newcommand \f {2} % focale de la lentille
	%% Calculs
	\newcommand \xAA {(\xA*\f)/(\xA+\f)} % position de l'image
	\newcommand \tailleIm {(\xAA*\tailleOb)/\xA} % hauteur de l'image
	%% Définitions des points
	\coordinate (O) at (0,0);
	\coordinate (F') at (\f,0);
	\coordinate (A) at (\xA,0);
	\coordinate (B) at (\xA,\tailleOb);
	\coordinate (A') at ({\xAA},0);
	\coordinate (B') at ({\xAA},{\tailleIm});
	%% Lentille
	\draw (O) node[below left] {O};
	\draw (\f,-2pt) --++ (0,4pt) node[above, xshift=4pt] {F'};
	\draw ({-\f},2pt) --++ (0,-4pt) node[below] {F};
	\draw[ultra thick ,<->,] (0,-\tailleL)--(0,\tailleL)
		node[above, yshift=-.1cm]{Lentille};
	%% Axe optique
	\draw[ultra thick,->](\xA-0.5,0)--({\xAA+2.5},0);
	%% Objet
	\draw [ultra thick,->, >=latex](A) node[below]{A} -- (B) node [above]{B};
	%% Image
	\draw [ultra thick,->,>=latex](A') node[above right]{A'}
		--++(0,{\tailleIm})
		node [below,right]{B'};
	%% Rayons
	%%% Parallèle à l’axe optique
	\draw[red,simple,thick] (B)--(0,\tailleOb);
	\draw[red, simple,thick] (0,\tailleOb)--(B');
	%%% Passant par F
	\draw[green!50!black,triple,thick] (B)--(0,{\tailleIm});
	\draw[green!50!black,triple,thick] (0,{\tailleIm})--(B');
	%%% Passant par F'
	\draw[blue,double,thick] (B)--(O);
	\draw[blue,double, thick] (O)--(B');
\end{tikzpicture}

\end{document}
```

### Explications
La définition des styles de <ctanpkg:tikz> est détaillée dans
[](/3_composition/illustrations/tracer_ligne_fleche_au_milieu.md).

On comprend ici q’ils sont utilisés pour les trois rayons.

:::{note}
Ici les styles sont définis dans l’environnement `tikzpicture`, ils ne seront donc valables
que dans cet environement et devront être redéfinis pour une autre figure.

Si on les définit *en dehors* de l’environnement, ils pourront être utilisés dans toutes
les figures du document.
:::

Pour le tracé du graphique, après avoir choisi la taille de la lentille ainsi que la
	position et la taille de l’objet, on calcule celles de l’image.

:::{attention}
Les « variables »  qui sont « calculées », doivent être mises entre
accolades dans les commandes `\draw` ou `\coordinate` pour que la compilation
réussisse.

Dans le cas présent `\tailleOb` n’en nécessite pas tandis que `\tailleIm` ou `\xAA` si.
:::

On définit ensuite les différents points d’intérêt pour le tracé.

:::{note}
Il est possible de réduire la longueur du code, mais cette proposition a le mérite d’être explicite
pour les débutant·es et (on l’espère) facilement modifiable et adaptable.
:::
### Ajout d’une grille et d’une échelle
Sur le schéma précédent (et tous les suivants) on peut indiquer une échelle et, en
commentant certaines lignes faire un schéma à compléter.

```
\documentclass[french]{article}
\usepackage[T1]{fontenc}
\usepackage[utf8]{inputenc}
\usepackage[a4paper]{geometry}
\usepackage{tikz}
\usetikzlibrary{decorations.markings, arrows.meta}
\usepackage{babel}

\begin{document}

\begin{tikzpicture}
	% Définitions des styles
	\tikzset{simple/.style={postaction={decorate,decoration={
		markings, mark= at position .5with{\arrow{Straight Barb[] }}}}}}
	\tikzset{double/.style={postaction={decorate,decoration={
        markings,mark=at position .5 with{\arrow{Straight Barb[] Straight Barb[]}}}}}}
	\tikzset{triple/.style={postaction={decorate,decoration={
        markings,mark=at position .5 with{\arrow{Straight Barb[] Straight Barb[] Straight Barb[]}}}}}}

	% Figure
	%% Graduations
	%%% Dimensions du repere
	\newcommand\xmin {-6}
	\newcommand\xmax {6}
	\newcommand\ymin {-4}
	\newcommand\ymax {4}
	%%% Grilles
	\draw [step=0.1cm,gray!20,ultra thin]  (\xmin,\ymin) grid (\xmax,\ymax);
	\draw [step=0.5cm,gray, thin] (\xmin,\ymin) grid (\xmax,\ymax);
	\draw [step=1cm,gray, thick] (\xmin,\ymin) grid (\xmax,\ymax);
	\draw [step=5cm,gray,very thick] (\xmin,\ymin) grid (\xmax,\ymax);
	%% Echelle
	\newcommand \xC {4}
	\newcommand \yC {2}
	\coordinate (C) at (\xC,\yC);
	\draw[->, ultra thick, >=latex] (C)--(\xC,\yC+1)
	node[midway,left] {1~cm};
	\draw[->, ultra thick, >=latex] (C)--(\xC+1,\yC)
		node[midway,below] {10~cm};
	%% paramètres modifiables
	\newcommand \tailleL {3.5} % taille de la lentille
	\newcommand \tailleOb {2.8} % hauteur de l'objet
	\newcommand \xA {-5} % position de l'objet
	\newcommand \f {2} % focale de la lentille
	%% Calculs
	\newcommand \xAA {(\xA*\f)/(\xA+\f)} % position de l'image
	\newcommand \tailleIm {(\xAA*\tailleOb)/\xA} % hauteur de l'image
	%% Définitions des points
	\coordinate (O) at (0,0);
	\coordinate (F') at (\f,0);
	\coordinate (A) at (\xA,0);
	\coordinate (B) at (\xA,\tailleOb);
	\coordinate (A') at ({\xAA},0);
	\coordinate (B') at ({\xAA},{\tailleIm});
	%% Lentille
	\draw (O) node[below left] {O};
	\draw (\f,-2pt) --++ (0,4pt) node[above, xshift=4pt] {F'};
	\draw ({-\f},2pt) --++ (0,-4pt) node[below] {F};
	\draw[ultra thick ,<->,] (0,-\tailleL)--(0,\tailleL)
		node[above, yshift=-.1cm]{
	Lentille};
	%% Axe optique
	\draw[ultra thick,->](\xA-0.5,0)--({\xAA+2.5},0);
	%% Objet
	\draw [ultra thick,->, >=latex](A) node[below]{A} -- (B)
		node [above]{B};
	%% Image
	\draw [ultra thick,->,>=latex](A') node[above right]{A'}
		--++(0,{\tailleIm}) node [below,right]{B'};
	%% Rayons
	%%% Parallèle à l’axe optique
	\draw[red,simple,thick] (B)--(0,\tailleOb);
	\draw[red,simple,thick] (0,\tailleOb)--(B');
	%%% Passant par F
	\draw[green!50!black,triple,thick] (B)--(0,{\tailleIm});
	\draw[green!50!black,triple,thick] (0,{\tailleIm})--(B');
	%%% Passant par F'
	\draw[blue,double,thick] (B)--(O);
	\draw[blue,double, thick] (O)--(B');
\end{tikzpicture}

\end{document}
```
	
## Construction d’une image virtuelle par une lentille mince convergente

### Le code

```
\documentclass[french]{article}
\usepackage[T1]{fontenc}
\usepackage[utf8]{inputenc}
\usepackage[a4paper]{geometry}
\usepackage{tikz}
\usetikzlibrary{decorations.markings, arrows.meta}
\usepackage{babel}

\begin{document}
% Définition des styles
\tikzset{simple/.style={postaction={decorate,decoration={
        markings, mark= at position .75 with{\arrow{Straight Barb[] }}}}}}
\tikzset{double/.style={postaction={decorate,decoration={
        markings,mark=at position .75 with{\arrow{Straight Barb[] Straight Barb[]}}}}}}
\tikzset{triple/.style={postaction={decorate,decoration={
        markings,mark=at position .75 with{\arrow{Straight Barb[] Straight Barb[] Straight Barb[]}}}}}}


\begin{tikzpicture}
    % paramètres modifiables
    \newcommand \tailleL {4}
    \newcommand \tailleOb {1.2}
    \newcommand \xA {-2.5} % position de l'objet
    \newcommand \f{4} % focale de la lentille
    % calculs
    \newcommand \xAA {(\xA*\f)/(\xA+\f)} % position de l'image
    \newcommand \tailleIm {(\xAA*\tailleOb)/\xA} % taille de l'image
    % Définitions  des points
    \coordinate (O) at (0,0);
    \coordinate (F') at (\f,0);
    \coordinate (F) at (-\f,0);
    \coordinate (A) at (\xA, 0);
    \coordinate (B) at (\xA, \tailleOb);
    \coordinate (A') at ({\xAA}, 0);
    \coordinate (B') at ({\xAA}, {\tailleIm}) ;
    % Schéma
    \draw (O) node[below left]{O};
    \draw (\f,-2pt) --++ (0,4pt) node[above] {F'};
    \draw ({-\f},2pt) --++ (0,-4pt) node[below] {F};
    \draw[shift={(O)},ultra thick ,<->,](0,-\tailleL)--(0,\tailleL)
		node[above]{ Lentille};
    % axe optique
    \draw[thin,->]({\xAA*1.2},0)--({-\xAA*1.2},0);
    % objet
    \draw[thick,->,>=latex] (A) node[below]{A} -- (B) node [above]{B};
    % image
    \draw [thick,->,>=latex](A') node[below=0.04cm]{A'}--++(0,{\tailleIm})
		node [below,left]{B'};
    % rayons
	%% Passant par O
    \draw[blue,double] (B)--(O);
    \draw[blue,dashed] (O)--(B');
    \draw[blue, double] (O)--({-\xAA},{-\tailleIm});
    %% Passant par F
    \draw[green!50!black,triple] (B)--(0,{\tailleIm});
    \draw[green!50!black, dashed](0,{\tailleIm})--(B');
    \draw[green!50!black,triple] (0,{\tailleIm})--(2*\f,{\tailleIm});
    \draw[green!50!black,dashed] (-\f,0)--(B);
    % Parallèle à l’axe optique
    \draw[red,simple] (B)--(0,\tailleOb);
    \draw[red,dashed] (0,\tailleOb)--(B');
    \draw[red, simple] (0,\tailleOb)--(2*\f, -\tailleOb);
\end{tikzpicture}
\end{document}
```
### Explications
Si la construction de l’image réelle est bien comprise, ce code ne devrait pas poser de
difficultés. Les calculs sont les mêmes, et on a simplement ajouté le prolongement des
rayons en pointillés.

:::{note}
Dans la défintion des styles, on peut choisir la position des flèches sur la ligne. Ici, on
a choisi de les placer au 3/4 de la longeur du chemin pour plus de visibilité.

On pourra essayer de les remettre au milieu comme la partie précédente avec `mark at
position .5` afin d’observer la différence.
:::

## Construction de l’image d’un objet à l’infini par une lentille mince convergente
### Le code
```
\documentclass[french]{article}
\usepackage[T1]{fontenc}
\usepackage[utf8]{inputenc}
\usepackage[a4paper]{geometry}
\usepackage{tikz}
\usetikzlibrary{decorations.markings, arrows.meta}
\usepackage{babel}

\begin{document}
% Définition des styles
\tikzset{simple/.style={postaction={decorate,decoration={
	markings, mark= at position .8with{\arrow{Straight Barb[] }}}}}}

\tikzset{simpleInv/.style={postaction={decorate,decoration={
	markings,mark=at position .8 with{\arrowreversed{Straight Barb[]}}}}}}

\tikzset{double/.style={postaction={decorate,decoration={
	markings,mark=at position .8 with{\arrow{Straight Barb[] Straight Barb[]}}}}}}

\tikzset{doubleInv/.style={postaction={decorate,decoration={
	markings,mark=at position .2 with{\arrowreversed{Straight Barb[] Straight Barb[]}}}}}}

\tikzset{triple/.style={postaction={decorate,decoration={
	markings,mark=at position .5 with{\arrow{Straight Barb[] Straight Barb[] Straight Barb[]}}}}}}

\tikzset{tripleInv/.style={postaction={decorate,decoration={
	markings,mark=at position .5 with{\arrowreversed{Straight Barb[] Straight Barb[] Straight Barb[]}}}}}}

\begin{tikzpicture}
    % Paramètres modifiables
	\newcommand\tailleL{3}
	\newcommand\tailleIm{-1}
	\newcommand\f{2}
    % Calculs
	\newcommand \angleB {\fpeval{atand(-1*\tailleIm / \f)}}
    % Définitions  des points
	\coordinate (O) at (0,0);
	\coordinate (F') at (\f,0);
	\coordinate (A') at (\f,0);
	\coordinate (B') at (\f,\tailleIm);
    % Schéma
	%% plan focal image
	\draw[gray!80!white, dashed] (\f,{-\tailleL})
		node[below]{\textcolor{gray!80!white}{Plan focal image}}
		-- (\f,{\tailleL});
    %% Lentille
	\draw[ultra thick ,<->,>=latex] (0,-\tailleL)--(0,\tailleL)
		node[above]{Lentille} node[midway, below left]{O};
	%% Axe optique
	\draw[thin,->](-\f*4,0)--(\f*2,0);
    %% Image
	\draw [thick,->,>=latex](A')--++(0,\tailleIm) node [below,right]{B'};
    %% Rayons
	%%% issus de A
	\draw[blue, simple] (-\f*3, \tailleL/2) node [above, left, blue]{$A_\infty$}
		-- (0, \tailleL/2);
	\draw[blue, simple] (0,\tailleL/2) -- (A');
	\draw[blue, simple] (-\f*3, -\tailleL/2) node [above, left, blue]{$A_\infty$}
		--(0, {-\tailleL/2});
	\draw[blue, simple] (0, -\tailleL/2) --(A') node[above right]{A};
    %%% issus de B
	\draw[red, doubleInv] (B') -- (O);
	\draw[red, doubleInv] (O) --++ (180-\angleB:7) node [above, left, red]{$B_\infty$};
	\draw[red, doubleInv] (B') -- (0, \tailleIm);
	\draw[red, doubleInv] (0, \tailleIm) --++ (180-\angleB:7)
	node [above, left, red]{$B_\infty$};
\end{tikzpicture}

\end{document}
```

### Explications
Pour ce schéma, on commence par fixer la taille de l’image et la focale de la lentille.

La position de l’image est définie de fait dans cette situation par la focale de la
lentille.

Les rayons issus de A ne posent pas de problèmes à tracer.

:::{important}
L’astuce ici est de *calculer l’angle entre le rayon passant O et l’axe optique* puis de
s’en servir pour *tracer les rayons issus de B « à l’envers »*, c’est à dire en partant de
B’. C’est pour cela qu’on a créé trois nouveaux styles dans lesquels les flèches sont à l’envers.
:::

:::{note}
On remarque qu’en utilisant les coordonées polaires, il n’est pas nécessaire de mettre les
« variables » calculées, ici `\angleB`, entre accolades.
:::

## Construction d’une image à l’infini par une lentille mince convergente

### Le code
```
\documentclass[french]{article}
\usepackage[T1]{fontenc}
\usepackage[utf8]{inputenc}
\usepackage[a4paper]{geometry}
\usepackage{tikz}
\usetikzlibrary{decorations.markings, arrows.meta}
\usepackage{babel}

\begin{document}

% Définition des styles
\tikzset{simple/.style={postaction={decorate,decoration={
	markings, mark= at position .8with{\arrow{Straight Barb[] }}}}}}

\tikzset{double/.style={postaction={decorate,decoration={
	markings,mark=at position .8 with{\arrow{Straight Barb[] Straight Barb[]}}}}}}

\tikzset{triple/.style={postaction={decorate,decoration={
	markings,mark=at position .5 with{\arrow{Straight Barb[] Straight Barb[] Straight Barb[]}}}}}}

\begin{tikzpicture}
    % Paramètres modifiables
	\newcommand\tailleL{3}
	\newcommand\tailleOb{1}
	\newcommand\f{2}
    % Calculs
	%% Angle entre les rayons issus de B et l’axe optique
	\newcommand \angleB {\fpeval{atand(\tailleOb / \f)}}
    % Définitions des points
	\coordinate (O) at (0,0);
	\coordinate (F) at (-\f,0);
	\coordinate (A) at (-\f,0);
	\coordinate (B) at (-\f,\tailleOb);
    % Schéma
	%% Lentille
	\draw[ultra thick ,<->,>=latex] (0,-\tailleL)--(0,\tailleL)
		node[above]{Lentille};
    %% Axe optique
	\draw[thin,->](-\f*2,0)--(\f*2.5,0);
	%% Placements des points
	\draw (O) node[below left] {O};
	\draw (-\f,4pt) --++ (0,-4pt) node[below left] {F};
	\draw (\f,4pt) --++ (0,-4pt) node[above right, yshift=3pt]{F'};
    %% Objet
	\draw[thick, ->, >=latex] (A) node[below right]{A'} -- (B) node[above right]{B};
    %% Rayons issus de B
	\draw[blue, simple] (B) -- (O);
	\draw[blue, simple] (O) -- (-\angleB:4)node[right]{$B_\infty$};
	\draw[red, double] (B) -- (0,\tailleOb);
	\draw[red, double] (0, \tailleOb) --++ (-\angleB:4)node[right]{$B_\infty$};
	%% Prolongement des rayons
    \draw[red, dashed] (0, \tailleOb) --++ (180 - \angleB:4);
    \draw[blue, dashed] (B) --++ (180 - \angleB:2);
\end{tikzpicture}

\end{document}
```	
### Explications
Ici l’astuce est de calculer l’angle entre le rayon bleu et l’axe optique.

Ensuite, on utilise cet angle, et les coordonnées polaires et relatives pour tracer la
deuxième partie du rayon rouge, ainsi que le prolongement des rayons.



## La lunette astronomique

### Le code
```
\documentclass[french]{article}
\usepackage[T1]{fontenc}
\usepackage[utf8]{inputenc}
\usepackage[a4paper]{geometry}
\usepackage{tikz}
\usetikzlibrary{decorations.markings, arrows.meta}
\usepackage{babel}

\begin{document}

% Définition des styles
\tikzset{simple/.style={postaction={decorate,decoration={
	markings, mark= at position .8with{\arrow{Straight Barb[] }}}}}}

\tikzset{simpleInv/.style={postaction={decorate,decoration={
	markings,mark=at position .8 with{\arrowreversed{Straight Barb[]}}}}}}

\tikzset{double/.style={postaction={decorate,decoration={
	markings,mark=at position .8 with{\arrow{Straight Barb[] Straight Barb[]}}}}}}

\tikzset{doubleInv/.style={postaction={decorate,decoration={
	markings,mark=at position .2 with{\arrowreversed{Straight Barb[] Straight Barb[]}}}}}}

\tikzset{triple/.style={postaction={decorate,decoration={
	markings,mark=at position .5 with{\arrow{Straight Barb[] Straight Barb[] Straight Barb[]}}}}}}

\tikzset{tripleInv/.style={postaction={decorate,decoration={
	markings,mark=at position .5 with{\arrowreversed{Straight Barb[] Straight Barb[] Straight Barb[]}}}}}}

\begin{tikzpicture}
  \newcommand{\oeil}[1][]
  {
    \draw[#1,thick,rounded corners] (-15:1)--(0,0)--(15:1);
    \draw[#1] (-15:0.8) arc (-15:15:0.8);
    \draw[#1,fill=black] (0:0.75) ellipse(0.05 and 0.1);
  }
  % Paramètre modifiables
  \newcommand \tailleLUn {3}
  \newcommand \tailleLDe {3}
  \newcommand \fUn {4} % focale de l’objectif
  \newcommand \fDe {2} % focale de l’oculaire
  \newcommand \tailleIm {1.1} % taille image intermédiaire
  \newcommand \RDeAO {\fpeval{ \fUn / 2 }} %intersection R2 et axe optique

  % Calculs
  \newcommand \posLDe {\fUn + \fDe} % position L2
  \newcommand \aUn {\fpeval{atand(\tailleIm / \fUn)}} % alpha 1
  \newcommand \aDe {\fpeval{atand(\tailleIm / \fDe)}} % alpha 2
  \newcommand \RUnLDe {\fpeval{ ( \posLDe ) / \fUn * \tailleIm }} % rayon 1 sur L2
  %% angle du rayon 2 avec l’axe optique
  \newcommand \aRDe {\fpeval { atand(\tailleIm / ( \fUn - \RDeAO))}}
  %% hauteur de l’intersection entre les rayons et les lentilles
  \newcommand \RDeLUn {\fpeval{ tand(\aRDe) * \RDeAO}} % rayon 2 sur L1
  \newcommand \RDeLDe {\fpeval{ tand(\aRDe) * (\posLDe - \RDeAO)}} % rayon 2 sur L2

  % Définition des points
  \coordinate (O1) at (0,0);
  \coordinate (O2) at (\posLDe,0);
  \coordinate (FUn) at ({-\fUn},0);
  \coordinate (FpUn) at ({\fUn},0);
  \coordinate (B1) at (\fUn, {-\tailleIm});
  \coordinate (R1L2) at (\posLDe, {-\RUnLDe}); % arrivée du rayon 1 sur L2
  \coordinate (R2L1) at (0, \RDeLUn); %rayon 2 sur L1
  \coordinate (R2L2) at (\posLDe, {-\RDeLDe}); %arrivé rayon 2 sur L2

  % Tracés
  %% Image 1
  \draw[thick, ->, >=latex] (FpUn) node[below left] {$F'_1$}
    node[above left]{$F_2$} -- (B1)
  node[below left]{$B_1$};

  \draw[thin,->](-2.5,0) -- (13,0);
  \draw[shift={(O1)},ultra thick,<->] (0, {-\tailleLUn}) -- (0,\tailleLUn)
  node[above]{$L_1$} node[midway, below left]{$O_1$};
  \draw[shift={(O2)},ultra thick,<->] (0, {-\tailleLDe}) -- (0,\tailleLDe)
  node[above]{$L_2$} node[midway, below right]{$O_2$};

  %% Rayon 1 issu de B (passant par O1)
  \draw[simpleInv, blue] (O1) -- (180 -\aUn : 4);
  \draw[simpleInv, blue] (B1) -- (O1);
  \draw[blue] (B1) -- (R1L2);
  \draw[simple, blue] (R1L2) --++ (\aDe : 6);
  \draw[dashed, blue] (R1L2) --++ (180 + \aDe :5)
	node[below left]{$B_{2\infty}$};

  %% Rayon 2 issu de B (quelconque)
  \draw[doubleInv, red] (B1) -- (R2L1);
  \draw[doubleInv, red] (R2L1) --++ (180-\aUn:4);
  \draw[red] (B1) -- (R2L2);
  \draw[double, red] (R2L2) --++ (\aDe:6);
  \draw[dashed, red] (R2L2) --++ (180 + \aDe:5)
	node[below left]{$B_{2\infty}$};

  %% Trait de construction
  \draw[dashed] (B1) -- (O2) --++ (\aDe:6);

  %% Angles
  \draw (-1.5,0) arc (180 : 180 - \aUn : 1.5) node[midway, left] {$\alpha_1$};
  \draw[shift={(O2)}] (-1,0) arc (180 : 180 + \aDe : 1)
	node[midway, left] {$\alpha_2$};
  \draw[shift={(O2)}] (1,0) arc (0 :  \aDe : 1)
	node[midway, right] {$\alpha_2$};
  % oeil -- à placer manuellement
  \oeil[shift={(12,1.5)},rotate={180 + \aDe },scale=0.75]
\end{tikzpicture}

\end{document}
```
### Explications

Pour la lunette astronomique, on reprend sensiblement les mêmes méthodes que pour les
parties précédentes.

Dans ce cas, l’astuce concerne le rayon rouge et l’ordre dans lequel on fait la
construction :
- on fixe d’abord le point d’intersection (`\RDeAO` dans le code) entre ce rayon et l’axe
  optique ainsi que la taille de l’image intermédiaire;
- comme on connait également la position de $F_2$, on peut calculer la distance entre ce
  point et $F_2$;
- on en déduit la valeur de l’angle entre ce rayon et l’axe optique (`\aDe`);
- avec cet angle, on peut calculer les points d’intersection entre ce rayon et chaque
  lentille (`\RDeLUn` et `\RDeLDe`);
- les calculs de  $\alpha_1$ et $\alpha_2$ se font sans difficultés puisqu’on a fixé la
  taille de l’image intermédiaire;
- ce qui permet de terminer le tracé du rayon rouge : avec la partie avant la lentille 1
  et celle après la lentille.

:::{note}
Pour le tracé des angles, on pourra se référer aux pages 16 et 17 de [Ti*k*Z pour
l’impatient](http://math.et.info.free.fr/TikZ/bdd/TikZ-Impatient.pdf).
:::
:::{sources}
[physagreg.fr](https://www.physagreg.fr/schemas-figures-physique-svg-tikz.php)
:::
