//
// /home/ms/source/sidplay/libsidplay/fformat/RCS/mus_.cpp,v
//

#include "mus_.h"


const char text_format[] = "C64 Sidplayer format (MUS)";
const uword HLTcmd = 0x14F;


const char CHRtab[256] =  // CHR$ conversion table (0x01 = no output)
{
   0x0, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0xd, 0x1, 0x1,
   0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
  0x20,0x21, 0x1,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,0x2e,0x2f,
  0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
  0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,
  0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,0x5b,0x24,0x5d,0x20,0x20,
  // alternative: CHR$(92=0x5c) => ISO Latin-1(0xa3)
  0x2d,0x23,0x7c,0x2d,0x2d,0x2d,0x2d,0x7c,0x7c,0x5c,0x5c,0x2f,0x5c,0x5c,0x2f,0x2f,
  0x5c,0x23,0x5f,0x23,0x7c,0x2f,0x58,0x4f,0x23,0x7c,0x23,0x2b,0x7c,0x7c,0x26,0x5c,
  // 0x80-0xFF
   0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
   0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1, 0x1,
  0x20,0x7c,0x23,0x2d,0x2d,0x7c,0x23,0x7c,0x23,0x2f,0x7c,0x7c,0x2f,0x5c,0x5c,0x2d,
  0x2f,0x2d,0x2d,0x7c,0x7c,0x7c,0x7c,0x2d,0x2d,0x2d,0x2f,0x5c,0x5c,0x2f,0x2f,0x23,
  0x2d,0x23,0x7c,0x2d,0x2d,0x2d,0x2d,0x7c,0x7c,0x5c,0x5c,0x2f,0x5c,0x5c,0x2f,0x2f,
  0x5c,0x23,0x5f,0x23,0x7c,0x2f,0x58,0x4f,0x23,0x7c,0x23,0x2b,0x7c,0x7c,0x26,0x5c,
  0x20,0x7c,0x23,0x2d,0x2d,0x7c,0x23,0x7c,0x23,0x2f,0x7c,0x7c,0x2f,0x5c,0x5c,0x2d,
  0x2f,0x2d,0x2d,0x7c,0x7c,0x7c,0x7c,0x2d,0x2d,0x2d,0x2f,0x5c,0x5c,0x2f,0x2f,0x23
};


bool sidTune::MUS_fileSupport(const void* musBuffer, udword musLength)
{
    // Remove any format description or format error string.
    info.formatString = 0;
	
    smartPtr<const ubyte> spMus((const ubyte*)musBuffer,musLength);
    // Skip load address and 3x length entry.
    udword voice1Index = (2+3*2);
    // Add length of voice 1 data.
    voice1Index += readEndian(spMus[3],spMus[2]);
    // Add length of voice 2 data.
    udword voice2Index = voice1Index + readEndian(spMus[5],spMus[4]);
    // Add length of voice 3 data.
    udword voice3Index = voice2Index + readEndian(spMus[7],spMus[6]);
    if ( (readEndian(spMus[voice1Index-2],spMus[voice1Index+1-2]) != HLTcmd)
         || (readEndian(spMus[voice2Index-2],spMus[voice2Index+1-2]) != HLTcmd)
         || (readEndian(spMus[voice3Index-2],spMus[voice3Index+1-2]) != HLTcmd)
         || !spMus )
    {
        return false;
    }

    infoString[0][0] = 0;  // clear info strings
    infoString[1][0] = 0;
    infoString[2][0] = 0;
    infoString[3][0] = 0;
    infoString[4][0] = 0;

    // Voice3Index now is offset to text lines (uppercase Pet-strings).
    smartPtr<const ubyte> spPet((const ubyte*)musBuffer,musLength);
    spPet += voice3Index;
    for ( int line = 0; line < 5; line++ )
    {
        char c;
        char si = 0;  // count copied characters
        do
        {
            // ASCII CHR$ conversion
            c = CHRtab[*spPet];
            if ((c>=0x20) && (si<=31))
            {
                infoString[line][si++] = c;  // copy to info string
            }
            // If character is 0x9d (left arrow key) then move back.
            if ((*spPet == 0x9d) && (si>0))
            {
                si--;
            }
            spPet++;
        }
        while ( !((c == 0x0D) || (c == 0x00) || spPet.fail()) );
        info.infoString[line] = &infoString[line][0];
    }
    info.numberOfInfoStrings = 5;

    info.loadAddr = 0x0900;
    info.initAddr = 0xcc90;
    info.playAddr = 0;
    info.songs = (info.startSong = 1);
    info.musPlayer = true;
    fileOffset = 2;  // to data
    songSpeed[0] = SIDTUNE_SPEED_CIA_1A;
    info.formatString = text_format;
    return true;
}


const ubyte sidplayer[] =  // "Compute's Basic Sidplayer" by Craig Chamberlain.
{
   0x0,0xc0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,0xff,0xff,0xff,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x8, 0x8, 0x8,0x40,0x40,0x40,0x20,
  0x20,0x20,0xf5,0xf5,0xf5, 0x0, 0x0, 0x8, 0x8,0x90,0x60, 0xc, 0x1, 0x1, 0x1, 0x0,
   0x0, 0x0, 0x1, 0x1, 0x1, 0x0, 0x1, 0x2, 0x0, 0x0, 0x0, 0x4, 0x4, 0x4, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
  0x5b,0x5b,0x5b, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x8, 0x8,
   0x8, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6,0x98,0xec, 0x9,0x15,0x20,
   0x0, 0x1, 0x2, 0x6,0x98,0xec, 0x9,0x15,0x20, 0x0, 0x0, 0x0, 0x0, 0x5, 0xa, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,0xff, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
  0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
  0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
   0x0,0x61,0xc1,0x60, 0x1, 0x2, 0x4, 0x0, 0x7, 0xe, 0x2, 0x2,0xfe, 0x2, 0x2,0xfe,
  0xfe, 0x0, 0x1, 0x0,0xff, 0x0, 0x2, 0x4, 0x5, 0x7, 0x9, 0xb,0x1e,0x18,0x8b,0x7e,
  0xfa, 0x6,0xac,0xf3,0xe6,0x8f,0xf8,0x2e,0x86,0x8e,0x96,0x9f,0xa8,0xb3,0xbd,0xc8,
  0xd4,0xe1,0xee,0xfd,0x8c,0x78,0x64,0x50,0x3c,0x28,0x14, 0x0, 0x0, 0x2, 0x3, 0x5,
   0x7, 0x8, 0xa, 0xc, 0xd, 0xf,0x11,0x12, 0x0,0xe0, 0x0, 0x5, 0xa, 0xf,0xf9, 0x0,
  0xf5, 0x0, 0x0, 0x0,0x10, 0x0, 0x0,0x20, 0x0, 0x0,0x30, 0x0, 0x0,0x40, 0x0, 0x0,
  0x50, 0x0, 0x0,0x60, 0x0, 0x0,0x70, 0x0, 0x0,0x80, 0x0, 0x0,0x90, 0x0, 0x0,0xa0,
   0x0,0xa9, 0x0,0x8d, 0x0,0xc0,0xa2,0x95,0xa0,0x42,0xad,0xa6, 0x2,0xf0, 0x4,0xa2,
  0x25,0xa0,0x40,0x8e,0x5b,0xc1,0x8c,0x5c,0xc1,0xad,0x14, 0x3,0x8d,0x5d,0xc1,0xad,
  0x15, 0x3,0x8d,0x5e,0xc1,0x78,0xa9,0x26,0x8d,0x14, 0x3,0xa9,0xc3,0x8d,0x15, 0x3,
  0x58,0x60,0xa9, 0x0,0x8d, 0x0,0xc0,0x86,0x61,0x84,0x62,0xa0,0xbc,0x99, 0x0,0xc0,
  0x88,0xd0,0xfa,0xa0,0x72,0x99,0xbc,0xc0,0x88,0xd0,0xfa,0x8d,0x15,0xd4,0x8d,0x16,
  0xd4,0xa9, 0x8,0x8d,0x25,0xc0,0x8d,0x17,0xd4,0x8d,0x26,0xc0,0x8d,0x18,0xd4,0xa9,
  0x90,0x8d,0x27,0xc0,0xa9,0x60,0x8d,0x28,0xc0,0xa9, 0xc,0x8d,0x29,0xc0,0xad,0x5b,
  0xc1,0x8d,0x2d,0xc0,0xad,0x5c,0xc1,0x8d,0x2e,0xc0,0xa9,0xff,0x8d,0xcc,0xc0,0xa9,
  0xd4,0x85,0x64,0xa2, 0x2,0xa9,0xff,0x9d, 0xb,0xc0,0xa9, 0x1,0x9d,0x30,0xc0,0x9d,
  0x2a,0xc0,0x8a,0x9d,0x33,0xc0,0x9d,0xae,0xc0,0xa9, 0x4,0x9d,0x39,0xc0,0xbd,0xa8,
  0xc1,0x9d,0xba,0xc0,0xa9,0x5b,0x9d,0x7e,0xc0,0xbd,0x65,0xc1,0x85,0x63,0xa9, 0x0,
  0xa8,0x91,0x63,0xc8,0x91,0x63,0xc8,0x91,0x63,0xa9, 0x8,0x9d,0x17,0xc0,0x9d,0x9c,
  0xc0,0xc8,0x91,0x63,0xc8,0x91,0x63,0xa9,0x40,0x9d,0x1a,0xc0,0x91,0x63,0xa9,0x20,
  0x9d,0x1d,0xc0,0xc8,0x91,0x63,0xa9,0xf5,0x9d,0x20,0xc0,0xc8,0x91,0x63,0xca,0x10,
  0xa4,0x8a,0xa2,0x17,0x9d,0x3e,0xc1,0xca,0x10,0xfa,0xa5,0x61,0x18,0x69, 0x6,0x85,
  0x63,0xa9, 0x0,0xaa,0xa8,0x65,0x62,0x85,0x64,0x9d,0xab,0xc0,0x9d,0xb4,0xc0,0xa5,
  0x63,0x9d,0xa8,0xc0,0x9d,0xb1,0xc0,0x18,0x71,0x61,0x85,0x63,0xa5,0x64,0xc8,0x71,
  0x61,0xc8,0xe8,0xe0, 0x3,0xd0,0xe0,0xa6,0x63,0xa8,0x60,0xa9, 0x0,0x8d, 0x4,0xd4,
  0x8d, 0xb,0xd4,0x8d,0x12,0xd4,0x8d, 0x1,0xd4,0x8d, 0x8,0xd4,0x8d, 0xf,0xd4,0xa9,
   0x8,0x8d,0x17,0xd4,0xad,0x5b,0xc1,0x8d, 0x4,0xdc,0xad,0x5c,0xc1,0x8d, 0x5,0xdc,
  0x60,0x78,0xad,0x5d,0xc1,0x8d,0x14, 0x3,0xad,0x5e,0xc1,0x8d,0x15, 0x3,0x58,0x60,
  0xa9, 0x8,0x8d, 0x0,0xc0,0x4c,0x70,0xcc,0xad, 0xd,0xdc,0xad, 0x0,0xc0,0x30,0xf0,
   0x9,0x80,0xa8,0x29, 0x7,0xf0,0xee,0xd8,0x8c, 0x0,0xc0,0x58,0xa5,0xfb,0x8d,0x56,
  0xc1,0xa5,0xfc,0x8d,0x57,0xc1,0xa5,0xfd,0x8d,0x58,0xc1,0xa5,0xfe,0x8d,0x59,0xc1,
  0xa5,0xff,0x8d,0x5a,0xc1,0xad,0x23,0xc0,0x18,0x6d,0xd9,0xc0,0x48,0x29, 0x7,0xa8,
  0xad,0xdc,0xc0,0x69, 0x0,0x85,0xff,0x68,0x46,0xff,0x6a,0x46,0xff,0x6a,0x46,0xff,
  0x6a,0x18,0x6d,0x24,0xc0,0x8c,0x15,0xd4,0x8d,0x16,0xd4,0xad,0x25,0xc0,0x8d,0x17,
  0xd4,0xad,0x26,0xc0,0x8d,0x18,0xd4,0xa9,0xd4,0x85,0xfc,0xa2, 0x0,0xad, 0x0,0xc0,
  0x3d,0x62,0xc1,0xf0,0x51,0xbd,0x65,0xc1,0x85,0xfb,0xbd, 0xe,0xc0,0x18,0x7d,0x51,
  0xc0,0xa8,0xbd,0x11,0xc0,0x7d,0x54,0xc0,0x48,0x98,0x18,0x7d,0xcd,0xc0,0xa0, 0x0,
  0x91,0xfb,0x68,0x7d,0xd0,0xc0,0xc8,0x91,0xfb,0xbd,0x14,0xc0,0x18,0x7d,0x69,0xc0,
  0x85,0xff,0xbd,0x17,0xc0,0x7d,0x6c,0xc0,0x48,0xa5,0xff,0x18,0x7d,0xd3,0xc0,0xc8,
  0x91,0xfb,0x68,0x7d,0xd6,0xc0,0xc8,0x91,0xfb,0xbd,0x1d,0xc0,0xc8,0xc8,0x91,0xfb,
  0xbd,0x20,0xc0,0xc8,0x91,0xfb,0xe8,0xe0, 0x3,0xd0,0xa2,0xac,0x1a,0xc0,0xae,0x1b,
  0xc0,0xad,0x1c,0xc0,0x8c, 0x4,0xd4,0x8e, 0xb,0xd4,0x8d,0x12,0xd4,0xae,0x2d,0xc0,
  0xac,0x2e,0xc0,0x8e, 0x4,0xdc,0x8c, 0x5,0xdc,0xad,0x1b,0xd4,0x8d,0xbe,0xc0,0xad,
  0x1c,0xd4,0x8d,0xbf,0xc0,0xa2, 0x0,0xad, 0x0,0xc0,0x3d,0x62,0xc1,0xf0,0x10,0x8e,
  0x2f,0xc0,0x20,0x36,0xc5,0xad, 0x0,0xc0,0x29,0x78,0xf0, 0x3,0x4c, 0xc,0xc5,0xe8,
  0xe0, 0x3,0xd0,0xe3,0xad,0xc9,0xc0,0xd0,0x52,0xad,0xca,0xc0, 0xd,0xcb,0xc0,0xf0,
  0x78,0xad,0xdf,0xc0,0xd0,0x28,0xad,0xca,0xc0,0xf0,0x28,0x18,0x6d,0xbd,0xc0,0xb0,
   0x7,0xcd,0xcc,0xc0,0x90,0x60,0xf0,0x5e,0xa9, 0x0,0x8d,0xdf,0xc0,0xad,0xcb,0xc0,
  0xf0,0x54,0xee,0xdf,0xc0,0xad,0xbd,0xc0,0xed,0xcb,0xc0,0x4c,0xb4,0xc4,0xad,0xcb,
  0xc0,0xf0,0xd3,0xad,0xbd,0xc0,0x38,0xed,0xcb,0xc0,0xb0,0x3a,0xa9, 0x0,0x8d,0xdf,
  0xc0,0xad,0xca,0xc0,0xd0,0x30,0xee,0xdf,0xc0,0xd0,0x28,0xce,0xe0,0xc0,0xd0,0x29,
  0xad,0xdf,0xc0,0xd0,0x11,0xee,0xdf,0xc0,0xad,0xcb,0xc0,0xd0, 0x2,0xa9,0x20,0x8d,
  0xe0,0xc0,0xa9, 0x0,0xf0,0x10,0xce,0xdf,0xc0,0xad,0xca,0xc0,0xd0, 0x2,0xa9,0x20,
  0x8d,0xe0,0xc0,0xad,0xcc,0xc0,0x8d,0xbd,0xc0,0xa2, 0x0,0xbd,0xc3,0xc0,0xf0,0x44,
  0xa9, 0x0,0x85,0xff,0xbc,0xc0,0xc0,0xb9,0xbd,0xc0,0xbc,0xc6,0xc0,0xf0, 0xe,0x30,
   0x8, 0xa,0x26,0xff,0x88,0xd0,0xfa,0xf0, 0x4,0x4a,0xc8,0xd0,0xfc,0xbc,0xc3,0xc0,
  0x88,0xd0, 0xb,0x9d,0xcd,0xc0,0xa5,0xff,0x9d,0xd0,0xc0,0x4c, 0x2,0xc5,0x88,0xd0,
   0xb,0x9d,0xd3,0xc0,0xa5,0xff,0x9d,0xd6,0xc0,0x4c, 0x2,0xc5,0x8d,0xd9,0xc0,0xa5,
  0xff,0x8d,0xdc,0xc0,0xe8,0xe0, 0x3,0xd0,0xb2,0xad, 0x0,0xc0,0x29,0x7f,0x8d, 0x0,
  0xc0,0xad,0x56,0xc1,0x85,0xfb,0xad,0x57,0xc1,0x85,0xfc,0xad,0x58,0xc1,0x85,0xfd,
  0xad,0x59,0xc1,0x85,0xfe,0xad,0x5a,0xc1,0x85,0xff,0x4c,0x70,0xcc,0xbd,0x60,0xc0,
  0xd0, 0x3,0x4c,0x9f,0xc6,0x4c,0xba,0xc5,0xde,0x30,0xc0,0xd0, 0x3,0x4c,0xa0,0xc6,
  0xbd,0x36,0xc0,0x30,0xe8,0xd0,0x1a,0xbd,0x3f,0xc0,0xf0, 0x5,0xde,0x3f,0xc0,0xd0,
  0x10,0xbd,0x39,0xc0,0xdd,0x30,0xc0,0x90, 0x8,0xbd,0x1a,0xc0,0x29,0xfe,0x9d,0x1a,
  0xc0,0xbd,0x42,0xc0,0xf0,0x56, 0xa,0xbd, 0xe,0xc0,0xb0,0x1d,0x7d,0x45,0xc0,0x9d,
   0xe,0xc0,0xa8,0xbd,0x11,0xc0,0x7d,0x48,0xc0,0x9d,0x11,0xc0,0x48,0x98,0xdd,0x8d,
  0xc0,0x68,0xfd,0x90,0xc0,0xb0,0x1f,0x90,0x2e,0xfd,0x45,0xc0,0x9d, 0xe,0xc0,0xbd,
  0x11,0xc0,0xfd,0x48,0xc0,0x9d,0x11,0xc0,0xbd,0x8d,0xc0,0xdd, 0xe,0xc0,0xbd,0x90,
  0xc0,0xfd,0x11,0xc0,0x90,0x11,0xbd,0x8d,0xc0,0x9d, 0xe,0xc0,0xbd,0x90,0xc0,0x9d,
  0x11,0xc0,0xa9, 0x0,0x9d,0x42,0xc0,0xbd,0x60,0xc0,0xf0,0x55,0xbd,0x4b,0xc0,0xf0,
  0x4b,0xa0, 0x0,0xde,0x4e,0xc0,0xd0,0x31,0xbd,0x51,0xc0,0x1d,0x54,0xc0,0xd0,0x1b,
  0xbd,0x5d,0xc0,0x9d,0x57,0xc0,0x9d,0x4e,0xc0,0xbd,0x4b,0xc0, 0xa,0xbd,0x5a,0xc0,
  0x90, 0x4,0x49,0xff,0x69, 0x0,0x9d,0x4b,0xc0,0xd0,0x10,0xbd,0x57,0xc0,0x9d,0x4e,
  0xc0,0x98,0x38,0xfd,0x4b,0xc0,0x9d,0x4b,0xc0,0xc9, 0x0,0x10, 0x1,0x88,0x18,0x7d,
  0x51,0xc0,0x9d,0x51,0xc0,0x98,0x7d,0x54,0xc0,0x9d,0x54,0xc0,0xbd,0x36,0xc0,0x30,
  0x15,0xbd,0x93,0xc0,0xf0,0x10,0x18,0x7d,0x14,0xc0,0x9d,0x14,0xc0,0xbd,0x96,0xc0,
  0x7d,0x17,0xc0,0x9d,0x17,0xc0,0xbd,0x63,0xc0,0xf0,0x4b,0xa0, 0x0,0xde,0x66,0xc0,
  0xd0,0x31,0xbd,0x69,0xc0,0x1d,0x6c,0xc0,0xd0,0x1b,0xbd,0x72,0xc0,0x9d,0x6f,0xc0,
  0x9d,0x66,0xc0,0xbd,0x63,0xc0, 0xa,0xbd,0x75,0xc0,0x90, 0x4,0x49,0xff,0x69, 0x0,
  0x9d,0x63,0xc0,0xd0,0x10,0xbd,0x6f,0xc0,0x9d,0x66,0xc0,0x98,0x38,0xfd,0x63,0xc0,
  0x9d,0x63,0xc0,0xc9, 0x0,0x10, 0x1,0x88,0x18,0x7d,0x69,0xc0,0x9d,0x69,0xc0,0x98,
  0x7d,0x6c,0xc0,0x9d,0x6c,0xc0,0xbd,0x36,0xc0,0x10, 0x3,0x4c,0x9f,0xc6,0xa0, 0x0,
  0xbd,0xa2,0xc0,0xf0,0x1c,0x10, 0x1,0xc8,0x18,0x6d,0x23,0xc0,0x48,0x29, 0x7,0x8d,
  0x23,0xc0,0x68,0x6a,0x4a,0x4a,0x18,0x79,0xa6,0xc1,0x18,0x6d,0x24,0xc0,0x8d,0x24,
  0xc0,0x60,0xbd,0xa8,0xc0,0x85,0xfd,0xbd,0xab,0xc0,0x85,0xfe,0xd0, 0x4,0x60,0x20,
  0x98,0xc8,0xad, 0x0,0xc0,0x3d,0x62,0xc1,0xf0,0xf4,0xa0, 0x0,0xb1,0xfd,0x85,0xff,
  0xc8,0xb1,0xfd,0xa8,0xa5,0xfd,0x18,0x69, 0x2,0x85,0xfd,0x9d,0xa8,0xc0,0xa5,0xfe,
  0x69, 0x0,0x85,0xfe,0x9d,0xab,0xc0,0xa5,0xff,0x29, 0x3,0xd0,0xd2,0xbd,0x8d,0xc0,
  0x9d, 0xe,0xc0,0xbd,0x90,0xc0,0x9d,0x11,0xc0,0xa5,0xff,0x9d, 0x5,0xc0,0x98,0x9d,
   0x2,0xc0,0x29, 0x7,0xa8,0xb9,0x67,0xc1,0x8d,0x6f,0xc1,0xbd, 0x2,0xc0,0x29,0x38,
  0x4a,0x4a,0x4a,0x7d,0x81,0xc0,0x85,0xfd,0xbd, 0x2,0xc0,0x29,0xc0, 0xa,0x2a,0x2a,
  0xa8,0xb9,0x6f,0xc1,0x85,0xfe,0xbd, 0x2,0xc0,0x29, 0x7,0xf0,0x62,0xa8,0xb9,0x72,
  0xc1,0x65,0xfe,0x18,0x7d,0x84,0xc0,0x10, 0x5,0x18,0x69, 0xc,0xe6,0xfd,0xc9, 0xc,
  0x90, 0x4,0xe9, 0xc,0xc6,0xfd,0x85,0xfe,0xa8,0xb9,0x86,0xc1,0x85,0xff,0xb9,0x7a,
  0xc1,0xa4,0xfd,0x88,0x30, 0x6,0x46,0xff,0x6a,0x88,0x10,0xfa,0x18,0x7d,0x87,0xc0,
  0x9d,0x8d,0xc0,0xa5,0xff,0x7d,0x8a,0xc0,0x9d,0x90,0xc0,0xbd, 0x5,0xc0,0xd0, 0x3,
  0x4c,0xa0,0xc6,0xbd,0x45,0xc0,0x1d,0x48,0xc0,0xf0,0x16,0xbd, 0xe,0xc0,0xdd,0x8d,
  0xc0,0xbd,0x11,0xc0,0xfd,0x90,0xc0,0xa9,0xfe,0x6a,0x9d,0x42,0xc0,0x90,0x11,0xf0,
  0x4a,0x9d,0x42,0xc0,0xbd,0x8d,0xc0,0x9d, 0xe,0xc0,0xbd,0x90,0xc0,0x9d,0x11,0xc0,
  0xbd,0x36,0xc0, 0xa,0xd0,0x35,0xbd,0x93,0xc0,0xf0, 0xc,0xbd,0x99,0xc0,0x9d,0x14,
  0xc0,0xbd,0x9c,0xc0,0x9d,0x17,0xc0,0xbd,0x9f,0xc0,0xf0, 0xf,0xa4,0xfd,0x18,0x79,
  0x92,0xc1,0xa4,0xfe,0x18,0x79,0x9a,0xc1,0x18,0x90, 0x8,0xbd,0xa2,0xc0,0xf0, 0xb,
  0xbd,0xa5,0xc0,0x8d,0x24,0xc0,0xa9, 0x0,0x8d,0x23,0xc0,0xbd,0x3c,0xc0,0x9d,0x3f,
  0xc0,0xbd, 0x5,0xc0,0x29,0x40,0x9d,0x36,0xc0,0xbd, 0x5,0xc0,0x4a,0x4a,0x29, 0x7,
  0xd0,0x30,0xbd, 0x5,0xc0,0x30,0x14,0xad,0x27,0xc0,0x29,0x3c,0xd0,0x1e,0xad,0x27,
  0xc0, 0xa,0x2a,0x2a,0xd0, 0x2,0xa9, 0x4,0x4c,0x70,0xc8,0xad,0x28,0xc0,0xf0, 0xc,
  0x29,0x3f,0xd0, 0x8,0xad,0x28,0xc0, 0xa,0x2a,0x2a,0xd0,0x66,0xa9,0x10,0x8d, 0x0,
  0xc0,0x60,0xc9, 0x1,0xd0,0x13,0xbd, 0x5,0xc0,0x29,0x20,0xd0, 0x6,0xad,0x29,0xc0,
  0x4c,0x70,0xc8,0xbd,0x2a,0xc0,0x4c,0x70,0xc8,0xa8,0xbd, 0x5,0xc0,0x29,0xa0,0xc9,
  0x80,0xf0,0x30,0x85,0xff,0x18,0xad,0x27,0xc0,0xd0, 0x1,0x38,0x88,0x88,0xf0, 0x6,
  0x6a,0xb0,0x4e,0x88,0xd0,0xfa,0xa4,0xff,0x85,0xff,0xf0,0x26,0x46,0xff,0xb0,0x41,
  0xf0,0x42,0x65,0xff,0xb0,0x3e,0xc8,0x10,0x19,0x46,0xff,0xb0,0x34,0x65,0xff,0x90,
  0x11,0xb0,0x31,0xad,0x28,0xc0,0xf0,0x29,0x88,0x88,0xf0, 0x6,0x4a,0xb0,0x22,0x88,
  0xd0,0xfa,0x9d,0x30,0xc0,0xbd,0x1a,0xc0,0x29,0xf6,0x9d,0x1a,0xc0,0x38,0xbd, 0x2,
  0xc0,0x29, 0x7,0xd0, 0x3,0x7e,0x36,0xc0,0xbd,0x1a,0xc0,0x69, 0x0,0x9d,0x1a,0xc0,
  0x60,0xa9,0x10,0x2c,0xa9,0x18,0x8d, 0x0,0xc0,0x60,0x98,0x48,0xa5,0xff,0x4a,0x90,
   0x3,0x4c,0x42,0xca,0x4a,0x4a,0xb0,0x1e,0x4a,0xb0, 0xe,0x9d,0x9c,0xc0,0x9d,0x17,
  0xc0,0x68,0x9d,0x99,0xc0,0x9d,0x14,0xc0,0x60,0x4a,0x90, 0x2, 0x9,0xf8,0x9d,0x8a,
  0xc0,0x68,0x9d,0x87,0xc0,0x60,0x4a,0xb0, 0x3,0x4c,0x4a,0xc9,0x4a,0xb0,0x61,0x4a,
  0xb0, 0xf,0xd0, 0x8,0x68,0x9d,0xa5,0xc0,0x8d,0x24,0xc0,0x60,0x68,0x9d,0x3c,0xc0,
  0x60,0xd0,0x48,0x68,0x9d,0x7e,0xc0,0xc9,0x5b,0xf0,0x33,0xa8,0x4a,0x4a,0x4a,0x38,
  0xe9, 0xb,0x18,0x7d,0x84,0xc0,0x30, 0xc,0xc9, 0xc,0x90,0x11,0xe9, 0xc,0xde,0x81,
  0xc0,0x4c, 0xb,0xc9,0xc9,0xf5,0xb0, 0x5,0x69, 0xc,0xfe,0x81,0xc0,0x9d,0x84,0xc0,
  0x98,0x29, 0x7,0x38,0xe9, 0x3,0x18,0x7d,0x81,0xc0,0x9d,0x81,0xc0,0x60,0xbd,0x78,
  0xc0,0x9d,0x81,0xc0,0xbd,0x7b,0xc0,0x9d,0x84,0xc0,0x60,0x68,0x9d,0xc6,0xc0,0x60,
  0x4a,0xb0, 0x8,0x9d, 0xb,0xc0,0x68,0x9d, 0x8,0xc0,0x60,0x4a,0x6a,0x6a,0x6d,0x5b,
  0xc1,0x8d,0x2d,0xc0,0x68,0x6d,0x5c,0xc1,0x8d,0x2e,0xc0,0x60,0x4a,0x90, 0x3,0x4c,
  0xd3,0xc9,0x4a,0xb0,0x40,0x4a,0xb0,0x17,0x4a,0xb0, 0xf,0x68,0x8d,0x27,0xc0,0x4a,
  0x4a,0x4a,0xa8,0xb9,0xaf,0xc1,0x8d,0x28,0xc0,0x60,0x68,0x9d,0x5d,0xc0,0x60,0x4a,
  0xb0, 0x5,0x68,0x8d, 0x1,0xc0,0x60,0x68,0xf0,0x11,0x9d,0x75,0xc0,0xbc,0x63,0xc0,
  0xd0, 0x8,0x9d,0x63,0xc0,0xa9, 0x1,0x9d,0x66,0xc0,0x60,0x9d,0x63,0xc0,0x9d,0x69,
  0xc0,0x9d,0x6c,0xc0,0x60,0x4a,0xb0,0x30,0x4a,0xb0, 0x5,0x68,0x9d,0x39,0xc0,0x60,
  0x68,0xa0, 0x0,0x4a,0x90, 0x2,0xc8,0x18,0x48,0x29, 0x7,0x79,0xac,0xc1,0x9d,0x78,
  0xc0,0x9d,0x81,0xc0,0x68,0x4a,0x4a,0x4a,0x18,0x79,0xad,0xc1,0x9d,0x7b,0xc0,0x9d,
  0x84,0xc0,0xa9,0x5b,0x9d,0x7e,0xc0,0x60,0x4a,0xb0, 0x5,0x68,0x9d,0xa2,0xc0,0x60,
  0x68,0x8d,0xcc,0xc0,0x60,0x4a,0xb0,0x27,0x4a,0xb0, 0xd,0x4a,0xb0, 0x5,0x68,0x8d,
  0x29,0xc0,0x60,0x68,0x9d,0x9f,0xc0,0x60,0x4a,0xb0, 0xf,0x68,0x9d,0x93,0xc0,0xa0,
   0x0, 0xa,0x90, 0x1,0x88,0x98,0x9d,0x96,0xc0,0x60,0x68,0x9d,0x72,0xc0,0x60,0x4a,
  0xb0,0x1c,0x4a,0xb0,0x15,0x68,0x9d,0xb7,0xc0,0xa5,0xfd,0x9d,0xb1,0xc0,0xa5,0xfe,
  0x9d,0xb4,0xc0,0xbd,0x33,0xc0,0x9d,0xae,0xc0,0x60,0x68,0x4c,0x70,0xcc,0x4a,0xb0,
  0x1e,0x68,0xd0, 0xa,0x9d,0x4b,0xc0,0x9d,0x51,0xc0,0x9d,0x54,0xc0,0x60,0x9d,0x5a,
  0xc0,0xbc,0x4b,0xc0,0xd0, 0x8,0x9d,0x4b,0xc0,0xa9, 0x1,0x9d,0x4e,0xc0,0x60,0x68,
  0x9d,0x2a,0xc0,0x60,0x4a,0x90, 0x8,0x9d,0x48,0xc0,0x68,0x9d,0x45,0xc0,0x60,0x68,
  0x4a,0xb0,0x61,0x4a,0xb0,0x25,0x4a,0xb0, 0x5,0x4a,0xa0,0xf0,0xd0, 0x6, 0xa, 0xa,
   0xa, 0xa,0xa0, 0xf,0x85,0xff,0x98,0xb0, 0x9,0x3d,0x1d,0xc0, 0x5,0xff,0x9d,0x1d,
  0xc0,0x60,0x3d,0x20,0xc0, 0x5,0xff,0x9d,0x20,0xc0,0x60,0x4a,0xb0,0x38,0x4a,0xb0,
  0x64,0x85,0xff,0xbd,0xba,0xc0,0xdd,0xa9,0xc1,0xf0,0x54,0xfe,0xba,0xc0,0xa8,0xa5,
  0xfd,0x99,0xe1,0xc0,0xa5,0xfe,0x99,0xf0,0xc0,0xbd,0x33,0xc0,0x99,0x2f,0xc1,0xa4,
  0xff,0xb9,0x17,0xc1,0xf0,0x36,0x85,0xfe,0xb9,0xff,0xc0,0x85,0xfd,0xb9,0x3e,0xc1,
  0x9d,0x33,0xc0,0x60,0xb0,0x4b,0x4a,0xb0,0x3c,0xa8,0xa5,0xfd,0x99,0xff,0xc0,0xa5,
  0xfe,0x99,0x17,0xc1,0xbd,0x33,0xc0,0x99,0x3e,0xc1,0xbd,0xba,0xc0,0xdd,0xa9,0xc1,
  0xf0, 0xd,0xfe,0xba,0xc0,0xa8,0xa9, 0x0,0x99,0xf0,0xc0,0x60,0xa9,0x30,0x2c,0xa9,
  0x28,0x8d, 0x0,0xc0,0x60, 0xa, 0xa, 0xa, 0xa,0x4d,0x25,0xc0,0x29,0xf0,0x4d,0x25,
  0xc0,0x8d,0x25,0xc0,0x60,0x4d,0x26,0xc0,0x29, 0xf,0x4d,0x26,0xc0,0x8d,0x26,0xc0,
  0x60,0x4a,0xb0, 0xb,0x4a,0xb0, 0x4,0x8d,0xca,0xc0,0x60,0x8d,0xcb,0xc0,0x60,0x4a,
  0x90, 0x3,0x4c,0xa5,0xcb,0x4a,0xa8,0xf0,0x21,0x88,0xf0,0x34,0x88,0xf0,0x42,0x88,
  0xf0,0x4a,0x88,0xf0,0x52,0x88,0xf0,0x5c,0x88,0xf0,0x66,0x88,0xf0,0x73,0x29, 0x7,
   0x9,0x10,0xb0, 0x3,0x4c,0xb7,0xca,0x4c,0x7f,0xca,0xac,0x26,0xc0,0xb0, 0x7,0xc8,
  0x98,0x29, 0xf,0xd0, 0x7,0x60,0x98,0x29, 0xf,0xf0, 0x4,0x88,0x8c,0x26,0xc0,0x60,
  0xbd,0x62,0xc1,0x49,0xff,0x2d,0x25,0xc0,0x90, 0x3,0x1d,0x62,0xc1,0x8d,0x25,0xc0,
  0x60,0xbd,0x1a,0xc0,0x29,0xfb,0x90,0x55, 0x9, 0x4,0xb0,0x51,0xbd,0x1a,0xc0,0x29,
  0xfd,0x90,0x4a, 0x9, 0x2,0xb0,0x46,0xad,0x25,0xc0,0x29,0xf7,0x90, 0x2, 0x9, 0x8,
  0x8d,0x25,0xc0,0x60,0xad,0x26,0xc0,0x29,0x7f,0x90, 0x2, 0x9,0x80,0x8d,0x26,0xc0,
  0x60,0x98,0x8d,0xbd,0xc0,0x8d,0xdf,0xc0,0xc8,0x8c,0xe0,0xc0,0x2a,0x8d,0xc9,0xc0,
  0x60,0x98,0x2a,0x9d,0x60,0xc0,0x60,0x4a,0xb0,0x27,0x4a,0xb0,0x14,0xd0, 0x2,0xa9,
   0x8, 0xa, 0xa, 0xa, 0xa,0x5d,0x1a,0xc0,0x29,0xf0,0x5d,0x1a,0xc0,0x9d,0x1a,0xc0,
  0x60, 0xa, 0xa, 0xa, 0xa,0x4d,0x26,0xc0,0x29,0x70,0x4d,0x26,0xc0,0x8d,0x26,0xc0,
  0x60,0x4a,0x90, 0x4,0x9d,0xc0,0xc0,0x60,0xa8,0xf0,0x20,0x88,0xf0,0x40,0x88,0xf0,
  0x63,0x29, 0x3,0x9d,0xc3,0xc0,0xa9, 0x0,0x9d,0xcd,0xc0,0x9d,0xd0,0xc0,0x9d,0xd3,
  0xc0,0x9d,0xd6,0xc0,0x8d,0xd9,0xc0,0x8d,0xdc,0xc0,0x60,0xbd,0xb7,0xc0,0xf0, 0x5,
  0xde,0xb7,0xc0,0xf0,0x12,0xbd,0x33,0xc0,0xdd,0xae,0xc0,0xd0, 0xb,0xbd,0xb1,0xc0,
  0x85,0xfd,0xbd,0xb4,0xc0,0x85,0xfe,0x60,0xa9,0x38,0x8d, 0x0,0xc0,0x60,0xbd,0xba,
  0xc0,0xdd,0xa8,0xc1,0xf0,0x18,0xde,0xba,0xc0,0xa8,0x88,0xb9,0xf0,0xc0,0xf0, 0xd,
  0x85,0xfe,0xb9,0xe1,0xc0,0x85,0xfd,0xb9,0x2f,0xc1,0x9d,0x33,0xc0,0x60,0xa9,0x20,
  0x8d, 0x0,0xc0,0x60,0xad, 0x0,0xc0,0x5d,0x62,0xc1,0x8d, 0x0,0xc0,0xa9, 0x1,0x9d,
  0x30,0xc0,0x60,0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea, 0x0,0xea,0xea,0xea,0xea,
  0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea,0xea,
  0xea,0xea,0xad,0x59,0xcc,0xc9, 0x7,0xf0, 0x9,0xad, 0x0,0xc0,0x29, 0x7,0xc9, 0x0,
  0xf0, 0x3,0x6c,0x5d,0xc1,0x20,0xe9,0xc2,0x20, 0xf,0xc3,0x4c,0x80,0xcc, 0x0, 0x0,
   0x0, 0x0,0xa9, 0x0,0x8d, 0x0,0xc0,0xa9, 0x7,0x8d,0x59,0xcc,0x20,0xcf,0xc1,0xa2,
   0x0,0xa0, 0x9,0x20, 0x0,0xc2,0xa9, 0x7,0x8d, 0x0,0xc0,0xa9, 0x0,0x8d,0x59,0xc0,
  0x60
};


void sidTune::MUS_installPlayer(ubyte *c64buf)
{
    if (status && (c64buf != 0))
    {
        uword dest = readEndian(sidplayer[1],sidplayer[0]);
        for ( uword i = 2; i < sizeof(sidplayer); i++ )
        {
            c64buf[dest+i-2] = sidplayer[i];
        }
        info.loadAddr = 0x0900;
        info.initAddr = 0xcc90;
        info.playAddr = 0;
    }
}
