/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.eviction;

import org.jboss.cache.config.ConfigurationException;
import org.jboss.cache.config.Dynamic;

/**
 * Configuration for {@link ElementSizePolicy}.
 * <p/>
 * If configured via XML, expects the following:
 * <p/>
 * <pre>
 * <region name="/region/">
 *    <attribute name="maxElementsPerNode">100</attribute>
 *    <attribute name="maxNodes">10000</attribute>
 * </region>
 * </pre>
 * <p/>
 * Requires a positive "maxElementsPerNode" value otherwise a ConfigurationException is thrown.
 *
 * @author Daniel Huang
 * @author Brian Stansberry
 * @version $Revision: 6120 $
 */
public class ElementSizeConfiguration extends EvictionPolicyConfigBase
{
   /**
    * The serialVersionUID
    */
   private static final long serialVersionUID = 2510593544656833758L;

   @Dynamic
   private int maxElementsPerNode;

   public ElementSizeConfiguration()
   {
      super();
      // Force configuration of maxElementsPerNode
      setMaxElementsPerNode(-1);
   }

   @Override
   protected void setEvictionPolicyClassName()
   {
      setEvictionPolicyClass(ElementSizePolicy.class.getName());
   }

   public int getMaxElementsPerNode()
   {
      return maxElementsPerNode;
   }

   public void setMaxElementsPerNode(int maxElementsPerNode)
   {
      testImmutability("maxElementsPerNode");
      this.maxElementsPerNode = maxElementsPerNode;
   }

   /**
    * Requires a positive maxElementsPerNode value or ConfigurationException
    * is thrown.
    */
   @Override
   public void validate() throws ConfigurationException
   {
      if (maxElementsPerNode < 0)
      {
         throw new ConfigurationException("maxElementsPerNode must be must be " +
               "configured to a value greater than or equal to 0");
      }
   }

   @Override
   public String toString()
   {
      StringBuilder str = new StringBuilder();
      str.append("ElementSizeConfiguration: maxElementsPerNode =");
      str.append(getMaxElementsPerNode()).append(" maxNodes =").append(getMaxNodes());
      return str.toString();
   }

   @Override
   public boolean equals(Object obj)
   {
      if (this == obj)
         return true;
      if (super.equals(obj))
      {
         return this.maxElementsPerNode == ((ElementSizeConfiguration) obj).maxElementsPerNode;
      }
      return false;
   }

   @Override
   public int hashCode()
   {
      int result = super.hashCode();
      result = 31 * result + maxElementsPerNode;
      return result;
   }

   @Override
   public void reset()
   {
      super.reset();
      setMaxElementsPerNode(-1);
   }

   @Override
   public ElementSizeConfiguration clone() throws CloneNotSupportedException
   {
      return (ElementSizeConfiguration) super.clone();
   }


}
