/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.cache;

import java.util.Set;

import org.jboss.cache.config.Configuration;
import org.jgroups.ChannelFactory;

/**
 * Factory and registry for JBoss Cache instances configured using
 * named configurations.
 * 
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 1 $
 */
public interface CacheManager {

    /**
     * Gets the names of all the configurations of which this object
     * is aware.
     * 
     * @return a set of names. Will not be <code>null</code>.
     */
    Set<String> getConfigurationNames();
    
    /**
     * Gets the name of all caches that are registered, i.e. that can be
     * by a call to {@link #getCache(String, boolean) getCache(name, false)}.
     * 
     * @return a set of names. Will not be <code>null</code>.
     */
    Set<String> getCacheNames();
    
    /**
     * Gets the JGroups <code>ChannelFactory</code> that will be injected
     * into any {@link Configuration} that has a 
     * {@link Configuration#getMultiplexerStack() multiplexer stack}
     * configured.
     * 
     * @return
     */
    ChannelFactory getChannelFactory();
    
    /**
     * Get a cache configured according to the given configuration name, 
     * optionally instantiating the cache if it hasn't already been 
     * instantiated.
     * <p>
     * The caller is free to invoke the {@link Cache#create()} and
     * {@link Cache#start()} lifecycle methods on the returned cache, but
     * the @link Cache#stop()} and {@link Cache#destroy()} methods should not
     * be invoked, since it is quite possible other users are still using the 
     * cache. Use {@link #releaseCache(String)} to notify this 
     * registry that the caller is no longer using a cache; let the registry 
     * control stopping and destroying the cache.
     * </p>
     * <p>
     * If invoking this method leads to the instantiation of the cache,
     * <code>create()</code> and <code>start()</code> will not be invoked
     * on the cache before it is returned.
     * </p>
     * 
     * @param configName    the name of the configuration
     * @param create        should the cache be instantiated if it
     *                      hasn't already been?
     * @return              the cache, or <code>null</code> if 
     *                      <code>create</code> is false and the cache hasn't
     *                      been created previously.
     *                      
     * @throws IllegalArgumentException if this object is unaware of 
     *                                  <code>configName</code>; i.e. the set
     *                                  returned from {@link #getConfigurationNames()}
     *                                  does not include <code>configName</code>
     * @throws Exception if there is a problem instantiating the cache
     */
    Cache<Object, Object> getCache(String configName, boolean create) throws Exception;
    
    /**
     * Notifies the registry that the caller is no longer using the given
     * cache.  The registry may perform cleanup operations, such as 
     * stopping and destroying the cache.
     * 
     * @param configName
     */
    void releaseCache(String configName);

}