/*
 *  @(#)StateImpl.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.mbtf.v1.engine;


import net.sourceforge.groboutils.mbtf.v1.IState;
import net.sourceforge.groboutils.mbtf.v1.IValidate;
import net.sourceforge.groboutils.mbtf.v1.ITransition;


/**
 * Immutable IState implementation.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:52:26 $
 * @since      June 12, 2002
 */
public class StateImpl implements IState
{
    private String name;
    private ITransition[] trans;
    private IValidate[] validate;
    
    
    public StateImpl( String name, ITransition[] t, IValidate[] v )
    {
        if (name == null)
        {
            throw new IllegalArgumentException("no null name");
        }
        
        if (t == null)
        {
            t = new ITransition[0];
        }
        else
        {
            int len = t.length;
            ITransition[] tt = new ITransition[ len ];
            for (int i = len; --i >= 0;)
            {
                if (t[i] == null)
                {
                    throw new IllegalArgumentException(
                        "no nulls allowed in ITransition array (index "+i+")");
                }
                // else
                tt[i] = t[i];
            }
            t = tt;
        }
        
        if (v == null)
        {
            v = new IValidate[0];
        }
        else
        {
            int len = v.length;
            IValidate[] vv = new IValidate[ len ];
            for (int i = len; --i >= 0;)
            {
                if (v[i] == null)
                {
                    throw new IllegalArgumentException(
                        "no nulls allowed in IValidate array (index "+i+")");
                }
                // else
                vv[i] = v[i];
            }
            v = vv;
        }
        
        this.name = name;
        this.trans = t;
        this.validate = v;
    }


    /**
     * Retrieves the name for the state.  This should be unique for
     * state-machine assembly purposes, but it does not have to be.
     *
     * @return a non-null name for this state.
     */
    public String getName()
    {
        return this.name;
    }
    
    
    /**
     * Retrieves the list of transitions leading from this state.
     *
     * @return a non-null list of all transitions from this state.  If the
     *      length of the list is 0, then this is a terminal state.
     */
    public ITransition[] getTransitions()
    {
        int len = this.trans.length;
        ITransition t[] = new ITransition[ len ];
        System.arraycopy( this.trans, 0, t, 0, len );
        return t;
    }
    
    
    /**
     * Retrieves the list of all validation instances used to assert that
     * the current state is valid.
     *
     * @return a non-null list of all validation instances for this state.
     *      An empty list will cause a warning on all but the first
     *      (initial) state.  This list should not include the included
     *      Transitions' validates.
     */
    public IValidate[] getValidates()
    {
        int len = this.validate.length;
        IValidate v[] = new IValidate[ len ];
        System.arraycopy( this.validate, 0, v, 0, len );
        return v;
    }
    
    
    public String toString()
    {
        return "[State "+getName()+"]";
    }
}

