/*
 * @(#)ModifiedClassUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.compiler;

import java.io.IOException;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.BytecodeLoaderUtil;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.datastore.AnalysisModuleSet;
import net.sourceforge.groboutils.codecoverage.v2.datastore.ClassRecord;


/**
 * Tests the ModifiedClass class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 13, 2003
 */
public class ModifiedClassUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = ModifiedClassUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public ModifiedClassUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstructor1() throws Exception
    {
        try
        {
            new ModifiedClass( null, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor2() throws Exception
    {
        try
        {
            new ModifiedClass( "", null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor3() throws Exception
    {
        try
        {
            new ModifiedClass( null, new byte[0] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor4() throws Exception
    {
        try
        {
            new ModifiedClass( null, null, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor5() throws Exception
    {
        try
        {
            new ModifiedClass( null, "", null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor6() throws Exception
    {
        try
        {
            new ModifiedClass( null, null, new byte[0] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor7() throws Exception
    {
        try
        {
            new ModifiedClass( new ParseCoverageLogger(), null, null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor8() throws Exception
    {
        try
        {
            new ModifiedClass( new ParseCoverageLogger(), "", null );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor9() throws Exception
    {
        try
        {
            new ModifiedClass( new ParseCoverageLogger(), null, new byte[0] );
            fail( "Did not throw IllegalArgumentException." );
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor10() throws Exception
    {
        try
        {
            new ModifiedClass( new ParseCoverageLogger(), "", new byte[0] );
            fail( "Did not throw an exception." );
        }
        catch (IOException io)
        {
            // test exception
        }
    }
    
    
    public void testConstructor11() throws Exception
    {
        String filename = BytecodeLoaderUtil.getClassFilename( THIS_CLASS );
        new ModifiedClass( filename,
            BytecodeLoaderUtil.loadBytecode( filename ) );
    }
    
    
    public void testConstructor12() throws Exception
    {
        String filename = BytecodeLoaderUtil.getClassFilename( THIS_CLASS );
        new ModifiedClass( new ParseCoverageLogger(), filename,
            BytecodeLoaderUtil.loadBytecode( filename ) );
    }
    
    
    public void testGetClassName1() throws Exception
    {
        ModifiedClass mc = createModifiedClass( THIS_CLASS );
        assertEquals(
            "Created class and returned classname don't match.",
            THIS_CLASS.getName(),
            mc.getClassName() );
    }
    
    
    public void testGetClassCRC1() throws Exception
    {
        ModifiedClass mc = createModifiedClass( THIS_CLASS );
        assertTrue(
            "Checksum is zero or -1.",
            mc.getClassCRC() != 0L && mc.getClassCRC() != -1L );
    }
    
    
    public void testGetClassSignature1() throws Exception
    {
        ModifiedClass mc = createModifiedClass( THIS_CLASS );
        long crc = mc.getClassCRC();
        String name = THIS_CLASS.getName();
        assertTrue(
            "Does not contain the class name.",
            mc.getClassSignature().indexOf( name ) >= 0 );
        assertTrue(
            "Does not contain the class CRC.",
            mc.getClassSignature().indexOf( Long.toString( crc ) ) >= 0 );
    }
    
    
    public void testCreateClassRecord1() throws Exception
    {
        ModifiedClass mc = createModifiedClass( THIS_CLASS );
        ClassRecord cr = mc.createClassRecord( createAnalysisModuleSet() );
        assertNotNull(
            "Returned null class record.",
            cr );
        assertEquals(
            "Mismatch class name.",
            THIS_CLASS.getName(),
            cr.getClassName() );
        assertEquals(
            "Mismatch class CRC.",
            mc.getClassCRC(),
            cr.getClassCRC() );
        assertEquals(
            "Mismatch signature.",
            mc.getClassSignature(),
            cr.getClassSignature() );
        assertEquals(
            "Mismatch method count.",
            mc.getMethods().length,
            cr.getMethodCount() );
    }
    
    
    public void testGetMethods1() throws Exception
    {
        ModifiedClass mc = createModifiedClass( THIS_CLASS );
        ModifiedMethod[] mmL = mc.getMethods();
        assertNotNull(
            "Returned null method list.",
            mmL );
        /* never could get this to work
        java.lang.reflect.Method mL[] = THIS_CLASS.getDeclaredMethods();
        assertTrue(
            "Returned incorrect list size (found "+mmL.length+
                ", expected close to "+mL.length+").",
            mmL.length == mL.length || mmL.length == mL.length + 1 );
        */
    }
    
    
    public void testGetModifiedClass1() throws Exception
    {
        ModifiedClass mc = createModifiedClass( THIS_CLASS );
        byte[] orig = BytecodeLoaderUtil.loadBytecode(
            BytecodeLoaderUtil.getClassFilename( THIS_CLASS ) );
        byte[] c = mc.getModifiedClass();
        assertNotNull(
            "Returned null class bytecode.",
            c );
        assertTrue(
            "Returned smaller length bytecode than the original.",
            c.length >= orig.length );
    }
    
    
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected ModifiedClass createModifiedClass( Class c )
            throws IOException
    {
        return CCCreatorUtil.createModifiedClass( c );
    }
    
    protected AnalysisModuleSet createAnalysisModuleSet()
    {
        return CCCreatorUtil.createAnalysisModuleSet( 2 );
    }
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

