/*--------------------------------------------------------------------*/
/*--- ALBERTA:  an Adaptive multi Level finite element toolbox using -*/
/*---           Bisectioning refinement and Error control by Residual */
/*---           Techniques for scientific Applications             ---*/
/*---                                                              ---*/
/*--- file: lagrange_3_1d.c                                        ---*/
/*---                                                              ---*/
/*--- description: implementation of the basis functions           ---*/
/*---              lagrange3 in 1d                                 ---*/
/*---                                                              ---*/
/*--- created by: kgs on host enigma                               ---*/
/*---           at 16:50 on 28 of March 2003                       ---*/
/*--------------------------------------------------------------------*/
/*---                                                              ---*/
/*--- authors:   Alfred Schmidt                                    ---*/
/*---            Zentrum fuer Technomathematik                     ---*/
/*---            Fachbereich 3 Mathematik/Informatik               ---*/
/*---            Universitaet Bremen                               ---*/
/*---            Bibliothekstr. 2                                  ---*/
/*---            D-28359 Bremen, Germany                           ---*/
/*---                                                              ---*/
/*---            Kunibert G. Siebert                               ---*/
/*---            Institut fuer Mathematik                          ---*/
/*---            Universitaet Augsburg                             ---*/
/*---            Universitaetsstr. 14                              ---*/
/*---            D-86159 Augsburg, Germany                         ---*/
/*---                                                              ---*/
/*--- http://www.mathematik.uni-freiburg.de/IAM/ALBERTA            ---*/
/*---                                                              ---*/
/*--- (c) by A. Schmidt and K.G. Siebert (1996-2003)               ---*/
/*---                                                              ---*/
/*--------------------------------------------------------------------*/

static const REAL   bary3_1d[4][N_LAMBDA] = {{1.0,    0.0,     0.0, 0.0},
					     {0.0,    1.0,     0.0, 0.0},
					     {2.0/3.0,1.0/3.0, 0.0, 0.0},
					     {1.0/3.0,2.0/3.0, 0.0, 0.0}};

/*--------------------------------------------------------------------------*/
/*---  basisfunction 0 located at vertex 0                               ---*/
/*--------------------------------------------------------------------------*/

static REAL phi3_0_1d(const REAL lambda[N_LAMBDA])
{
  return((4.5*(lambda[0] - 1.0)*lambda[0] + 1.0)*lambda[0]);
}

static const REAL *grd_phi3_0_1d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0};
  grd[0] = (13.5*lambda[0] - 9.0)*lambda[0] + 1;
  return((const REAL *) grd);
}

static const REAL (*D2_phi3_0_1d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};
  D2[0][0] = 27.0*lambda[0] - 9.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*---  basisfunction 1 located at vertex 1                               ---*/
/*--------------------------------------------------------------------------*/

static REAL phi3_1_1d(const REAL lambda[N_LAMBDA])
{
  return((4.5*(lambda[1] - 1.0)*lambda[1] + 1.0)*lambda[1]);
}

static const REAL *grd_phi3_1_1d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0};
  grd[1] = (13.5*lambda[1] - 9.0)*lambda[1] + 1;
  return((const REAL *) grd);
}

static const REAL (*D2_phi3_1_1d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};
  D2[1][1] = 27.0*lambda[1] - 9.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*---  basisfunction 2, first at center                                  ---*/
/*--------------------------------------------------------------------------*/

static REAL phi3_2_1d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[0] - 4.5)*lambda[0]*lambda[1]);
}

static const REAL *grd_phi3_2_1d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0};
  grd[0] = (27.0*lambda[0] - 4.5)*lambda[1];
  grd[1] = (13.5*lambda[0] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3_2_1d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};
  D2[0][0] = 27.0*lambda[1];
  D2[0][1] = D2[1][0] = 27.0*lambda[0] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*---  basisfunction 3, second at center                                 ---*/
/*--------------------------------------------------------------------------*/

static REAL phi3_3_1d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[1] - 4.5)*lambda[1]*lambda[0]);
}

static const REAL *grd_phi3_3_1d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0.0};
  grd[0] = (13.5*lambda[1] - 4.5)*lambda[1];
  grd[1] = (27.0*lambda[1] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3_3_1d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA] = {{0.0}};
  D2[0][1] = D2[1][0] = 27.0*lambda[1] - 4.5;
  D2[1][1] = 27.0*lambda[0];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing local DOFs on an element              ---*/
/*--------------------------------------------------------------------*/

static const DOF *get_dof_indices3_1d(const EL *el, const DOF_ADMIN *admin,
				      DOF *idof)
{
  static DOF  dof_vec[4];
  DOF         *rvec = idof ? idof : dof_vec;
  int         ibas = 0, i, j, n0, node;
  DOF         **dof = el->dof;

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = admin->mesh->node[VERTEX];
  n0   = admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = dof[node+i][n0];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = admin->mesh->node[CENTER];
  n0   = admin->n0_dof[CENTER];
  for (j = 0; j < 2; j++)
    rvec[ibas++] = dof[node][n0+j];


  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing boundary type of DOFs                 ---*/
/*--------------------------------------------------------------------*/

static const S_CHAR *get_bound3_1d(const EL_INFO *el_info, S_CHAR *vec)
{
  FUNCNAME("get_bound3_1d");
  static S_CHAR  my_vec[4];
  S_CHAR         *rvec = vec ? vec : my_vec;
  int            ibas = 0, i, j;

  DEBUG_TEST_FLAG(FILL_BOUND, el_info);

/*--------------------------------------------------------------------*/
/*--- basis functions at vertices                                  ---*/
/*--------------------------------------------------------------------*/

  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = el_info->vertex_bound[i];

/*--------------------------------------------------------------------*/
/*--- basis functions at center                                    ---*/
/*--------------------------------------------------------------------*/

  for (j = 0; j < 2; j++)
    rvec[ibas++] = INTERIOR;

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(,3,1,4)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(,3,1,4)

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_INT_VEC                   ---*/
/*--------------------------------------------------------------------*/

static const int *get_int_vec3_1d(const EL *el, const DOF_INT_VEC *dv,
				  int *vec)
{
  FUNCNAME("get_int_vec3_1d");
  static int     my_vec[4];
  int            *rvec = vec ? vec : my_vec;
  int            *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_INT_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 2; j++)
    rvec[ibas++] = v[dof[node][n0+j]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_VEC                  ---*/
/*--------------------------------------------------------------------*/

static const REAL *get_real_vec3_1d(const EL *el, const DOF_REAL_VEC *dv,
				    REAL *vec)
{
  FUNCNAME("get_real_vec3_1d");
  static REAL    my_vec[4];
  REAL           *rvec = vec ? vec : my_vec;
  REAL           *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_REAL_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 2; j++)
    rvec[ibas++] = v[dof[node][n0+j]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_D_VEC                ---*/
/*--------------------------------------------------------------------*/

static const REAL_D *get_real_d_vec3_1d(const EL *el, const DOF_REAL_D_VEC *dv,
					REAL_D *vec)
{
  FUNCNAME("get_real_d_vec3_1d");
  static REAL_D  my_vec[4];
  REAL_D         *rvec = vec ? vec : my_vec;
  REAL_D         *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_REAL_D_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
  {
    for (n = 0; n < DIM_OF_WORLD; n++)
      rvec[ibas][n] = v[dof[node+i][n0]][n];
    ibas++;
  }

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 2; j++)
  {
    for (n = 0; n < DIM_OF_WORLD; n++)
      rvec[ibas][n] = v[dof[node][n0+j]][n];
    ibas++;
  }

  return((const REAL_D *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_SCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

static const S_CHAR *get_schar_vec3_1d(const EL *el, const DOF_SCHAR_VEC *dv,
				       S_CHAR *vec)
{
  FUNCNAME("get_schar_vec3_1d");
  static S_CHAR  my_vec[4];
  S_CHAR         *rvec = vec ? vec : my_vec;
  S_CHAR         *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_SCHAR_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 2; j++)
    rvec[ibas++] = v[dof[node][n0+j]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_UCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

static const U_CHAR *get_uchar_vec3_1d(const EL *el, const DOF_UCHAR_VEC *dv,
				       U_CHAR *vec)
{
  FUNCNAME("get_uchar_vec3_1d");
  static U_CHAR  my_vec[4];
  U_CHAR         *rvec = vec ? vec : my_vec;
  U_CHAR         *v = dv ? dv->vec : nil;
  int            ibas = 0, i, j, n0, node;
  DOF            **dof = el->dof;

  DEBUG_TEST_EXIT(v, "no DOF_UCHAR_VEC dv, or no dv->vec\n");

/*--------------------------------------------------------------------*/
/*--- DOFs at vertices                                             ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[VERTEX];
  n0   = dv->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_1D; i++)
    rvec[ibas++] = v[dof[node+i][n0]];

/*--------------------------------------------------------------------*/
/*--- DOFs at center                                               ---*/
/*--------------------------------------------------------------------*/

  node = dv->fe_space->mesh->node[CENTER];
  n0   = dv->fe_space->admin->n0_dof[CENTER];
  for (j = 0; j < 2; j++)
    rvec[ibas++] = v[dof[node][n0+j]];

  return(rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton of DOF_REAL_VECs during refinement ---*/
/*--------------------------------------------------------------------*/

static void refine_inter3_1d(DOF_REAL_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *cdof;
  const REAL      *pvec;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL            *v = dv->vec;

  el = list->el_info.el;
  pvec = (*bas_fcts->get_real_vec)(el, dv, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, first child                           ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[0], admin, nil);

  v[cdof[1]] = (-0.0625*pvec[0] - 0.0625*pvec[1] + 0.5625*pvec[2] + 0.5625*pvec[3]);
  v[cdof[2]] = (0.3125*pvec[0] + 0.0625*pvec[1] + 0.9375*pvec[2] - 0.3125*pvec[3]);
  v[cdof[3]] = (pvec[2]);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[1], admin, nil);

  v[cdof[2]] = (pvec[3]);
  v[cdof[3]] = (0.0625*pvec[0] + 0.3125*pvec[1] - 0.3125*pvec[2] + 0.9375*pvec[3]);

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton of DOF_REAL_D_VECs during refinement -*/
/*--------------------------------------------------------------------*/

static void refine_inter_d3_1d(DOF_REAL_D_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *cdof;
  const REAL_D    *pvec;
  int             n;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL_D          *v = dv->vec;

  el = list->el_info.el;
  pvec = (*bas_fcts->get_real_d_vec)(el, dv, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, first child                           ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[0], admin, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[cdof[1]][n] = (-0.0625*pvec[0][n] - 0.0625*pvec[1][n] + 0.5625*pvec[2][n] + 0.5625*pvec[3][n]);
    v[cdof[2]][n] = (0.3125*pvec[0][n] + 0.0625*pvec[1][n] + 0.9375*pvec[2][n] - 0.3125*pvec[3][n]);
    v[cdof[3]][n] = (pvec[2][n]);
  }

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cdof = (*bas_fcts->get_dof_indices)(el->child[1], admin, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[cdof[2]][n] = (pvec[3][n]);
    v[cdof[3]][n] = (0.0625*pvec[0][n] + 0.3125*pvec[1][n] - 0.3125*pvec[2][n] + 0.9375*pvec[3][n]);
  }

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton of DOF_REAL_VECs during coarsening ---*/
/*--------------------------------------------------------------------*/

static void coarse_inter3_1d(DOF_REAL_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL      *cvec;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL            *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[0], dv, nil);

  v[pdof[2]] = (cvec[3]);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[1], dv, nil);

  v[pdof[3]] = (cvec[2]);

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for interpolaton of DOF_REAL_D_VECs during coarsening -*/
/*--------------------------------------------------------------------*/

static void coarse_inter_d3_1d(DOF_REAL_D_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL_D    *cvec;
  int             n;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL_D          *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_d_vec)(el->child[0], dv, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[pdof[2]][n] = (cvec[3][n]);
  }

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, second child                          ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_d_vec)(el->child[1], dv, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[pdof[3]][n] = (cvec[2][n]);
  }

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for restriction of DOF_REAL_VECs during coarsening  ---*/
/*--------------------------------------------------------------------*/

static void coarse_restr3_1d(DOF_REAL_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL      *cvec;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL            *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[0], dv, nil);

  v[pdof[0]] += (-0.0625*cvec[1] + 0.3125*cvec[2]);
  v[pdof[1]] += (-0.0625*cvec[1] + 0.0625*cvec[2]);
  v[pdof[2]]  = (0.5625*cvec[1] + 0.9375*cvec[2] + cvec[3]);
  v[pdof[3]]  = (0.5625*cvec[1] - 0.3125*cvec[2]);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from second child                     ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_vec)(el->child[1], dv, nil);

  v[pdof[0]] += (0.0625*cvec[3]);
  v[pdof[1]] += (0.3125*cvec[3]);
  v[pdof[2]] += (-0.3125*cvec[3]);
  v[pdof[3]] += (cvec[2] + 0.9375*cvec[3]);

  return;
}

/*--------------------------------------------------------------------*/
/*--- function for restriction of DOF_REAL_D_VECs during coarsening --*/
/*--------------------------------------------------------------------*/

static void coarse_restr_d3_1d(DOF_REAL_D_VEC *dv, RC_LIST_EL *list, int n_el)
{
  EL              *el;
  const DOF       *pdof;
  const REAL_D    *cvec;
  int             n;
  const BAS_FCTS  *bas_fcts = dv->fe_space->bas_fcts;
  const DOF_ADMIN *admin = dv->fe_space->admin;
  REAL_D          *v = dv->vec;

  el = list->el_info.el;
  pdof = (*bas_fcts->get_dof_indices)(el, admin, nil);

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from first child                      ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_d_vec)(el->child[0], dv, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[pdof[0]][n] += (-0.0625*cvec[1][n] + 0.3125*cvec[2][n]);
    v[pdof[1]][n] += (-0.0625*cvec[1][n] + 0.0625*cvec[2][n]);
    v[pdof[2]][n]  = (0.5625*cvec[1][n] + 0.9375*cvec[2][n] + cvec[3][n]);
    v[pdof[3]][n]  = (0.5625*cvec[1][n] - 0.3125*cvec[2][n]);
  }

/*--------------------------------------------------------------------*/
/*--- DOFs on first element, from second child                     ---*/
/*--------------------------------------------------------------------*/

  cvec = (*bas_fcts->get_real_d_vec)(el->child[1], dv, nil);

  for (n = 0; n < DIM_OF_WORLD; n++)
  {
    v[pdof[0]][n] += (0.0625*cvec[3][n]);
    v[pdof[1]][n] += (0.3125*cvec[3][n]);
    v[pdof[2]][n] += (-0.3125*cvec[3][n]);
    v[pdof[3]][n] += (cvec[2][n] + 0.9375*cvec[3][n]);
  }

  return;
}

/*--------------------------------------------------------------------*/
/*--- Collect all information about basis functions                ---*/
/*--------------------------------------------------------------------*/

static BAS_FCT *phi3_1d[4] =
{
  phi3_0_1d, phi3_1_1d, phi3_2_1d, phi3_3_1d
};

static GRD_BAS_FCT *grd_phi3_1d[4] =
{
  grd_phi3_0_1d, grd_phi3_1_1d, grd_phi3_2_1d, grd_phi3_3_1d
};
static D2_BAS_FCT *D2_phi3_1d[4] =
{
  D2_phi3_0_1d, D2_phi3_1_1d, D2_phi3_2_1d, D2_phi3_3_1d
};

static BAS_FCTS lagrange3_1d =
{
  "lagrange3_1d", 1, 4, 3,
  {1, 2, 0, 0},  /* VERTEX, CENTER, EDGE, FACE   */
  nil,
  phi3_1d, grd_phi3_1d, D2_phi3_1d,
  get_dof_indices3_1d,
  get_bound3_1d,
  interpol3_1d,
  interpol_d3_1d,
  get_int_vec3_1d,
  get_real_vec3_1d,
  get_real_d_vec3_1d,
  get_uchar_vec3_1d,
  get_schar_vec3_1d,
  refine_inter3_1d,
  coarse_inter3_1d,
  coarse_restr3_1d,
  refine_inter_d3_1d,
  coarse_inter_d3_1d,
  coarse_restr_d3_1d,
  bary3_1d,
};
