// Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
// GNU Lesser General Public License version 3 (see the file COPYING).

// Define the role sizes available in Azure.

package gwacl

import (
	"fmt"
)

// RoleSize is a representation of the machine specs available in the Azure
// documentation here:
//   http://msdn.microsoft.com/en-us/library/windowsazure/dn197896.aspx
//
// Pricing from here:
//   http://azure.microsoft.com/en-us/pricing/details/virtual-machines
//
// Detailed specifications here:
//   http://msdn.microsoft.com/en-us/library/windowsazure/dn197896.aspx
//
// Our specifications may be inaccurate or out of date.  When in doubt, check!
//
// The Disk Space values are only the maxumim permitted; actual space is
// determined by the OS image being used.
//
// Sizes and costs last updated 2014-06-23.
type RoleSize struct {
	// Name is the name that Azure assigns to role sizes.
	Name string

	// Alias is an alternative name for the role size. This may be empty.
	//
	// Alias exists because the canonical role size names are inconsistent.
	// There are currently three different formats for "standard" role sizes:
	// e.g. ExtraSmall, A6, Standard_D1.
	Alias string

	CpuCores      uint64
	Mem           uint64 // In MB
	OSDiskSpace   uint64 // In MB
	TempDiskSpace uint64 // In MB
	MaxDataDisks  uint64 // 1TB each
}

// decicentsPerHour is the unit of cost we store for RoleSizeCost.
type decicentsPerHour uint64

const (
	// MB is the unit in which we specify sizes, so it's 1.
	// But please include it anyway, so that units are always explicit.
	MB = 1
	GB = 1024 * MB
	TB = 1024 * GB
)

// Basic tier roles.
var basicRoleSizes = []RoleSize{{ // A0..A4: general purpose
	Name:          "Basic_A0",
	CpuCores:      1, // shared
	Mem:           768 * MB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 20 * GB,
	MaxDataDisks:  1,
}, {
	Name:          "Basic_A1",
	CpuCores:      1,
	Mem:           1.75 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 40 * GB,
	MaxDataDisks:  2,
}, {
	Name:          "Basic_A2",
	CpuCores:      2,
	Mem:           3.5 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 60 * GB,
	MaxDataDisks:  4,
}, {
	Name:          "Basic_A3",
	CpuCores:      4,
	Mem:           7 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 120 * GB,
	MaxDataDisks:  8,
}, {
	Name:          "Basic_A4",
	CpuCores:      8,
	Mem:           14 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 240 * GB,
	MaxDataDisks:  16,
}}

// Standard tier roles.
var standardRoleSizes = []RoleSize{{ // A0..A4: general purpose
	Name:          "ExtraSmall",
	Alias:         "A0",
	CpuCores:      1, // shared
	Mem:           768 * MB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 20 * GB,
	MaxDataDisks:  1,
}, {
	Name:          "Small",
	Alias:         "A1",
	CpuCores:      1,
	Mem:           1.75 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 70 * GB,
	MaxDataDisks:  2,
}, {
	Name:          "Medium",
	Alias:         "A2",
	CpuCores:      2,
	Mem:           3.5 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 135 * GB,
	MaxDataDisks:  4,
}, {
	Name:          "Large",
	Alias:         "A3",
	CpuCores:      4,
	Mem:           7 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 285 * GB,
	MaxDataDisks:  8,
}, {
	Name:          "ExtraLarge",
	Alias:         "A4",
	CpuCores:      8,
	Mem:           14 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 605 * GB,
	MaxDataDisks:  16,
}, { // A5..A7: memory intensive
	Name:          "A5",
	CpuCores:      2,
	Mem:           14 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 135 * GB,
	MaxDataDisks:  4,
}, {
	Name:          "A6",
	CpuCores:      4,
	Mem:           28 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 285 * GB,
	MaxDataDisks:  8,
}, {
	Name:          "A7",
	CpuCores:      8,
	Mem:           56 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 605 * GB,
	MaxDataDisks:  16,
}, { // A8..A9: compute intensive / fast network & infiniband suport
	Name:          "A8",
	CpuCores:      8,
	Mem:           56 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 382 * GB,
	MaxDataDisks:  16,
}, {
	Name:          "A9",
	CpuCores:      16,
	Mem:           112 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 382 * GB,
	MaxDataDisks:  16,
}, { // D Series - xeon v3 and ssd local
	Name:          "Standard_D1",
	Alias:         "D1",
	CpuCores:      1,
	Mem:           3.5 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 50 * GB,
	MaxDataDisks:  2,
}, {
	Name:          "Standard_D2",
	Alias:         "D2",
	CpuCores:      2,
	Mem:           7 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 100 * GB,
	MaxDataDisks:  4,
}, {
	Name:          "Standard_D3",
	Alias:         "D3",
	CpuCores:      4,
	Mem:           14 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 200 * GB,
	MaxDataDisks:  8,
}, {
	Name:          "Standard_D4",
	Alias:         "D4",
	CpuCores:      8,
	Mem:           28 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 400 * GB,
	MaxDataDisks:  16,
}, {
	Name:          "Standard_D11",
	Alias:         "D11",
	CpuCores:      2,
	Mem:           14 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 100 * GB,
	MaxDataDisks:  4,
}, {
	Name:          "Standard_D12",
	Alias:         "D12",
	CpuCores:      4,
	Mem:           28 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 200 * GB,
	MaxDataDisks:  8,
}, {
	Name:          "Standard_D13",
	Alias:         "D12",
	CpuCores:      8,
	Mem:           56 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 400 * GB,
	MaxDataDisks:  16,
}, {
	Name:          "Standard_D14",
	Alias:         "D12",
	CpuCores:      16,
	Mem:           112 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 800 * GB,
	MaxDataDisks:  16,
}, { // G Series Large Compute + SSD Local
	Name:          "Standard_G1",
	Alias:         "G1",
	CpuCores:      2,
	Mem:           28 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 406 * GB,
	MaxDataDisks:  4,
}, {
	Name:          "Standard_G2",
	Alias:         "G2",
	CpuCores:      4,
	Mem:           56 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 812 * GB,
	MaxDataDisks:  8,
}, {
	Name:          "Standard_G3",
	Alias:         "G3",
	CpuCores:      8,
	Mem:           112 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 1630 * GB,
	MaxDataDisks:  16,
}, {
	Name:          "Standard_G4",
	Alias:         "G4",
	CpuCores:      16,
	Mem:           224 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 3250 * GB,
	MaxDataDisks:  16,
}, {
	Name:          "Standard_G5",
	Alias:         "G5",
	CpuCores:      32,
	Mem:           448 * GB,
	OSDiskSpace:   127 * GB,
	TempDiskSpace: 6500 * GB,
	MaxDataDisks:  16,
}}

// RoleSizes describes all known role sizes.
var RoleSizes = append(append([]RoleSize{}, basicRoleSizes...), standardRoleSizes...)

var allRegionRoleCosts = map[string]map[string]decicentsPerHour{
	"East US": {
		"Basic_A0":     18,
		"Basic_A1":     44,
		"Basic_A2":     88,
		"Basic_A3":     176,
		"Basic_A4":     352,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           250,
		"A6":           500,
		"A7":           1000,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  94,
		"Standard_D2":  188,
		"Standard_D3":  376,
		"Standard_D4":  752,
		"Standard_D11": 238,
		"Standard_D12": 476,
		"Standard_D13": 857,
		"Standard_D14": 1542,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"East US 2": {
		"Basic_A0":     18,
		"Basic_A1":     44,
		"Basic_A2":     88,
		"Basic_A3":     176,
		"Basic_A4":     352,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           220,
		"A6":           440,
		"A7":           880,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  85,
		"Standard_D2":  170,
		"Standard_D3":  340,
		"Standard_D4":  680,
		"Standard_D11": 214,
		"Standard_D12": 428,
		"Standard_D13": 770,
		"Standard_D14": 1387,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"West US": {
		"Basic_A0":     18,
		"Basic_A1":     47,
		"Basic_A2":     94,
		"Basic_A3":     188,
		"Basic_A4":     376,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           250,
		"A6":           500,
		"A7":           1000,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  94,
		"Standard_D2":  188,
		"Standard_D3":  376,
		"Standard_D4":  752,
		"Standard_D11": 238,
		"Standard_D12": 476,
		"Standard_D13": 857,
		"Standard_D14": 1542,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},

	"Central US": {
		"Basic_A0":     18,
		"Basic_A1":     60,
		"Basic_A2":     94,
		"Basic_A3":     188,
		"Basic_A4":     376,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           250,
		"A6":           500,
		"A7":           1000,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  94,
		"Standard_D2":  188,
		"Standard_D3":  376,
		"Standard_D4":  752,
		"Standard_D11": 238,
		"Standard_D12": 476,
		"Standard_D13": 857,
		"Standard_D14": 1542,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"North Central US": {
		"Basic_A0":     18,
		"Basic_A1":     47,
		"Basic_A2":     94,
		"Basic_A3":     188,
		"Basic_A4":     376,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           250,
		"A6":           500,
		"A7":           1000,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  94,
		"Standard_D2":  188,
		"Standard_D3":  376,
		"Standard_D4":  752,
		"Standard_D11": 238,
		"Standard_D12": 476,
		"Standard_D13": 857,
		"Standard_D14": 1542,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"South Central US": {
		"Basic_A0":     18,
		"Basic_A1":     44,
		"Basic_A2":     88,
		"Basic_A3":     176,
		"Basic_A4":     352,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           220,
		"A6":           440,
		"A7":           880,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  85,
		"Standard_D2":  170,
		"Standard_D3":  340,
		"Standard_D4":  680,
		"Standard_D11": 214,
		"Standard_D12": 428,
		"Standard_D13": 770,
		"Standard_D14": 1387,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"North Europe": {
		"Basic_A0":     18,
		"Basic_A1":     47,
		"Basic_A2":     94,
		"Basic_A3":     188,
		"Basic_A4":     376,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           248,
		"A6":           496,
		"A7":           992,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  94,
		"Standard_D2":  188,
		"Standard_D3":  376,
		"Standard_D4":  752,
		"Standard_D11": 238,
		"Standard_D12": 476,
		"Standard_D13": 857,
		"Standard_D14": 1542,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"West Europe": {
		"Basic_A0":     18,
		"Basic_A1":     51,
		"Basic_A2":     102,
		"Basic_A3":     204,
		"Basic_A4":     408,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           270,
		"A6":           540,
		"A7":           1080,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  115,
		"Standard_D2":  230,
		"Standard_D3":  460,
		"Standard_D4":  920,
		"Standard_D11": 273,
		"Standard_D12": 546,
		"Standard_D13": 983,
		"Standard_D14": 1769,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"Southeast Asia": {
		"Basic_A0":     18,
		"Basic_A1":     58,
		"Basic_A2":     116,
		"Basic_A3":     232,
		"Basic_A4":     464,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           270,
		"A6":           540,
		"A7":           1080,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  120,
		"Standard_D2":  240,
		"Standard_D3":  480,
		"Standard_D4":  960,
		"Standard_D11": 256,
		"Standard_D12": 512,
		"Standard_D13": 922,
		"Standard_D14": 1659,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"East Asia": {
		"Basic_A0":     18,
		"Basic_A1":     58,
		"Basic_A2":     116,
		"Basic_A3":     232,
		"Basic_A4":     464,
		"ExtraSmall":   20,
		"Small":        60,
		"Medium":       120,
		"Large":        240,
		"ExtraLarge":   480,
		"A5":           294,
		"A6":           588,
		"A7":           1176,
		"A8":           1970,
		"A9":           4470,
		"Standard_D1":  138,
		"Standard_D2":  276,
		"Standard_D3":  552,
		"Standard_D4":  1104,
		"Standard_D11": 295,
		"Standard_D12": 590,
		"Standard_D13": 1062,
		"Standard_D14": 1912,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"Japan East": {
		"Basic_A0":     18,
		"Basic_A1":     69,
		"Basic_A2":     138,
		"Basic_A3":     276,
		"Basic_A4":     552,
		"ExtraSmall":   27,
		"Small":        81,
		"Medium":       162,
		"Large":        324,
		"ExtraLarge":   648,
		"A5":           281,
		"A6":           562,
		"A7":           1124,
		"A8":           2325,
		"A9":           5275,
		"Standard_D1":  142,
		"Standard_D2":  284,
		"Standard_D3":  568,
		"Standard_D4":  1136,
		"Standard_D11": 295,
		"Standard_D12": 590,
		"Standard_D13": 1062,
		"Standard_D14": 1912,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"Japan West": {
		"Basic_A0":     19,
		"Basic_A1":     61,
		"Basic_A2":     122,
		"Basic_A3":     244,
		"Basic_A4":     488,
		"ExtraSmall":   21,
		"Small":        73,
		"Medium":       146,
		"Large":        292,
		"ExtraLarge":   584,
		"A5":           258,
		"A6":           516,
		"A7":           1032,
		"A8":           2088,
		"A9":           4738,
		"Standard_D1":  123,
		"Standard_D2":  246,
		"Standard_D3":  492,
		"Standard_D4":  984,
		"Standard_D11": 256,
		"Standard_D12": 512,
		"Standard_D13": 922,
		"Standard_D14": 1659,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"Brazil South": {
		"Basic_A0":     22,
		"Basic_A1":     58,
		"Basic_A2":     116,
		"Basic_A3":     232,
		"Basic_A4":     464,
		"ExtraSmall":   24,
		"Small":        80,
		"Medium":       160,
		"Large":        320,
		"ExtraLarge":   640,
		"A5":           291,
		"A6":           582,
		"A7":           1164,
		"A8":           2403,
		"A9":           5453,
		"Standard_D1":  116,
		"Standard_D2":  232,
		"Standard_D3":  464,
		"Standard_D4":  928,
		"Standard_D11": 290,
		"Standard_D12": 580,
		"Standard_D13": 1044,
		"Standard_D14": 1879,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"Australia East": {
		"Basic_A0":     24,
		"Basic_A1":     58,
		"Basic_A2":     116,
		"Basic_A3":     232,
		"Basic_A4":     464,
		"ExtraSmall":   29,
		"Small":        71,
		"Medium":       142,
		"Large":        284,
		"ExtraLarge":   568,
		"A5":           278,
		"A6":           556,
		"A7":           1112,
		"A8":           2224,
		"A9":           4448,
		"Standard_D1":  120,
		"Standard_D2":  239,
		"Standard_D3":  478,
		"Standard_D4":  956,
		"Standard_D11": 256,
		"Standard_D12": 512,
		"Standard_D13": 922,
		"Standard_D14": 1660,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
	"Australia Southeast": {
		"Basic_A0":     24,
		"Basic_A1":     58,
		"Basic_A2":     116,
		"Basic_A3":     232,
		"Basic_A4":     464,
		"ExtraSmall":   29,
		"Small":        71,
		"Medium":       142,
		"Large":        284,
		"ExtraLarge":   568,
		"A5":           278,
		"A6":           556,
		"A7":           1112,
		"A8":           2224,
		"A9":           4448,
		"Standard_D1":  120,
		"Standard_D2":  239,
		"Standard_D3":  478,
		"Standard_D4":  956,
		"Standard_D11": 256,
		"Standard_D12": 512,
		"Standard_D13": 922,
		"Standard_D14": 1660,
		"Standard_G1":  8940, // Estimate, price is not public
		"Standard_G2":  8950, // Estimate, price is not public
		"Standard_G3":  8960, // Estimate, price is not public
		"Standard_G4":  8970, // Estimate, price is not public
		"Standard_G5":  8980, // Estimate, price is not public
	},
}

// RoleSizeCost returns the cost associated with the given role size and region.
func RoleSizeCost(region string, roleSize string) (decicentsPerHour uint64, err error) {
	costs, ok := allRegionRoleCosts[region]
	if !ok {
		return 0, fmt.Errorf("no cost data for region %q", region)
	}
	cost, ok := costs[roleSize]
	if ok {
		return uint64(cost), nil
	}
	return 0, fmt.Errorf(
		"no cost data for role size %q in region %q",
		roleSize, region,
	)
}
