/* Ergo, version 3.3, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2013 Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Kohn−Sham Density Functional Theory Electronic Structure Calculations 
 * with Linearly Scaling Computational Time and Memory Usage,
 * Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek,
 * J. Chem. Theory Comput. 7, 340 (2011),
 * <http://dx.doi.org/10.1021/ct100611z>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */
#ifdef USE_CHUNKS_AND_TASKS

#include <cstring>
#include "MoleculeChunk.h"

CHT_CHUNK_TYPE_IMPLEMENTATION((MoleculeChunk));

void MoleculeChunk::writeToBuffer ( char * dataBuffer, 
				    size_t const bufferSize ) const {
  // First store one int number: the number of atoms.
  char* p = dataBuffer;
  if(bufferSize < getSize())
    throw std::runtime_error("Error: bufferSize too small.");
  int nAtoms = m.getNoOfAtoms();
  memcpy(p, &nAtoms, sizeof(int));
  p += sizeof(int);
  // Now store the charge.
  ergo_real netCharge = m.getNetCharge();
  memcpy(p, &netCharge, sizeof(ergo_real));
  p += sizeof(ergo_real);
  // Now copy the list of atoms.
  memcpy(p, m.getAtomListPtr(), m.getNoOfAtoms() * sizeof(Atom));
  p += m.getNoOfAtoms() * sizeof(Atom);
  // DONE!
}

size_t MoleculeChunk::getSize() const {
  return 1 * sizeof(int) 
    + 1 * sizeof(ergo_real)
    + m.getNoOfAtoms() * sizeof(Atom);
}

void MoleculeChunk::assignFromBuffer ( char const * dataBuffer, 
				       size_t const bufferSize) {
  // First get the number of atoms.
  const char* p = dataBuffer;
  if(bufferSize < sizeof(int) + sizeof(ergo_real))
    throw std::runtime_error("Error: bufferSize too small.");
  int nAtoms = 0;
  memcpy(&nAtoms, p, sizeof(int));
  p += sizeof(int);
  // Now get the charge.
  ergo_real netCharge = 0;
  memcpy(&netCharge, p, sizeof(ergo_real));
  p += sizeof(ergo_real);
  // Now get the atom list.
  std::vector<Atom> atomList(nAtoms);
  memcpy(&atomList[0], p, nAtoms*sizeof(Atom));
  p += nAtoms*sizeof(Atom);
  // Ok, now we have exctracted all needed info.
  if((p-dataBuffer) > bufferSize)
    throw std::runtime_error("Error: (p > bufferSize).");
  // Now create molecule object.
  m.setNetCharge(netCharge);
  m.setAtomList(atomList);
  // DONE!
}

size_t MoleculeChunk::memoryUsage() const {
  return getSize();
}

void MoleculeChunk::getChildChunks(std::list<cht::ChunkID> & childChunkIDs) const { }


#endif
