# coding: utf8
#
# Copyright (C) 2013  Enrico Zini <enrico@debian.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
from __future__ import print_function
from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals
import debiancontributors.parser as parser
import unittest

class TestParser(unittest.TestCase):
    def testGetKey(self):
        self.assertEquals(parser.get_key({"foo": "bar"}, "foo"), "bar")
        self.assertRaises(parser.Fail, parser.get_key, {}, "foo")

    def testGetKeyInt(self):
        self.assertEquals(parser.get_key_int({"foo": "7"}, "foo"), 7)
        self.assertRaises(parser.Fail, parser.get_key_int, {}, "foo")
        self.assertRaises(parser.Fail, parser.get_key_int, {"foo": ""}, "foo")
        self.assertRaises(parser.Fail, parser.get_key_int, {"foo": "seven"}, "foo")

    def testGetKeyString(self):
        self.assertEquals(parser.get_key_string({"foo": "7"}, "foo"), "7")
        self.assertEquals(parser.get_key_string({"foo": ""}, "foo", True), "")
        self.assertEquals(parser.get_key_string({"foo": None}, "foo", True), "")
        self.assertEquals(parser.get_key_string({}, "foo", True), "")
        self.assertRaises(parser.Fail, parser.get_key_string, {}, "foo")
        self.assertRaises(parser.Fail, parser.get_key_string, {"foo": ""}, "foo")

    def testGetKeyUnicode(self):
        self.assertEquals(parser.get_key_unicode({"foo": "7"}, "foo"), "7")
        self.assertEquals(parser.get_key_unicode({"foo": b"\xe2\x99\xa5"}, "foo"), "♥")
        self.assertEquals(parser.get_key_unicode({"foo": ""}, "foo", True), "")
        self.assertEquals(parser.get_key_unicode({"foo": None}, "foo", True), "")
        self.assertEquals(parser.get_key_unicode({}, "foo", True), "")
        self.assertRaises(parser.Fail, parser.get_key_unicode, {}, "foo")
        self.assertRaises(parser.Fail, parser.get_key_unicode, {"foo": ""}, "foo")
        self.assertRaises(parser.Fail, parser.get_key_unicode, {"foo": b'\xe0'}, "foo")

    def testGetKeySequence(self):
        self.assertEquals(parser.get_key_sequence({"foo": []}, "foo"), [])
        self.assertEquals(parser.get_key_sequence({"foo": [1, 2, "three"]}, "foo"), [1, 2, "three"])
        self.assertEquals(parser.get_key_sequence({"foo": ()}, "foo"), ())
        self.assertRaises(parser.Fail, parser.get_key_sequence, {}, "foo")
        self.assertRaises(parser.Fail, parser.get_key_sequence, {"foo": "bar"}, "foo")
        self.assertRaises(parser.Fail, parser.get_key_sequence, {"foo": {}}, "foo")

    def testGetKeySequenceOrObject(self):
        self.assertEquals(parser.get_key_sequence_or_object({"foo": []}, "foo"), [])
        self.assertEquals(parser.get_key_sequence_or_object({"foo": {}}, "foo"), [{}])
        self.assertEquals(parser.get_key_sequence_or_object({"foo": [{}]}, "foo"), [{}])
        self.assertRaises(parser.Fail, parser.get_key_sequence_or_object, {}, "foo")
        self.assertRaises(parser.Fail, parser.get_key_sequence_or_object, {"foo": "bar"}, "foo")

    def testGetKeyDateOrNone(self):
        from datetime import date
        self.assertEquals(parser.get_key_date_or_none({"foo": "2013-11-16"}, "foo"), date(2013, 11, 16))
        self.assertEquals(parser.get_key_date_or_none({"foo": ""}, "foo"), None)
        self.assertEquals(parser.get_key_date_or_none({"foo": None}, "foo"), None)
        self.assertEquals(parser.get_key_date_or_none({}, "foo"), None)
        self.assertRaises(parser.Fail, parser.get_key_date_or_none, {"foo": "2013"}, "foo")

if __name__ == '__main__':
    unittest.main()
