/**
 * FreeRDP: A Remote Desktop Protocol Implementation
 * Extended Input channel server-side implementation
 *
 * Copyright 2014 Thincast Technologies Gmbh.
 * Copyright 2014 David FORT <contact@hardening-consulting.com>
 * Copyright 2015 Thincast Technologies GmbH
 * Copyright 2015 DI (FH) Martin Haimberger <martin.haimberger@thincast.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <freerdp/config.h>

#include <winpr/cast.h>
#include <winpr/crt.h>
#include <winpr/print.h>
#include <winpr/stream.h>

#include "rdpei_main.h"
#include "../rdpei_common.h"
#include <freerdp/channels/rdpei.h>
#include <freerdp/server/rdpei.h>

typedef enum
{
	RDPEI_INITIAL,
	RDPEI_OPENED,
} eRdpEiChannelState;

enum RdpEiState
{
	STATE_INITIAL,
	STATE_WAITING_CLIENT_READY,
	STATE_WAITING_FRAME,
	STATE_SUSPENDED,
};

struct s_rdpei_server_private
{
	HANDLE channelHandle;
	HANDLE eventHandle;

	HANDLE stopEvent;
	HANDLE thread;

	/* Channel state */
	eRdpEiChannelState channelState;

	UINT32 expectedBytes;
	BOOL waitingHeaders;
	wStream* inputStream;
	wStream* outputStream;

	UINT16 currentMsgType;

	RDPINPUT_TOUCH_EVENT touchEvent;
	RDPINPUT_PEN_EVENT penEvent;

	enum RdpEiState automataState;
};

static UINT rdpei_server_open_channel(RdpeiServerContext* context)
{
	DWORD error = ERROR_SUCCESS;
	DWORD bytesReturned = 0;
	PULONG pSessionId = NULL;
	BOOL status = TRUE;

	WINPR_ASSERT(context);

	RdpeiServerPrivate* priv = context->priv;
	WINPR_ASSERT(priv);

	if (WTSQuerySessionInformationA(context->vcm, WTS_CURRENT_SESSION, WTSSessionId,
	                                (LPSTR*)&pSessionId, &bytesReturned) == FALSE)
	{
		WLog_ERR(TAG, "WTSQuerySessionInformationA failed!");
		return ERROR_INTERNAL_ERROR;
	}

	DWORD sessionId = (DWORD)*pSessionId;
	WTSFreeMemory(pSessionId);

	priv->channelHandle =
	    WTSVirtualChannelOpenEx(sessionId, RDPEI_DVC_CHANNEL_NAME, WTS_CHANNEL_OPTION_DYNAMIC);
	if (!priv->channelHandle)
	{
		error = GetLastError();
		WLog_ERR(TAG, "WTSVirtualChannelOpenEx failed with error %" PRIu32 "!", error);
		return error;
	}

	const UINT32 channelId = WTSChannelGetIdByHandle(priv->channelHandle);

	IFCALLRET(context->onChannelIdAssigned, status, context, channelId);
	if (!status)
	{
		WLog_ERR(TAG, "context->onChannelIdAssigned failed!");
		return ERROR_INTERNAL_ERROR;
	}

	return error;
}

static UINT rdpei_server_context_poll_int(RdpeiServerContext* context)
{
	RdpeiServerPrivate* priv = NULL;
	UINT error = ERROR_INTERNAL_ERROR;

	WINPR_ASSERT(context);
	priv = context->priv;
	WINPR_ASSERT(priv);

	switch (priv->channelState)
	{
		case RDPEI_INITIAL:
			error = rdpei_server_open_channel(context);
			if (error)
				WLog_ERR(TAG, "rdpei_server_open_channel failed with error %" PRIu32 "!", error);
			else
				priv->channelState = RDPEI_OPENED;
			break;
		case RDPEI_OPENED:
			error = rdpei_server_handle_messages(context);
			break;
		default:
			break;
	}

	return error;
}

static HANDLE rdpei_server_get_channel_handle(RdpeiServerContext* context)
{
	RdpeiServerPrivate* priv = NULL;
	void* buffer = NULL;
	DWORD bytesReturned = 0;
	HANDLE channelEvent = NULL;

	WINPR_ASSERT(context);
	priv = context->priv;
	WINPR_ASSERT(priv);

	if (WTSVirtualChannelQuery(priv->channelHandle, WTSVirtualEventHandle, &buffer,
	                           &bytesReturned) == TRUE)
	{
		if (bytesReturned == sizeof(HANDLE))
			channelEvent = *(HANDLE*)buffer;

		WTSFreeMemory(buffer);
	}

	return channelEvent;
}

static DWORD WINAPI rdpei_server_thread_func(LPVOID arg)
{
	RdpeiServerContext* context = (RdpeiServerContext*)arg;
	RdpeiServerPrivate* priv = NULL;
	HANDLE events[2] = { 0 };
	DWORD nCount = 0;
	UINT error = CHANNEL_RC_OK;
	DWORD status = 0;

	WINPR_ASSERT(context);
	priv = context->priv;
	WINPR_ASSERT(priv);

	events[nCount++] = priv->stopEvent;

	while ((error == CHANNEL_RC_OK) && (WaitForSingleObject(events[0], 0) != WAIT_OBJECT_0))
	{
		switch (priv->channelState)
		{
			case RDPEI_INITIAL:
				error = rdpei_server_context_poll_int(context);
				if (error == CHANNEL_RC_OK)
				{
					events[1] = rdpei_server_get_channel_handle(context);
					nCount = 2;
				}
				break;
			case RDPEI_OPENED:
				status = WaitForMultipleObjects(nCount, events, FALSE, INFINITE);
				switch (status)
				{
					case WAIT_OBJECT_0:
						break;
					case WAIT_OBJECT_0 + 1:
					case WAIT_TIMEOUT:
						error = rdpei_server_context_poll_int(context);
						break;

					case WAIT_FAILED:
					default:
						error = ERROR_INTERNAL_ERROR;
						break;
				}
				break;
			default:
				break;
		}
	}

	(void)WTSVirtualChannelClose(priv->channelHandle);
	priv->channelHandle = NULL;

	ExitThread(error);
	return error;
}

static UINT rdpei_server_open(RdpeiServerContext* context)
{
	RdpeiServerPrivate* priv = NULL;

	priv = context->priv;
	WINPR_ASSERT(priv);

	if (!priv->thread)
	{
		priv->stopEvent = CreateEvent(NULL, TRUE, FALSE, NULL);
		if (!priv->stopEvent)
		{
			WLog_ERR(TAG, "CreateEvent failed!");
			return ERROR_INTERNAL_ERROR;
		}

		priv->thread = CreateThread(NULL, 0, rdpei_server_thread_func, context, 0, NULL);
		if (!priv->thread)
		{
			WLog_ERR(TAG, "CreateThread failed!");
			(void)CloseHandle(priv->stopEvent);
			priv->stopEvent = NULL;
			return ERROR_INTERNAL_ERROR;
		}
	}

	return CHANNEL_RC_OK;
}

static UINT rdpei_server_close(RdpeiServerContext* context)
{
	RdpeiServerPrivate* priv = NULL;
	UINT error = CHANNEL_RC_OK;

	priv = context->priv;
	WINPR_ASSERT(priv);

	if (priv->thread)
	{
		(void)SetEvent(priv->stopEvent);

		if (WaitForSingleObject(priv->thread, INFINITE) == WAIT_FAILED)
		{
			error = GetLastError();
			WLog_ERR(TAG, "WaitForSingleObject failed with error %" PRIu32 "", error);
			return error;
		}

		(void)CloseHandle(priv->thread);
		(void)CloseHandle(priv->stopEvent);
		priv->thread = NULL;
		priv->stopEvent = NULL;
	}

	return error;
}

RdpeiServerContext* rdpei_server_context_new(HANDLE vcm)
{
	RdpeiServerContext* ret = calloc(1, sizeof(*ret));

	if (!ret)
		return NULL;

	ret->Open = rdpei_server_open;
	ret->Close = rdpei_server_close;

	ret->priv = calloc(1, sizeof(*ret->priv));
	if (!ret->priv)
		goto fail;

	ret->priv->inputStream = Stream_New(NULL, 256);
	if (!ret->priv->inputStream)
		goto fail;

	ret->priv->outputStream = Stream_New(NULL, 200);
	if (!ret->priv->outputStream)
		goto fail;

	ret->vcm = vcm;
	rdpei_server_context_reset(ret);
	return ret;

fail:
	WINPR_PRAGMA_DIAG_PUSH
	WINPR_PRAGMA_DIAG_IGNORED_MISMATCHED_DEALLOC
	rdpei_server_context_free(ret);
	WINPR_PRAGMA_DIAG_POP
	return NULL;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
UINT rdpei_server_init(RdpeiServerContext* context)
{
	RdpeiServerPrivate* priv = context->priv;
	UINT error = rdpei_server_open_channel(context);
	if (error)
		return error;

	priv->eventHandle = rdpei_server_get_channel_handle(context);
	if (!priv->eventHandle)
	{
		WLog_ERR(TAG, "Failed to get channel handle!");
		goto out_close;
	}

	return CHANNEL_RC_OK;

out_close:
	(void)WTSVirtualChannelClose(priv->channelHandle);
	return CHANNEL_RC_INITIALIZATION_ERROR;
}

void rdpei_server_context_reset(RdpeiServerContext* context)
{
	RdpeiServerPrivate* priv = context->priv;

	priv->channelHandle = INVALID_HANDLE_VALUE;
	priv->expectedBytes = RDPINPUT_HEADER_LENGTH;
	priv->waitingHeaders = TRUE;
	priv->automataState = STATE_INITIAL;
	Stream_SetPosition(priv->inputStream, 0);
}

void rdpei_server_context_free(RdpeiServerContext* context)
{
	if (!context)
		return;
	RdpeiServerPrivate* priv = context->priv;
	if (priv)
	{
		if (priv->channelHandle && priv->channelHandle != INVALID_HANDLE_VALUE)
			(void)WTSVirtualChannelClose(priv->channelHandle);
		Stream_Free(priv->inputStream, TRUE);
		Stream_Free(priv->outputStream, TRUE);
	}
	free(priv);
	free(context);
}

HANDLE rdpei_server_get_event_handle(RdpeiServerContext* context)
{
	return context->priv->eventHandle;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
static UINT read_cs_ready_message(RdpeiServerContext* context, wStream* s)
{
	UINT error = CHANNEL_RC_OK;
	if (!Stream_CheckAndLogRequiredLength(TAG, s, 10))
		return ERROR_INVALID_DATA;

	Stream_Read_UINT32(s, context->protocolFlags);
	Stream_Read_UINT32(s, context->clientVersion);
	Stream_Read_UINT16(s, context->maxTouchPoints);

	switch (context->clientVersion)
	{
		case RDPINPUT_PROTOCOL_V10:
		case RDPINPUT_PROTOCOL_V101:
		case RDPINPUT_PROTOCOL_V200:
		case RDPINPUT_PROTOCOL_V300:
			break;
		default:
			WLog_ERR(TAG, "unhandled RPDEI protocol version 0x%" PRIx32 "", context->clientVersion);
			break;
	}

	IFCALLRET(context->onClientReady, error, context);
	if (error)
		WLog_ERR(TAG, "context->onClientReady failed with error %" PRIu32 "", error);

	return error;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
static UINT read_touch_contact_data(RdpeiServerContext* context, wStream* s,
                                    RDPINPUT_CONTACT_DATA* contactData)
{
	WINPR_UNUSED(context);
	if (!Stream_CheckAndLogRequiredLength(TAG, s, 1))
		return ERROR_INVALID_DATA;

	Stream_Read_UINT8(s, contactData->contactId);
	if (!rdpei_read_2byte_unsigned(s, &contactData->fieldsPresent) ||
	    !rdpei_read_4byte_signed(s, &contactData->x) ||
	    !rdpei_read_4byte_signed(s, &contactData->y) ||
	    !rdpei_read_4byte_unsigned(s, &contactData->contactFlags))
	{
		WLog_ERR(TAG, "rdpei_read_ failed!");
		return ERROR_INTERNAL_ERROR;
	}

	if (contactData->fieldsPresent & CONTACT_DATA_CONTACTRECT_PRESENT)
	{
		if (!rdpei_read_2byte_signed(s, &contactData->contactRectLeft) ||
		    !rdpei_read_2byte_signed(s, &contactData->contactRectTop) ||
		    !rdpei_read_2byte_signed(s, &contactData->contactRectRight) ||
		    !rdpei_read_2byte_signed(s, &contactData->contactRectBottom))
		{
			WLog_ERR(TAG, "rdpei_read_ failed!");
			return ERROR_INTERNAL_ERROR;
		}
	}

	if ((contactData->fieldsPresent & CONTACT_DATA_ORIENTATION_PRESENT) &&
	    !rdpei_read_4byte_unsigned(s, &contactData->orientation))
	{
		WLog_ERR(TAG, "rdpei_read_ failed!");
		return ERROR_INTERNAL_ERROR;
	}

	if ((contactData->fieldsPresent & CONTACT_DATA_PRESSURE_PRESENT) &&
	    !rdpei_read_4byte_unsigned(s, &contactData->pressure))
	{
		WLog_ERR(TAG, "rdpei_read_ failed!");
		return ERROR_INTERNAL_ERROR;
	}

	return CHANNEL_RC_OK;
}

static UINT read_pen_contact(RdpeiServerContext* context, wStream* s,
                             RDPINPUT_PEN_CONTACT* contactData)
{
	WINPR_UNUSED(context);
	if (!Stream_CheckAndLogRequiredLength(TAG, s, 1))
		return ERROR_INVALID_DATA;

	Stream_Read_UINT8(s, contactData->deviceId);
	if (!rdpei_read_2byte_unsigned(s, &contactData->fieldsPresent) ||
	    !rdpei_read_4byte_signed(s, &contactData->x) ||
	    !rdpei_read_4byte_signed(s, &contactData->y) ||
	    !rdpei_read_4byte_unsigned(s, &contactData->contactFlags))
	{
		WLog_ERR(TAG, "rdpei_read_ failed!");
		return ERROR_INTERNAL_ERROR;
	}

	if (contactData->fieldsPresent & RDPINPUT_PEN_CONTACT_PENFLAGS_PRESENT)
	{
		if (!rdpei_read_4byte_unsigned(s, &contactData->penFlags))
			return ERROR_INVALID_DATA;
	}
	if (contactData->fieldsPresent & RDPINPUT_PEN_CONTACT_PRESSURE_PRESENT)
	{
		if (!rdpei_read_4byte_unsigned(s, &contactData->pressure))
			return ERROR_INVALID_DATA;
	}
	if (contactData->fieldsPresent & RDPINPUT_PEN_CONTACT_ROTATION_PRESENT)
	{
		if (!rdpei_read_2byte_unsigned(s, &contactData->rotation))
			return ERROR_INVALID_DATA;
	}
	if (contactData->fieldsPresent & RDPINPUT_PEN_CONTACT_TILTX_PRESENT)
	{
		if (!rdpei_read_2byte_signed(s, &contactData->tiltX))
			return ERROR_INVALID_DATA;
	}
	if (contactData->fieldsPresent & RDPINPUT_PEN_CONTACT_TILTY_PRESENT)
	{
		if (!rdpei_read_2byte_signed(s, &contactData->tiltY))
			return ERROR_INVALID_DATA;
	}

	return CHANNEL_RC_OK;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
static UINT read_touch_frame(RdpeiServerContext* context, wStream* s, RDPINPUT_TOUCH_FRAME* frame)
{
	RDPINPUT_CONTACT_DATA* contact = NULL;
	UINT error = 0;

	if (!rdpei_read_2byte_unsigned(s, &frame->contactCount) ||
	    !rdpei_read_8byte_unsigned(s, &frame->frameOffset))
	{
		WLog_ERR(TAG, "rdpei_read_ failed!");
		return ERROR_INTERNAL_ERROR;
	}

	frame->contacts = contact = calloc(frame->contactCount, sizeof(RDPINPUT_CONTACT_DATA));
	if (!frame->contacts)
	{
		WLog_ERR(TAG, "calloc failed!");
		return CHANNEL_RC_NO_MEMORY;
	}

	for (UINT32 i = 0; i < frame->contactCount; i++, contact++)
	{
		if ((error = read_touch_contact_data(context, s, contact)))
		{
			WLog_ERR(TAG, "read_touch_contact_data failed with error %" PRIu32 "!", error);
			frame->contactCount = WINPR_ASSERTING_INT_CAST(UINT16, i);
			touch_frame_reset(frame);
			return error;
		}
	}
	return CHANNEL_RC_OK;
}

static UINT read_pen_frame(RdpeiServerContext* context, wStream* s, RDPINPUT_PEN_FRAME* frame)
{
	RDPINPUT_PEN_CONTACT* contact = NULL;
	UINT error = 0;

	if (!rdpei_read_2byte_unsigned(s, &frame->contactCount) ||
	    !rdpei_read_8byte_unsigned(s, &frame->frameOffset))
	{
		WLog_ERR(TAG, "rdpei_read_ failed!");
		return ERROR_INTERNAL_ERROR;
	}

	frame->contacts = contact = calloc(frame->contactCount, sizeof(RDPINPUT_PEN_CONTACT));
	if (!frame->contacts)
	{
		WLog_ERR(TAG, "calloc failed!");
		return CHANNEL_RC_NO_MEMORY;
	}

	for (UINT32 i = 0; i < frame->contactCount; i++, contact++)
	{
		if ((error = read_pen_contact(context, s, contact)))
		{
			WLog_ERR(TAG, "read_touch_contact_data failed with error %" PRIu32 "!", error);
			frame->contactCount = WINPR_ASSERTING_INT_CAST(UINT16, i);

			pen_frame_reset(frame);
			return error;
		}
	}
	return CHANNEL_RC_OK;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
static UINT read_touch_event(RdpeiServerContext* context, wStream* s)
{
	UINT16 frameCount = 0;
	RDPINPUT_TOUCH_EVENT* event = &context->priv->touchEvent;
	RDPINPUT_TOUCH_FRAME* frame = NULL;
	UINT error = CHANNEL_RC_OK;

	if (!rdpei_read_4byte_unsigned(s, &event->encodeTime) ||
	    !rdpei_read_2byte_unsigned(s, &frameCount))
	{
		WLog_ERR(TAG, "rdpei_read_ failed!");
		return ERROR_INTERNAL_ERROR;
	}

	event->frameCount = frameCount;
	event->frames = frame = calloc(event->frameCount, sizeof(RDPINPUT_TOUCH_FRAME));
	if (!event->frames)
	{
		WLog_ERR(TAG, "calloc failed!");
		return CHANNEL_RC_NO_MEMORY;
	}

	for (UINT32 i = 0; i < frameCount; i++, frame++)
	{
		if ((error = read_touch_frame(context, s, frame)))
		{
			WLog_ERR(TAG, "read_touch_contact_data failed with error %" PRIu32 "!", error);
			event->frameCount = WINPR_ASSERTING_INT_CAST(UINT16, i);

			goto out_cleanup;
		}
	}

	IFCALLRET(context->onTouchEvent, error, context, event);
	if (error)
		WLog_ERR(TAG, "context->onTouchEvent failed with error %" PRIu32 "", error);

out_cleanup:
	touch_event_reset(event);
	return error;
}

static UINT read_pen_event(RdpeiServerContext* context, wStream* s)
{
	UINT16 frameCount = 0;
	RDPINPUT_PEN_EVENT* event = &context->priv->penEvent;
	RDPINPUT_PEN_FRAME* frame = NULL;
	UINT error = CHANNEL_RC_OK;

	if (!rdpei_read_4byte_unsigned(s, &event->encodeTime) ||
	    !rdpei_read_2byte_unsigned(s, &frameCount))
	{
		WLog_ERR(TAG, "rdpei_read_ failed!");
		return ERROR_INTERNAL_ERROR;
	}

	event->frameCount = frameCount;
	event->frames = frame = calloc(event->frameCount, sizeof(RDPINPUT_PEN_FRAME));
	if (!event->frames)
	{
		WLog_ERR(TAG, "calloc failed!");
		return CHANNEL_RC_NO_MEMORY;
	}

	for (UINT32 i = 0; i < frameCount; i++, frame++)
	{
		if ((error = read_pen_frame(context, s, frame)))
		{
			WLog_ERR(TAG, "read_pen_frame failed with error %" PRIu32 "!", error);
			event->frameCount = WINPR_ASSERTING_INT_CAST(UINT16, i);

			goto out_cleanup;
		}
	}

	IFCALLRET(context->onPenEvent, error, context, event);
	if (error)
		WLog_ERR(TAG, "context->onPenEvent failed with error %" PRIu32 "", error);

out_cleanup:
	pen_event_reset(event);
	return error;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
static UINT read_dismiss_hovering_contact(RdpeiServerContext* context, wStream* s)
{
	BYTE contactId = 0;
	UINT error = CHANNEL_RC_OK;

	if (!Stream_CheckAndLogRequiredLength(TAG, s, 1))
		return ERROR_INVALID_DATA;

	Stream_Read_UINT8(s, contactId);

	IFCALLRET(context->onTouchReleased, error, context, contactId);
	if (error)
		WLog_ERR(TAG, "context->onTouchReleased failed with error %" PRIu32 "", error);

	return error;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
UINT rdpei_server_handle_messages(RdpeiServerContext* context)
{
	DWORD bytesReturned = 0;
	RdpeiServerPrivate* priv = context->priv;
	wStream* s = priv->inputStream;
	UINT error = CHANNEL_RC_OK;

	if (!WTSVirtualChannelRead(priv->channelHandle, 0, Stream_Pointer(s), priv->expectedBytes,
	                           &bytesReturned))
	{
		if (GetLastError() == ERROR_NO_DATA)
			return ERROR_READ_FAULT;

		WLog_DBG(TAG, "channel connection closed");
		return CHANNEL_RC_OK;
	}
	priv->expectedBytes -= bytesReturned;
	Stream_Seek(s, bytesReturned);

	if (priv->expectedBytes)
		return CHANNEL_RC_OK;

	Stream_SealLength(s);
	Stream_SetPosition(s, 0);

	if (priv->waitingHeaders)
	{
		UINT32 pduLen = 0;

		/* header case */
		Stream_Read_UINT16(s, priv->currentMsgType);
		Stream_Read_UINT32(s, pduLen);

		if (pduLen < RDPINPUT_HEADER_LENGTH)
		{
			WLog_ERR(TAG, "invalid pduLength %" PRIu32 "", pduLen);
			return ERROR_INVALID_DATA;
		}
		priv->expectedBytes = pduLen - RDPINPUT_HEADER_LENGTH;
		priv->waitingHeaders = FALSE;
		Stream_SetPosition(s, 0);
		if (priv->expectedBytes)
		{
			if (!Stream_EnsureCapacity(s, priv->expectedBytes))
			{
				WLog_ERR(TAG, "Stream_EnsureCapacity failed!");
				return CHANNEL_RC_NO_MEMORY;
			}
			return CHANNEL_RC_OK;
		}
	}

	/* when here we have the header + the body */
	switch (priv->currentMsgType)
	{
		case EVENTID_CS_READY:
			if (priv->automataState != STATE_WAITING_CLIENT_READY)
			{
				WLog_ERR(TAG, "not expecting a CS_READY packet in this state(%d)",
				         priv->automataState);
				return ERROR_INVALID_STATE;
			}

			if ((error = read_cs_ready_message(context, s)))
			{
				WLog_ERR(TAG, "read_cs_ready_message failed with error %" PRIu32 "", error);
				return error;
			}
			break;

		case EVENTID_TOUCH:
			if ((error = read_touch_event(context, s)))
			{
				WLog_ERR(TAG, "read_touch_event failed with error %" PRIu32 "", error);
				return error;
			}
			break;
		case EVENTID_DISMISS_HOVERING_CONTACT:
			if ((error = read_dismiss_hovering_contact(context, s)))
			{
				WLog_ERR(TAG, "read_dismiss_hovering_contact failed with error %" PRIu32 "", error);
				return error;
			}
			break;
		case EVENTID_PEN:
			if ((error = read_pen_event(context, s)))
			{
				WLog_ERR(TAG, "read_pen_event failed with error %" PRIu32 "", error);
				return error;
			}
			break;
		default:
			WLog_ERR(TAG, "unexpected message type 0x%" PRIx16 "", priv->currentMsgType);
	}

	Stream_SetPosition(s, 0);
	priv->waitingHeaders = TRUE;
	priv->expectedBytes = RDPINPUT_HEADER_LENGTH;
	return error;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
UINT rdpei_server_send_sc_ready(RdpeiServerContext* context, UINT32 version, UINT32 features)
{
	ULONG written = 0;
	RdpeiServerPrivate* priv = context->priv;
	UINT32 pduLen = 4;

	if (priv->automataState != STATE_INITIAL)
	{
		WLog_ERR(TAG, "called from unexpected state %d", priv->automataState);
		return ERROR_INVALID_STATE;
	}

	Stream_SetPosition(priv->outputStream, 0);

	if (version >= RDPINPUT_PROTOCOL_V300)
		pduLen += 4;

	if (!Stream_EnsureCapacity(priv->outputStream, RDPINPUT_HEADER_LENGTH + pduLen))
	{
		WLog_ERR(TAG, "Stream_EnsureCapacity failed!");
		return CHANNEL_RC_NO_MEMORY;
	}

	Stream_Write_UINT16(priv->outputStream, EVENTID_SC_READY);
	Stream_Write_UINT32(priv->outputStream, RDPINPUT_HEADER_LENGTH + pduLen);
	Stream_Write_UINT32(priv->outputStream, version);
	if (version >= RDPINPUT_PROTOCOL_V300)
		Stream_Write_UINT32(priv->outputStream, features);

	const size_t pos = Stream_GetPosition(priv->outputStream);

	WINPR_ASSERT(pos <= UINT32_MAX);
	if (!WTSVirtualChannelWrite(priv->channelHandle, Stream_BufferAs(priv->outputStream, char),
	                            (ULONG)pos, &written))
	{
		WLog_ERR(TAG, "WTSVirtualChannelWrite failed!");
		return ERROR_INTERNAL_ERROR;
	}

	priv->automataState = STATE_WAITING_CLIENT_READY;
	return CHANNEL_RC_OK;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
UINT rdpei_server_suspend(RdpeiServerContext* context)
{
	ULONG written = 0;
	RdpeiServerPrivate* priv = context->priv;

	switch (priv->automataState)
	{
		case STATE_SUSPENDED:
			WLog_ERR(TAG, "already suspended");
			return CHANNEL_RC_OK;
		case STATE_WAITING_FRAME:
			break;
		default:
			WLog_ERR(TAG, "called from unexpected state %d", priv->automataState);
			return ERROR_INVALID_STATE;
	}

	Stream_SetPosition(priv->outputStream, 0);
	if (!Stream_EnsureCapacity(priv->outputStream, RDPINPUT_HEADER_LENGTH))
	{
		WLog_ERR(TAG, "Stream_EnsureCapacity failed!");
		return CHANNEL_RC_NO_MEMORY;
	}

	Stream_Write_UINT16(priv->outputStream, EVENTID_SUSPEND_TOUCH);
	Stream_Write_UINT32(priv->outputStream, RDPINPUT_HEADER_LENGTH);

	const size_t pos = Stream_GetPosition(priv->outputStream);

	WINPR_ASSERT(pos <= UINT32_MAX);
	if (!WTSVirtualChannelWrite(priv->channelHandle, Stream_BufferAs(priv->outputStream, char),
	                            (ULONG)pos, &written))
	{
		WLog_ERR(TAG, "WTSVirtualChannelWrite failed!");
		return ERROR_INTERNAL_ERROR;
	}

	priv->automataState = STATE_SUSPENDED;
	return CHANNEL_RC_OK;
}

/**
 * Function description
 *
 * @return 0 on success, otherwise a Win32 error code
 */
UINT rdpei_server_resume(RdpeiServerContext* context)
{
	ULONG written = 0;
	RdpeiServerPrivate* priv = context->priv;

	switch (priv->automataState)
	{
		case STATE_WAITING_FRAME:
			WLog_ERR(TAG, "not suspended");
			return CHANNEL_RC_OK;
		case STATE_SUSPENDED:
			break;
		default:
			WLog_ERR(TAG, "called from unexpected state %d", priv->automataState);
			return ERROR_INVALID_STATE;
	}

	Stream_SetPosition(priv->outputStream, 0);
	if (!Stream_EnsureCapacity(priv->outputStream, RDPINPUT_HEADER_LENGTH))
	{
		WLog_ERR(TAG, "Stream_EnsureCapacity failed!");
		return CHANNEL_RC_NO_MEMORY;
	}

	Stream_Write_UINT16(priv->outputStream, EVENTID_RESUME_TOUCH);
	Stream_Write_UINT32(priv->outputStream, RDPINPUT_HEADER_LENGTH);

	const size_t pos = Stream_GetPosition(priv->outputStream);

	WINPR_ASSERT(pos <= UINT32_MAX);
	if (!WTSVirtualChannelWrite(priv->channelHandle, Stream_BufferAs(priv->outputStream, char),
	                            (ULONG)pos, &written))
	{
		WLog_ERR(TAG, "WTSVirtualChannelWrite failed!");
		return ERROR_INTERNAL_ERROR;
	}

	priv->automataState = STATE_WAITING_FRAME;
	return CHANNEL_RC_OK;
}
