/* pdf417_tabs.h - PDF417 tables and coefficients */

/*
    libzint - the open source barcode library
    Copyright (C) 2008-2022 Robin Stuart <rstuart114@gmail.com>
    Portions Copyright (C) 2004 Grandzebu

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */

/* this file contains the character table, the pre-calculated coefficients and the
   codeword patterns taken from lines 416 to 454 of pdf417.frm */

/* See "pdf417.h" for declarations */

#ifndef Z_PDF417_TABS_H
#define Z_PDF417_TABS_H

/* PDF417 error correction coefficients from Grand Zebu */
INTERNAL_DATA const unsigned short pdf_coefrs[1022] = {
    /* k = 2 */
    27, 917,

    /* k = 4 */
    522, 568, 723, 809,

    /* k = 8 */
    237, 308, 436, 284, 646, 653, 428, 379,

    /* k = 16 */
    274, 562, 232, 755, 599, 524, 801, 132, 295, 116, 442, 428, 295, 42, 176, 65,

    /* k = 32 */
    361, 575, 922, 525, 176, 586, 640, 321, 536, 742, 677, 742, 687, 284, 193, 517,
    273, 494, 263, 147, 593, 800, 571, 320, 803, 133, 231, 390, 685, 330, 63, 410,

    /* k = 64 */
    539, 422, 6, 93, 862, 771, 453, 106, 610, 287, 107, 505, 733, 877, 381, 612,
    723, 476, 462, 172, 430, 609, 858, 822, 543, 376, 511, 400, 672, 762, 283, 184,
    440, 35, 519, 31, 460, 594, 225, 535, 517, 352, 605, 158, 651, 201, 488, 502,
    648, 733, 717, 83, 404, 97, 280, 771, 840, 629, 4, 381, 843, 623, 264, 543,

    /* k = 128 */
    521, 310, 864, 547, 858, 580, 296, 379, 53, 779, 897, 444, 400, 925, 749, 415,
    822, 93, 217, 208, 928, 244, 583, 620, 246, 148, 447, 631, 292, 908, 490, 704,
    516, 258, 457, 907, 594, 723, 674, 292, 272, 96, 684, 432, 686, 606, 860, 569,
    193, 219, 129, 186, 236, 287, 192, 775, 278, 173, 40, 379, 712, 463, 646, 776,
    171, 491, 297, 763, 156, 732, 95, 270, 447, 90, 507, 48, 228, 821, 808, 898,
    784, 663, 627, 378, 382, 262, 380, 602, 754, 336, 89, 614, 87, 432, 670, 616,
    157, 374, 242, 726, 600, 269, 375, 898, 845, 454, 354, 130, 814, 587, 804, 34,
    211, 330, 539, 297, 827, 865, 37, 517, 834, 315, 550, 86, 801, 4, 108, 539,

    /* k = 256 */
    524, 894, 75, 766, 882, 857, 74, 204, 82, 586, 708, 250, 905, 786, 138, 720,
    858, 194, 311, 913, 275, 190, 375, 850, 438, 733, 194, 280, 201, 280, 828, 757,
    710, 814, 919, 89, 68, 569, 11, 204, 796, 605, 540, 913, 801, 700, 799, 137,
    439, 418, 592, 668, 353, 859, 370, 694, 325, 240, 216, 257, 284, 549, 209, 884,
    315, 70, 329, 793, 490, 274, 877, 162, 749, 812, 684, 461, 334, 376, 849, 521,
    307, 291, 803, 712, 19, 358, 399, 908, 103, 511, 51, 8, 517, 225, 289, 470,
    637, 731, 66, 255, 917, 269, 463, 830, 730, 433, 848, 585, 136, 538, 906, 90,
    2, 290, 743, 199, 655, 903, 329, 49, 802, 580, 355, 588, 188, 462, 10, 134,
    628, 320, 479, 130, 739, 71, 263, 318, 374, 601, 192, 605, 142, 673, 687, 234,
    722, 384, 177, 752, 607, 640, 455, 193, 689, 707, 805, 641, 48, 60, 732, 621,
    895, 544, 261, 852, 655, 309, 697, 755, 756, 60, 231, 773, 434, 421, 726, 528,
    503, 118, 49, 795, 32, 144, 500, 238, 836, 394, 280, 566, 319, 9, 647, 550,
    73, 914, 342, 126, 32, 681, 331, 792, 620, 60, 609, 441, 180, 791, 893, 754,
    605, 383, 228, 749, 760, 213, 54, 297, 134, 54, 834, 299, 922, 191, 910, 532,
    609, 829, 189, 20, 167, 29, 872, 449, 83, 402, 41, 656, 505, 579, 481, 173,
    404, 251, 688, 95, 497, 555, 642, 543, 307, 159, 924, 558, 648, 55, 497, 10,

    /* k = 512 */
    352, 77, 373, 504, 35, 599, 428, 207, 409, 574, 118, 498, 285, 380, 350, 492,
    197, 265, 920, 155, 914, 299, 229, 643, 294, 871, 306, 88, 87, 193, 352, 781,
    846, 75, 327, 520, 435, 543, 203, 666, 249, 346, 781, 621, 640, 268, 794, 534,
    539, 781, 408, 390, 644, 102, 476, 499, 290, 632, 545, 37, 858, 916, 552, 41,
    542, 289, 122, 272, 383, 800, 485, 98, 752, 472, 761, 107, 784, 860, 658, 741,
    290, 204, 681, 407, 855, 85, 99, 62, 482, 180, 20, 297, 451, 593, 913, 142,
    808, 684, 287, 536, 561, 76, 653, 899, 729, 567, 744, 390, 513, 192, 516, 258,
    240, 518, 794, 395, 768, 848, 51, 610, 384, 168, 190, 826, 328, 596, 786, 303,
    570, 381, 415, 641, 156, 237, 151, 429, 531, 207, 676, 710, 89, 168, 304, 402,
    40, 708, 575, 162, 864, 229, 65, 861, 841, 512, 164, 477, 221, 92, 358, 785,
    288, 357, 850, 836, 827, 736, 707, 94, 8, 494, 114, 521, 2, 499, 851, 543,
    152, 729, 771, 95, 248, 361, 578, 323, 856, 797, 289, 51, 684, 466, 533, 820,
    669, 45, 902, 452, 167, 342, 244, 173, 35, 463, 651, 51, 699, 591, 452, 578,
    37, 124, 298, 332, 552, 43, 427, 119, 662, 777, 475, 850, 764, 364, 578, 911,
    283, 711, 472, 420, 245, 288, 594, 394, 511, 327, 589, 777, 699, 688, 43, 408,
    842, 383, 721, 521, 560, 644, 714, 559, 62, 145, 873, 663, 713, 159, 672, 729,
    624, 59, 193, 417, 158, 209, 563, 564, 343, 693, 109, 608, 563, 365, 181, 772,
    677, 310, 248, 353, 708, 410, 579, 870, 617, 841, 632, 860, 289, 536, 35, 777,
    618, 586, 424, 833, 77, 597, 346, 269, 757, 632, 695, 751, 331, 247, 184, 45,
    787, 680, 18, 66, 407, 369, 54, 492, 228, 613, 830, 922, 437, 519, 644, 905,
    789, 420, 305, 441, 207, 300, 892, 827, 141, 537, 381, 662, 513, 56, 252, 341,
    242, 797, 838, 837, 720, 224, 307, 631, 61, 87, 560, 310, 756, 665, 397, 808,
    851, 309, 473, 795, 378, 31, 647, 915, 459, 806, 590, 731, 425, 216, 548, 249,
    321, 881, 699, 535, 673, 782, 210, 815, 905, 303, 843, 922, 281, 73, 469, 791,
    660, 162, 498, 308, 155, 422, 907, 817, 187, 62, 16, 425, 535, 336, 286, 437,
    375, 273, 610, 296, 183, 923, 116, 667, 751, 353, 62, 366, 691, 379, 687, 842,
    37, 357, 720, 742, 330, 5, 39, 923, 311, 424, 242, 749, 321, 54, 669, 316,
    342, 299, 534, 105, 667, 488, 640, 672, 576, 540, 316, 486, 721, 610, 46, 656,
    447, 171, 616, 464, 190, 531, 297, 321, 762, 752, 533, 175, 134, 14, 381, 433,
    717, 45, 111, 20, 596, 284, 736, 138, 646, 411, 877, 669, 141, 919, 45, 780,
    407, 164, 332, 899, 165, 726, 600, 325, 498, 655, 357, 752, 768, 223, 849, 647,
    63, 310, 863, 251, 366, 304, 282, 738, 675, 410, 389, 244, 31, 121, 303, 263
};

INTERNAL_DATA const unsigned short pdf_bitpattern[2787] = {
    0xEAE0, 0xF578, 0xFABE, 0xEA70, 0xF53C, 0xFA9F, 0xD460, 0xEA38, 0xD430, 0xA820,
    0xD418, 0xA810, 0xD6E0, 0xEB78, 0xF5BE, 0xD670, 0xEB3C, 0xF59F, 0xAC60, 0xD638,
    0xAC30, 0xAEE0, 0xD778, 0xEBBE, 0xAE70, 0xD73C, 0xEB9F, 0xAE38, 0xD71E, 0xAF78,
    0xD7BE, 0xAF3C, 0xD79F, 0xAFBE, 0xFAFD, 0xE970, 0xF4BC, 0xFA5F, 0xD260, 0xE938,
    0xF49E, 0xD230, 0xE91C, 0xA420, 0xD218, 0xE90E, 0xA410, 0xD20C, 0xA408, 0xD370,
    0xE9BC, 0xF4DF, 0xA660, 0xD338, 0xE99E, 0xA630, 0xD31C, 0xE98F, 0xA618, 0xD30E,
    0xA770, 0xD3BC, 0xE9DF, 0xA738, 0xD39E, 0xA71C, 0xD38F, 0xA7BC, 0xD3DF, 0xA79E,
    0xA78F, 0xD160, 0xE8B8, 0xF45E, 0xD130, 0xE89C, 0xF44F, 0xA220, 0xD118, 0xE88E,
    0xA210, 0xD10C, 0xA208, 0xA204, 0xA360, 0xD1B8, 0xE8DE, 0xA330, 0xD19C, 0xE8CF,
    0xA318, 0xD18E, 0xA30C, 0xA306, 0xA3B8, 0xD1DE, 0xA39C, 0xD1CF, 0xA38E, 0xA3DE,
    0xD0B0, 0xE85C, 0xF42F, 0xA120, 0xD098, 0xE84E, 0xA110, 0xD08C, 0xE847, 0xA108,
    0xD086, 0xA104, 0xD083, 0xA1B0, 0xD0DC, 0xE86F, 0xA198, 0xD0CE, 0xA18C, 0xD0C7,
    0xA186, 0xA183, 0xD0EF, 0xA1C7, 0xA0A0, 0xD058, 0xE82E, 0xA090, 0xD04C, 0xE827,
    0xA088, 0xD046, 0xA084, 0xD043, 0xA082, 0xA0D8, 0xA0CC, 0xA0C6, 0xA050, 0xE817,
    0xD026, 0xD023, 0xA041, 0xE570, 0xF2BC, 0xF95F, 0xCA60, 0xE538, 0xF29E, 0xCA30,
    0xE51C, 0xF28F, 0x9420, 0xCA18, 0x9410, 0xCB70, 0xE5BC, 0xF2DF, 0x9660, 0xCB38,
    0xE59E, 0x9630, 0xCB1C, 0x9618, 0x960C, 0x9770, 0xCBBC, 0xE5DF, 0x9738, 0xCB9E,
    0x971C, 0x970E, 0x97BC, 0xCBDF, 0x979E, 0x97DF, 0xED60, 0xF6B8, 0xFB5E, 0xED30,
    0xF69C, 0xFB4F, 0xDA20, 0xED18, 0xF68E, 0xDA10, 0xED0C, 0xF687, 0xDA08, 0xED06,
    0xC960, 0xE4B8, 0xF25E, 0xDB60, 0xC930, 0xE49C, 0xF24F, 0xDB30, 0xED9C, 0xF6CF,
    0xB620, 0x9210, 0xC90C, 0xE487, 0xB610, 0xDB0C, 0xB608, 0x9360, 0xC9B8, 0xE4DE,
    0xB760, 0x9330, 0xC99C, 0xE4CF, 0xB730, 0xDB9C, 0xEDCF, 0xB718, 0x930C, 0xB70C,
    0x93B8, 0xC9DE, 0xB7B8, 0x939C, 0xC9CF, 0xB79C, 0xDBCF, 0xB78E, 0x93DE, 0xB7DE,
    0x93CF, 0xB7CF, 0xECB0, 0xF65C, 0xFB2F, 0xD920, 0xEC98, 0xF64E, 0xD910, 0xEC8C,
    0xF647, 0xD908, 0xEC86, 0xD904, 0xD902, 0xC8B0, 0xE45C, 0xF22F, 0xD9B0, 0xC898,
    0xE44E, 0xB320, 0x9110, 0xECCE, 0xE447, 0xB310, 0x9108, 0xC886, 0xB308, 0xD986,
    0xC883, 0x9102, 0x91B0, 0xC8DC, 0xE46F, 0xB3B0, 0x9198, 0xC8CE, 0xB398, 0xD9CE,
    0xC8C7, 0xB38C, 0x9186, 0x9183, 0x91DC, 0xC8EF, 0xB3DC, 0x91CE, 0xB3CE, 0x91C7,
    0xB3C7, 0xB3EF, 0xD8A0, 0xEC58, 0xF62E, 0xD890, 0xEC4C, 0xF627, 0xD888, 0xEC46,
    0xD884, 0xEC43, 0xD882, 0xD881, 0x90A0, 0xC858, 0xE42E, 0xB1A0, 0x9090, 0xC84C,
    0xE427, 0xB190, 0xD8CC, 0xEC67, 0xB188, 0x9084, 0xC843, 0xB184, 0xD8C3, 0xB182,
    0x90D8, 0xC86E, 0xB1D8, 0x90CC, 0xC867, 0xB1CC, 0xD8E7, 0xB1C6, 0x90C3, 0xB1C3,
    0xB1EE, 0xB1E7, 0xD850, 0xEC2C, 0xF617, 0xD848, 0xEC26, 0xD844, 0xEC23, 0xD842,
    0xD841, 0x9050, 0xC82C, 0xE417, 0xB0D0, 0x9048, 0xC826, 0xB0C8, 0xD866, 0xC823,
    0xB0C4, 0x9042, 0xB0C2, 0x9041, 0x906C, 0xB0EC, 0xB0E6, 0xB0E3, 0xEC16, 0xEC13,
    0xD821, 0xC816, 0x9024, 0xB064, 0xB062, 0xB061, 0xC560, 0xE2B8, 0xF15E, 0xC530,
    0xE29C, 0x8A20, 0xC518, 0xE28E, 0x8A10, 0xC50C, 0x8A08, 0x8A04, 0x8B60, 0xC5B8,
    0xE2DE, 0x8B30, 0xC59C, 0xE2CF, 0x8B18, 0xC58E, 0x8B0C, 0x8B06, 0x8BB8, 0xC5DE,
    0x8B9C, 0xC5CF, 0x8B8E, 0x8BDE, 0x8BCF, 0xE6B0, 0xF35C, 0xF9AF, 0xCD20, 0xE698,
    0xF34E, 0xCD10, 0xE68C, 0xF347, 0xCD08, 0xE686, 0xCD04, 0xE683, 0xC4B0, 0xE25C,
    0xF12F, 0xCDB0, 0xC498, 0xE24E, 0x9B20, 0x8910, 0xE6CE, 0xE247, 0x9B10, 0xCD8C,
    0xC486, 0x9B08, 0x8904, 0x9B04, 0x89B0, 0xC4DC, 0xE26F, 0x9BB0, 0x8998, 0xE6EF,
    0x9B98, 0xCDCE, 0xC4C7, 0x9B8C, 0x8986, 0x9B86, 0x89DC, 0xC4EF, 0x9BDC, 0x89CE,
    0x9BCE, 0x89C7, 0x89EF, 0x9BEF, 0xEEA0, 0xF758, 0xFBAE, 0xEE90, 0xF74C, 0xFBA7,
    0xEE88, 0xF746, 0xEE84, 0xF743, 0xEE82, 0xCCA0, 0xE658, 0xF32E, 0xDDA0, 0xCC90,
    0xF76E, 0xF327, 0xDD90, 0xEECC, 0xF767, 0xDD88, 0xCC84, 0xE643, 0xDD84, 0xEEC3,
    0xCC81, 0x88A0, 0xC458, 0xE22E, 0x99A0, 0x8890, 0xC44C, 0xE227, 0xBBA0, 0x9990,
    0xCCCC, 0xE667, 0xBB90, 0xDDCC, 0xEEE7, 0xC443, 0xBB88, 0x9984, 0xCCC3, 0xBB84,
    0x8881, 0x88D8, 0xC46E, 0x99D8, 0x88CC, 0xC467, 0xBBD8, 0x99CC, 0xCCE7, 0xBBCC,
    0xDDE7, 0x88C3, 0x99C3, 0x88EE, 0x99EE, 0x88E7, 0xBBEE, 0x99E7, 0xEE50, 0xF72C,
    0xFB97, 0xEE48, 0xF726, 0xEE44, 0xF723, 0xEE42, 0xEE41, 0xCC50, 0xE62C, 0xF317,
    0xDCD0, 0xCC48, 0xF737, 0xDCC8, 0xEE66, 0xE623, 0xDCC4, 0xCC42, 0xDCC2, 0xCC41,
    0xDCC1, 0x8850, 0xC42C, 0xE217, 0x98D0, 0x8848, 0xC426, 0xB9D0, 0x98C8, 0xCC66,
    0xC423, 0xB9C8, 0xDCE6, 0x8842, 0xB9C4, 0x98C2, 0x8841, 0x98C1, 0x886C, 0xC437,
    0x98EC, 0x8866, 0xB9EC, 0x98E6, 0x8863, 0xB9E6, 0x98E3, 0x8877, 0xB9F7, 0xEE28,
    0xF716, 0xEE24, 0xF713, 0xEE22, 0xEE21, 0xCC28, 0xE616, 0xDC68, 0xCC24, 0xE613,
    0xDC64, 0xEE33, 0xDC62, 0xCC21, 0xDC61, 0x8828, 0xC416, 0x9868, 0x8824, 0xC413,
    0xB8E8, 0x9864, 0xCC33, 0xB8E4, 0xDC73, 0x8821, 0xB8E2, 0x9861, 0xB8E1, 0x9876,
    0xB8F6, 0xB8F3, 0xF70B, 0xEE11, 0xE60B, 0xCC12, 0xCC11, 0x8814, 0x9834, 0xB874,
    0x8811, 0x9831, 0xC2B0, 0x8520, 0xC298, 0x8510, 0xC28C, 0xE147, 0x8508, 0xC286,
    0x8504, 0xC283, 0x85B0, 0xC2DC, 0xE16F, 0x8598, 0xC2CE, 0x858C, 0xC2C7, 0x8586,
    0x8583, 0x85DC, 0xC2EF, 0x85CE, 0x85C7, 0x85EF, 0xC6A0, 0xE358, 0xF1AE, 0xC690,
    0xE34C, 0xC688, 0xE346, 0xC684, 0xE343, 0xC682, 0x84A0, 0xC258, 0xE12E, 0x8DA0,
    0x8490, 0xE36E, 0xE127, 0x8D90, 0xC6CC, 0xE367, 0x8D88, 0x8484, 0xC243, 0x8D84,
    0xC6C3, 0x8481, 0x84D8, 0xC26E, 0x8DD8, 0x84CC, 0xC267, 0x8DCC, 0xC6E7, 0x8DC6,
    0x84C3, 0x84EE, 0x8DEE, 0x84E7, 0x8DE7, 0xE750, 0xF3AC, 0xF9D7, 0xE748, 0xF3A6,
    0xE744, 0xF3A3, 0xE742, 0xE741, 0xC650, 0xE32C, 0xCED0, 0xC648, 0xE326, 0xCEC8,
    0xE766, 0xE323, 0xCEC4, 0xC642, 0xCEC2, 0xC641, 0xCEC1, 0x8450, 0xC22C, 0x8CD0,
    0x8448, 0xE337, 0x9DD0, 0x8CC8, 0xC666, 0xC223, 0x9DC8, 0xCEE6, 0x8442, 0x9DC4,
    0x8CC2, 0x8441, 0x8CC1, 0x846C, 0xC237, 0x8CEC, 0x8466, 0x9DEC, 0x8CE6, 0x8463,
    0x9DE6, 0x8CE3, 0x8477, 0x8CF7, 0x9DF7, 0xF7A8, 0xFBD6, 0xF7A4, 0xFBD3, 0xF7A2,
    0xF7A1, 0xE728, 0xF396, 0xEF68, 0xF7B6, 0xF393, 0xEF64, 0xF7B3, 0xEF62, 0xE721,
    0xEF61, 0xC628, 0xE316, 0xCE68, 0xC624, 0xE313, 0xDEE8, 0xCE64, 0xE733, 0xDEE4,
    0xEF73, 0xC621, 0xDEE2, 0xCE61, 0xDEE1, 0x8428, 0xC216, 0x8C68, 0x8424, 0xC213,
    0x9CE8, 0x8C64, 0xC633, 0xBDE8, 0x9CE4, 0xCE73, 0x8421, 0xBDE4, 0xDEF3, 0x8C61,
    0xBDE2, 0x8436, 0x8C76, 0x8433, 0x9CF6, 0x8C73, 0xBDF6, 0x9CF3, 0xBDF3, 0xF794,
    0xFBCB, 0xF792, 0xF791, 0xE714, 0xF38B, 0xEF34, 0xF79B, 0xEF32, 0xE711, 0xEF31,
    0xC614, 0xE30B, 0xCE34, 0xC612, 0xDE74, 0xCE32, 0xC611, 0xDE72, 0xCE31, 0xDE71,
    0x8414, 0xC20B, 0x8C34, 0xC61B, 0x9C74, 0x8C32, 0x8411, 0xBCF4, 0x9C72, 0x8C31,
    0xBCF2, 0x9C71, 0xBCF1, 0x8C3B, 0xBCFB, 0xF789, 0xEF1A, 0xEF19, 0xCE1A, 0xDE3A,
    0xDE39, 0x8C1A, 0x9C3A, 0xBC7A, 0xBC79, 0x82A0, 0x8290, 0xC14C, 0x8288, 0x8284,
    0x8282, 0x82D8, 0x82CC, 0x82C6, 0x82C3, 0x82EE, 0x82E7, 0xC350, 0xC348, 0xE1A6,
    0xC344, 0xE1A3, 0xC342, 0xC341, 0x8250, 0xC12C, 0x86D0, 0xC36C, 0xC126, 0x86C8,
    0xC366, 0x86C4, 0xC363, 0x86C2, 0x8241, 0x86C1, 0x826C, 0xC137, 0x86EC, 0xC377,
    0x86E6, 0x8263, 0x86E3, 0x8277, 0x86F7, 0xE3A8, 0xE3A4, 0xE3A2, 0xE3A1, 0xC328,
    0xC768, 0xE3B6, 0xE193, 0xC764, 0xE3B3, 0xC762, 0xC321, 0xC761, 0x8228, 0x8668,
    0x8224, 0xC113, 0x8EE8, 0x8664, 0x8222, 0x8EE4, 0x8662, 0x8221, 0x8EE2, 0x8661,
    0x8236, 0x8676, 0x8233, 0x8EF6, 0x8673, 0x8EF3, 0xF3D4, 0xF3D2, 0xF3D1, 0xE394,
    0xE7B4, 0xF3DB, 0xE7B2, 0xE391, 0xE7B1, 0xC314, 0xE18B, 0xC734, 0xE39B, 0xCF74,
    0xC732, 0xC311, 0xCF72, 0xC731, 0xCF71, 0x8214, 0xC10B, 0x8634, 0xC31B, 0x8E74,
    0x8632, 0x8211, 0x9EF4, 0x8E72, 0x8631, 0x9EF2, 0x8E71, 0x821B, 0x863B, 0x8E7B,
    0x9EFB, 0xFBEA, 0xFBE9, 0xF3CA, 0xF7DA, 0xF3C9, 0xF7D9, 0xE38A, 0xE79A, 0xE389,
    0xEFBA, 0xE799, 0xEFB9, 0xC30A, 0xC71A, 0xC309, 0xCF3A, 0xC719, 0xDF7A, 0xFAB0,
    0xFD5C, 0xF520, 0xFA98, 0xFD4E, 0xF510, 0xFA8C, 0xFD47, 0xF508, 0xFA86, 0xF504,
    0xFA83, 0xF502, 0xF5B0, 0xFADC, 0xFD6F, 0xEB20, 0xF598, 0xFACE, 0xEB10, 0xF58C,
    0xFAC7, 0xEB08, 0xF586, 0xEB04, 0xF583, 0xEB02, 0xEBB0, 0xF5DC, 0xFAEF, 0xD720,
    0xEB98, 0xF5CE, 0xD710, 0xEB8C, 0xF5C7, 0xD708, 0xEB86, 0xD704, 0xEB83, 0xD702,
    0xD7B0, 0xEBDC, 0xF5EF, 0xAF20, 0xD798, 0xEBCE, 0xAF10, 0xD78C, 0xEBC7, 0xAF08,
    0xD786, 0xAF04, 0xD783, 0xAFB0, 0xD7DC, 0xEBEF, 0xAF98, 0xD7CE, 0xAF8C, 0xD7C7,
    0xAF86, 0xAFDC, 0xD7EF, 0xAFCE, 0xAFC7, 0xF4A0, 0xFA58, 0xFD2E, 0xF490, 0xFA4C,
    0xFD27, 0xF488, 0xFA46, 0xF484, 0xFA43, 0xF482, 0xF481, 0xE9A0, 0xF4D8, 0xFA6E,
    0xE990, 0xF4CC, 0xFA67, 0xE988, 0xF4C6, 0xE984, 0xF4C3, 0xE982, 0xE981, 0xD3A0,
    0xE9D8, 0xF4EE, 0xD390, 0xE9CC, 0xF4E7, 0xD388, 0xE9C6, 0xD384, 0xE9C3, 0xD382,
    0xD381, 0xA7A0, 0xD3D8, 0xE9EE, 0xA790, 0xD3CC, 0xE9E7, 0xA788, 0xD3C6, 0xA784,
    0xD3C3, 0xA782, 0xA7D8, 0xD3EE, 0xA7CC, 0xD3E7, 0xA7C6, 0xA7C3, 0xA7EE, 0xA7E7,
    0xF450, 0xFA2C, 0xFD17, 0xF448, 0xFA26, 0xF444, 0xFA23, 0xF442, 0xF441, 0xE8D0,
    0xF46C, 0xFA37, 0xE8C8, 0xF466, 0xE8C4, 0xF463, 0xE8C2, 0xE8C1, 0xD1D0, 0xE8EC,
    0xF477, 0xD1C8, 0xE8E6, 0xD1C4, 0xE8E3, 0xD1C2, 0xD1C1, 0xA3D0, 0xD1EC, 0xE8F7,
    0xA3C8, 0xD1E6, 0xA3C4, 0xD1E3, 0xA3C2, 0xA3C1, 0xA3EC, 0xD1F7, 0xA3E6, 0xA3E3,
    0xA3F7, 0xF428, 0xFA16, 0xF424, 0xFA13, 0xF422, 0xF421, 0xE868, 0xF436, 0xE864,
    0xF433, 0xE862, 0xE861, 0xD0E8, 0xE876, 0xD0E4, 0xE873, 0xD0E2, 0xD0E1, 0xA1E8,
    0xD0F6, 0xA1E4, 0xD0F3, 0xA1E2, 0xA1E1, 0xA1F6, 0xA1F3, 0xF414, 0xFA0B, 0xF412,
    0xF411, 0xE834, 0xF41B, 0xE832, 0xE831, 0xD074, 0xE83B, 0xD072, 0xD071, 0xA0F4,
    0xD07B, 0xA0F2, 0xA0F1, 0xF40A, 0xF409, 0xE81A, 0xE819, 0xD03A, 0xD039, 0xF2A0,
    0xF958, 0xFCAE, 0xF290, 0xF94C, 0xFCA7, 0xF288, 0xF946, 0xF284, 0xF943, 0xF282,
    0xF281, 0xE5A0, 0xF2D8, 0xF96E, 0xE590, 0xF2CC, 0xF967, 0xE588, 0xF2C6, 0xE584,
    0xF2C3, 0xE582, 0xE581, 0xCBA0, 0xE5D8, 0xF2EE, 0xCB90, 0xE5CC, 0xF2E7, 0xCB88,
    0xE5C6, 0xCB84, 0xE5C3, 0xCB82, 0xCB81, 0x97A0, 0xCBD8, 0xE5EE, 0x9790, 0xCBCC,
    0xE5E7, 0x9788, 0xCBC6, 0x9784, 0xCBC3, 0x9782, 0x97D8, 0xCBEE, 0x97CC, 0xCBE7,
    0x97C6, 0x97C3, 0x97EE, 0x97E7, 0xFB50, 0xFDAC, 0xB5F8, 0xFB48, 0xFDA6, 0xB4FC,
    0xFB44, 0xFDA3, 0xB47E, 0xFB42, 0xFB41, 0xF250, 0xF92C, 0xFC97, 0xF6D0, 0xF248,
    0xFDB7, 0xF6C8, 0xFB66, 0xF923, 0xF6C4, 0xF242, 0xF6C2, 0xF241, 0xF6C1, 0xE4D0,
    0xF26C, 0xF937, 0xEDD0, 0xE4C8, 0xF266, 0xEDC8, 0xF6E6, 0xF263, 0xEDC4, 0xE4C2,
    0xEDC2, 0xE4C1, 0xEDC1, 0xC9D0, 0xE4EC, 0xF277, 0xDBD0, 0xC9C8, 0xE4E6, 0xDBC8,
    0xEDE6, 0xE4E3, 0xDBC4, 0xC9C2, 0xDBC2, 0xC9C1, 0xDBC1, 0x93D0, 0xC9EC, 0xE4F7,
    0xB7D0, 0x93C8, 0xC9E6, 0xB7C8, 0xDBE6, 0xC9E3, 0xB7C4, 0x93C2, 0xB7C2, 0x93C1,
    0x93EC, 0xC9F7, 0xB7EC, 0x93E6, 0xB7E6, 0x93E3, 0xB7E3, 0x93F7, 0xFB28, 0xFD96,
    0xB2FC, 0xFB24, 0xFD93, 0xB27E, 0xFB22, 0xB23F, 0xFB21, 0xF228, 0xF916, 0xF668,
    0xF224, 0xF913, 0xF664, 0xFB33, 0xF662, 0xF221, 0xF661, 0xE468, 0xF236, 0xECE8,
    0xE464, 0xF233, 0xECE4, 0xF673, 0xECE2, 0xE461, 0xECE1, 0xC8E8, 0xE476, 0xD9E8,
    0xC8E4, 0xE473, 0xD9E4, 0xECF3, 0xD9E2, 0xC8E1, 0xD9E1, 0x91E8, 0xC8F6, 0xB3E8,
    0x91E4, 0xC8F3, 0xB3E4, 0xD9F3, 0xB3E2, 0x91E1, 0xB3E1, 0x91F6, 0xB3F6, 0x91F3,
    0xB3F3, 0xFB14, 0xFD8B, 0xB17E, 0xFB12, 0xB13F, 0xFB11, 0xF214, 0xF90B, 0xF634,
    0xFB1B, 0xF632, 0xF211, 0xF631, 0xE434, 0xF21B, 0xEC74, 0xE432, 0xEC72, 0xE431,
    0xEC71, 0xC874, 0xE43B, 0xD8F4, 0xEC7B, 0xD8F2, 0xC871, 0xD8F1, 0x90F4, 0xC87B,
    0xB1F4, 0x90F2, 0xB1F2, 0x90F1, 0xB1F1, 0x90FB, 0xB1FB, 0xFB0A, 0xB0BF, 0xFB09,
    0xF20A, 0xF61A, 0xF209, 0xF619, 0xE41A, 0xEC3A, 0xE419, 0xEC39, 0xC83A, 0xD87A,
    0xC839, 0xD879, 0x907A, 0xB0FA, 0x9079, 0xB0F9, 0xFB05, 0xF205, 0xF60D, 0xE40D,
    0xEC1D, 0xC81D, 0xD83D, 0xF150, 0xF8AC, 0xFC57, 0xF148, 0xF8A6, 0xF144, 0xF8A3,
    0xF142, 0xF141, 0xE2D0, 0xF16C, 0xF8B7, 0xE2C8, 0xF166, 0xE2C4, 0xF163, 0xE2C2,
    0xE2C1, 0xC5D0, 0xE2EC, 0xF177, 0xC5C8, 0xE2E6, 0xC5C4, 0xE2E3, 0xC5C2, 0xC5C1,
    0x8BD0, 0xC5EC, 0xE2F7, 0x8BC8, 0xC5E6, 0x8BC4, 0xC5E3, 0x8BC2, 0x8BC1, 0x8BEC,
    0xC5F7, 0x8BE6, 0x8BE3, 0x8BF7, 0xF9A8, 0xFCD6, 0x9AFC, 0xF9A4, 0xFCD3, 0x9A7E,
    0xF9A2, 0x9A3F, 0xF9A1, 0xF128, 0xF896, 0xF368, 0xF124, 0xF893, 0xF364, 0xF9B3,
    0xF362, 0xF121, 0xF361, 0xE268, 0xF136, 0xE6E8, 0xE264, 0xF133, 0xE6E4, 0xF373,
    0xE6E2, 0xE261, 0xE6E1, 0xC4E8, 0xE276, 0xCDE8, 0xC4E4, 0xE273, 0xCDE4, 0xE6F3,
    0xCDE2, 0xC4E1, 0xCDE1, 0x89E8, 0xC4F6, 0x9BE8, 0x89E4, 0xC4F3, 0x9BE4, 0xCDF3,
    0x9BE2, 0x89E1, 0x9BE1, 0x89F6, 0x9BF6, 0x89F3, 0x9BF3, 0xFDD4, 0xBAF8, 0xDD7E,
    0xFDD2, 0xBA7C, 0xDD3F, 0xFDD1, 0xBA3E, 0xBA1F, 0xF994, 0xFCCB, 0x997E, 0xFBB4,
    0xFDDB, 0xBB7E, 0x993F, 0xFBB2, 0xF991, 0xBB3F, 0xFBB1, 0xF114, 0xF88B, 0xF334,
    0xF112, 0xF774, 0xFBBB, 0xF111, 0xF772, 0xF331, 0xF771, 0xE234, 0xF11B, 0xE674,
    0xE232, 0xEEF4, 0xE672, 0xE231, 0xEEF2, 0xE671, 0xEEF1, 0xC474, 0xE23B, 0xCCF4,
    0xC472, 0xDDF4, 0xCCF2, 0xC471, 0xDDF2, 0xCCF1, 0xDDF1, 0x88F4, 0xC47B, 0x99F4,
    0x88F2, 0xBBF4, 0x99F2, 0x88F1, 0xBBF2, 0x99F1, 0xBBF1, 0x88FB, 0x99FB, 0xFDCA,
    0xB97C, 0xDCBF, 0xFDC9, 0xB93E, 0xB91F, 0xF98A, 0x98BF, 0xFB9A, 0xF989, 0xB9BF,
    0xFB99, 0xF10A, 0xF31A, 0xF109, 0xF73A, 0xF319, 0xF739, 0xE21A, 0xE63A, 0xE219,
    0xEE7A, 0xE639, 0xEE79, 0xC43A, 0xCC7A, 0xC439, 0xDCFA, 0xCC79, 0xDCF9, 0x887A,
    0x98FA, 0x8879, 0xB9FA, 0x98F9, 0xB9F9, 0xFDC5, 0xB8BE, 0xB89F, 0xF985, 0xFB8D,
    0xF105, 0xF30D, 0xF71D, 0xE20D, 0xE61D, 0xEE3D, 0xC41D, 0xCC3D, 0xDC7D, 0x883D,
    0x987D, 0xB8FD, 0xB85F, 0xF0A8, 0xF856, 0xF0A4, 0xF853, 0xF0A2, 0xF0A1, 0xE168,
    0xF0B6, 0xE164, 0xF0B3, 0xE162, 0xE161, 0xC2E8, 0xE176, 0xC2E4, 0xE173, 0xC2E2,
    0xC2E1, 0x85E8, 0xC2F6, 0x85E4, 0xC2F3, 0x85E2, 0x85E1, 0x85F6, 0x85F3, 0xF8D4,
    0xFC6B, 0x8D7E, 0xF8D2, 0x8D3F, 0xF8D1, 0xF094, 0xF84B, 0xF1B4, 0xF092, 0xF1B2,
    0xF091, 0xF1B1, 0xE134, 0xF09B, 0xE374, 0xE132, 0xE372, 0xE131, 0xE371, 0xC274,
    0xE13B, 0xC6F4, 0xC272, 0xC6F2, 0xC271, 0xC6F1, 0x84F4, 0xC27B, 0x8DF4, 0x84F2,
    0x8DF2, 0x84F1, 0x8DF1, 0x84FB, 0x8DFB, 0xFCEA, 0x9D7C, 0xCEBF, 0xFCE9, 0x9D3E,
    0x9D1F, 0xF8CA, 0x8CBF, 0xF9DA, 0xF8C9, 0x9DBF, 0xF9D9, 0xF08A, 0xF19A, 0xF089,
    0xF3BA, 0xF199, 0xF3B9, 0xE11A, 0xE33A, 0xE119, 0xE77A, 0xE339, 0xE779, 0xC23A,
    0xC67A, 0xC239, 0xCEFA, 0xC679, 0xCEF9, 0x847A, 0x8CFA, 0x8479, 0x9DFA, 0x8CF9,
    0x9DF9, 0xBD78, 0xDEBE, 0xBD3C, 0xDE9F, 0xBD1E, 0xBD0F, 0xFCE5, 0x9CBE, 0xFDED,
    0xBDBE, 0x9C9F, 0xBD9F, 0xF8C5, 0xF9CD, 0xFBDD, 0xF085, 0xF18D, 0xF39D, 0xF7BD,
    0xE10D, 0xE31D, 0xE73D, 0xEF7D, 0xC21D, 0xC63D, 0xCE7D, 0xDEFD, 0x843D, 0x8C7D,
    0x9CFD, 0xBCBC, 0xDE5F, 0xBC9E, 0xBC8F, 0x9C5F, 0xBCDF, 0xBC5E, 0xBC4F, 0xBC2F,
    0xF054, 0xF052, 0xF051, 0xE0B4, 0xF05B, 0xE0B2, 0xE0B1, 0xC174, 0xE0BB, 0xC172,
    0xC171, 0x82F4, 0xC17B, 0x82F2, 0x82F1, 0x82FB, 0xF86A, 0x86BF, 0xF869, 0xF04A,
    0xF0DA, 0xF049, 0xF0D9, 0xE09A, 0xE1BA, 0xE099, 0xE1B9, 0xC13A, 0xC37A, 0xC139,
    0xC379, 0x827A, 0x86FA, 0x8279, 0x86F9, 0xFC75, 0x8EBE, 0x8E9F, 0xF865, 0xF8ED,
    0xF045, 0xF0CD, 0xF1DD, 0xE08D, 0xE19D, 0xE3BD, 0xC11D, 0xC33D, 0xC77D, 0x823D,
    0x867D, 0x8EFD, 0x9EBC, 0xCF5F, 0x9E9E, 0x9E8F, 0x8E5F, 0x9EDF, 0xBEB8, 0xDF5E,
    0xBE9C, 0xDF4F, 0xBE8E, 0xBE87, 0x9E5E, 0xBEDE, 0x9E4F, 0xBECF, 0xBE5C, 0xDF2F,
    0xBE4E, 0xBE47, 0x9E2F, 0xBE6F, 0xBE2E, 0xBE27, 0xBE17, 0xE05A, 0xE059, 0xC0BA,
    0xC0B9, 0x817A, 0x8179, 0xF06D, 0xE04D, 0xE0DD, 0xC09D, 0xC1BD, 0x813D, 0x837D,
    0x875F, 0x8F5E, 0x8F4F, 0x9F5C, 0xCFAF, 0x9F4E, 0x9F47, 0x8F2F, 0x9F6F, 0xBF58,
    0xDFAE, 0xBF4C, 0xDFA7, 0xBF46, 0xBF43, 0x9F2E, 0xBF6E, 0x9F27, 0xBF67, 0xBF2C,
    0xDF97, 0xBF26, 0xBF23, 0x9F17, 0xBF37, 0xBF16, 0xBF13, 0x87AF, 0x8FAE, 0x8FA7,
    0x9FAC, 0xCFD7, 0x9FA6, 0x9FA3, 0x8F97, 0x9FB7, 0x9F96, 0x9F93, 0xD5F0, 0xEAFC,
    0xA9E0, 0xD4F8, 0xEA7E, 0xA8F0, 0xD47C, 0xEA3F, 0xA878, 0xD43E, 0xA83C, 0xFD68,
    0xADF0, 0xD6FC, 0xFD64, 0xACF8, 0xD67E, 0xFD62, 0xAC7C, 0xD63F, 0xFD61, 0xAC3E,
    0xFAE8, 0xFD76, 0xAEFC, 0xFAE4, 0xFD73, 0xAE7E, 0xFAE2, 0xAE3F, 0xFAE1, 0xF5E8,
    0xFAF6, 0xF5E4, 0xFAF3, 0xF5E2, 0xF5E1, 0xEBE8, 0xF5F6, 0xEBE4, 0xF5F3, 0xEBE2,
    0xEBE1, 0xD7E8, 0xEBF6, 0xD7E4, 0xEBF3, 0xD7E2, 0xA5E0, 0xD2F8, 0xE97E, 0xA4F0,
    0xD27C, 0xE93F, 0xA478, 0xD23E, 0xA43C, 0xD21F, 0xA41E, 0xFD34, 0xA6F8, 0xD37E,
    0xFD32, 0xA67C, 0xD33F, 0xFD31, 0xA63E, 0xA61F, 0xFA74, 0xFD3B, 0xA77E, 0xFA72,
    0xA73F, 0xFA71, 0xF4F4, 0xFA7B, 0xF4F2, 0xF4F1, 0xE9F4, 0xF4FB, 0xE9F2, 0xE9F1,
    0xD3F4, 0xE9FB, 0xD3F2, 0xD3F1, 0xA2F0, 0xD17C, 0xE8BF, 0xA278, 0xD13E, 0xA23C,
    0xD11F, 0xA21E, 0xA20F, 0xFD1A, 0xA37C, 0xD1BF, 0xFD19, 0xA33E, 0xA31F, 0xFA3A,
    0xA3BF, 0xFA39, 0xF47A, 0xF479, 0xE8FA, 0xE8F9, 0xD1FA, 0xD1F9, 0xA178, 0xD0BE,
    0xA13C, 0xD09F, 0xA11E, 0xA10F, 0xFD0D, 0xA1BE, 0xA19F, 0xFA1D, 0xF43D, 0xE87D,
    0xA0BC, 0xD05F, 0xA09E, 0xA08F, 0xA0DF, 0xA05E, 0xA04F, 0x95E0, 0xCAF8, 0xE57E,
    0x94F0, 0xCA7C, 0xE53F, 0x9478, 0xCA3E, 0x943C, 0xCA1F, 0x941E, 0xFCB4, 0x96F8,
    0xCB7E, 0xFCB2, 0x967C, 0xCB3F, 0xFCB1, 0x963E, 0x961F, 0xF974, 0xFCBB, 0x977E,
    0xF972, 0x973F, 0xF971, 0xF2F4, 0xF97B, 0xF2F2, 0xF2F1, 0xE5F4, 0xF2FB, 0xE5F2,
    0xE5F1, 0xCBF4, 0xE5FB, 0xCBF2, 0xCBF1, 0xDAF0, 0xED7C, 0xF6BF, 0xB4E0, 0xDA78,
    0xED3E, 0xB470, 0xDA3C, 0xED1F, 0xB438, 0xDA1E, 0xB41C, 0xDA0F, 0xB40E, 0x92F0,
    0xC97C, 0xE4BF, 0xB6F0, 0x9278, 0xC93E, 0xB678, 0xDB3E, 0xC91F, 0xB63C, 0x921E,
    0xB61E, 0x920F, 0xB60F, 0xFC9A, 0x937C, 0xC9BF, 0xFDBA, 0xFC99, 0xB77C, 0x933E,
    0xFDB9, 0xB73E, 0x931F, 0xB71F, 0xF93A, 0x93BF, 0xFB7A, 0xF939, 0xB7BF, 0xFB79,
    0xF27A, 0xF6FA, 0xF279, 0xF6F9, 0xE4FA, 0xEDFA, 0xE4F9, 0xEDF9, 0xC9FA, 0xC9F9,
    0xB2E0, 0xD978, 0xECBE, 0xB270, 0xD93C, 0xEC9F, 0xB238, 0xD91E, 0xB21C, 0xD90F,
    0xB20E, 0xB207, 0x9178, 0xC8BE, 0xB378, 0x913C, 0xC89F, 0xB33C, 0xD99F, 0xB31E,
    0x910F, 0xB30F, 0xFC8D, 0x91BE, 0xFD9D, 0xB3BE, 0x919F, 0xB39F, 0xF91D, 0xFB3D,
    0xF23D, 0xF67D, 0xE47D, 0xECFD, 0xC8FD, 0xB170, 0xD8BC, 0xEC5F, 0xB138, 0xD89E,
    0xB11C, 0xD88F, 0xB10E, 0xB107, 0x90BC, 0xC85F, 0xB1BC, 0x909E, 0xB19E, 0x908F,
    0xB18F, 0x90DF, 0xB1DF, 0xB0B8, 0xD85E, 0xB09C, 0xD84F, 0xB08E, 0xB087, 0x905E,
    0xB0DE, 0x904F, 0xB0CF, 0xB05C, 0xD82F, 0xB04E, 0xB047, 0x902F, 0xB06F, 0xB02E,
    0xB027, 0x8AF0, 0xC57C, 0xE2BF, 0x8A78, 0xC53E, 0x8A3C, 0xC51F, 0x8A1E, 0x8A0F,
    0xFC5A, 0x8B7C, 0xC5BF, 0xFC59, 0x8B3E, 0x8B1F, 0xF8BA, 0x8BBF, 0xF8B9, 0xF17A,
    0xF179, 0xE2FA, 0xE2F9, 0xC5FA, 0xC5F9, 0x9AE0, 0xCD78, 0xE6BE, 0x9A70, 0xCD3C,
    0xE69F, 0x9A38, 0xCD1E, 0x9A1C, 0xCD0F, 0x9A0E, 0x9A07, 0x8978, 0xC4BE, 0x9B78,
    0x893C, 0xC49F, 0x9B3C, 0xCD9F, 0x9B1E, 0x890F, 0x9B0F, 0xFC4D, 0x89BE, 0xFCDD,
    0x9BBE, 0x899F, 0x9B9F, 0xF89D, 0xF9BD, 0xF13D, 0xF37D, 0xE27D, 0xE6FD, 0xC4FD,
    0xDD70, 0xEEBC, 0xF75F, 0xBA60, 0xDD38, 0xEE9E, 0xBA30, 0xDD1C, 0xEE8F, 0xBA18,
    0xDD0E, 0xBA0C, 0xDD07, 0xBA06, 0x9970, 0xCCBC, 0xE65F, 0xBB70, 0x9938, 0xCC9E,
    0xBB38, 0xDD9E, 0xCC8F, 0xBB1C, 0x990E, 0xBB0E, 0x9907, 0xBB07, 0x88BC, 0xC45F,
    0x99BC, 0x889E, 0xBBBC, 0x999E, 0x888F, 0xBB9E, 0x998F, 0xBB8F, 0x88DF, 0x99DF,
    0xBBDF, 0xB960, 0xDCB8, 0xEE5E, 0xB930, 0xDC9C, 0xEE4F, 0xB918, 0xDC8E, 0xB90C,
    0xDC87, 0xB906, 0xB903, 0x98B8, 0xCC5E, 0xB9B8, 0x989C, 0xCC4F, 0xB99C, 0xDCCF,
    0xB98E, 0x9887, 0xB987, 0x885E, 0x98DE, 0x884F, 0xB9DE, 0x98CF, 0xB9CF, 0xB8B0,
    0xDC5C, 0xEE2F, 0xB898, 0xDC4E, 0xB88C, 0xDC47, 0xB886, 0xB883, 0x985C, 0xCC2F,
    0xB8DC, 0x984E, 0xB8CE, 0x9847, 0xB8C7, 0x882F, 0x986F, 0xB8EF, 0xB858, 0xDC2E,
    0xB84C, 0xDC27, 0xB846, 0xB843, 0x982E, 0xB86E, 0x9827, 0xB867, 0xB82C, 0xDC17,
    0xB826, 0xB823, 0x9817, 0xB837, 0xB816, 0xB813, 0x8578, 0xC2BE, 0x853C, 0xC29F,
    0x851E, 0x850F, 0x85BE, 0x859F, 0xF85D, 0xF0BD, 0xE17D, 0xC2FD, 0x8D70, 0xC6BC,
    0xE35F, 0x8D38, 0xC69E, 0x8D1C, 0xC68F, 0x8D0E, 0x8D07, 0x84BC, 0xC25F, 0x8DBC,
    0x849E, 0x8D9E, 0x848F, 0x8D8F, 0x84DF, 0x8DDF, 0x9D60, 0xCEB8, 0xE75E, 0x9D30,
    0xCE9C, 0xE74F, 0x9D18, 0xCE8E, 0x9D0C, 0xCE87, 0x9D06, 0x9D03, 0x8CB8, 0xC65E,
    0x9DB8, 0x8C9C, 0xC64F, 0x9D9C, 0x8C8E, 0x9D8E, 0x8C87, 0x9D87, 0x845E, 0x8CDE,
    0x844F, 0x9DDE, 0x8CCF, 0x9DCF, 0xDEB0, 0xEF5C, 0xF7AF, 0xBD20, 0xDE98, 0xEF4E,
    0xBD10, 0xDE8C, 0xEF47, 0xBD08, 0xDE86, 0xBD04, 0xDE83, 0xBD02, 0x9CB0, 0xCE5C,
    0xE72F, 0xBDB0, 0x9C98, 0xCE4E, 0xBD98, 0xDECE, 0xCE47, 0xBD8C, 0x9C86, 0xBD86,
    0x9C83, 0xBD83, 0x8C5C, 0xC62F, 0x9CDC, 0x8C4E, 0xBDDC, 0x9CCE, 0x8C47, 0xBDCE,
    0x9CC7, 0xBDC7, 0x842F, 0x8C6F, 0x9CEF, 0xBDEF, 0xBCA0, 0xDE58, 0xEF2E, 0xBC90,
    0xDE4C, 0xEF27, 0xBC88, 0xDE46, 0xBC84, 0xDE43, 0xBC82, 0xBC81, 0x9C58, 0xCE2E,
    0xBCD8, 0x9C4C, 0xCE27, 0xBCCC, 0xDE67, 0xBCC6, 0x9C43, 0xBCC3, 0x8C2E, 0x9C6E,
    0x8C27, 0xBCEE, 0x9C67, 0xBCE7, 0xBC50, 0xDE2C, 0xEF17, 0xBC48, 0xDE26, 0xBC44,
    0xDE23, 0xBC42, 0xBC41, 0x9C2C, 0xCE17, 0xBC6C, 0x9C26, 0xBC66, 0x9C23, 0xBC63,
    0x8C17, 0x9C37, 0xBC77, 0xBC28, 0xDE16, 0xBC24, 0xDE13, 0xBC22, 0xBC21, 0x9C16,
    0xBC36, 0x9C13, 0xBC33, 0xBC14, 0xDE0B, 0xBC12, 0xBC11, 0x9C0B, 0xBC1B, 0x82BC,
    0xC15F, 0x829E, 0x828F, 0x82DF, 0x86B8, 0xC35E, 0x869C, 0xC34F, 0x868E, 0x8687,
    0x825E, 0x86DE, 0x824F, 0x86CF, 0x8EB0, 0xC75C, 0xE3AF, 0x8E98, 0xC74E, 0x8E8C,
    0xC747, 0x8E86, 0x8E83, 0x865C, 0xC32F, 0x8EDC, 0x864E, 0x8ECE, 0x8647, 0x8EC7,
    0x822F, 0x866F, 0x8EEF, 0x9EA0, 0xCF58, 0xE7AE, 0x9E90, 0xCF4C, 0xE7A7, 0x9E88,
    0xCF46, 0x9E84, 0xCF43, 0x9E82, 0x9E81, 0x8E58, 0xC72E, 0x9ED8, 0x8E4C, 0xC727,
    0x9ECC, 0xCF67, 0x9EC6, 0x8E43, 0x9EC3, 0x862E, 0x8E6E, 0x8627, 0x9EEE, 0x8E67,
    0x9EE7, 0xDF50, 0xEFAC, 0xF7D7, 0xDF48, 0xEFA6, 0xDF44, 0xEFA3, 0xDF42, 0xDF41,
    0x9E50, 0xCF2C, 0xE797, 0xBED0, 0x9E48, 0xCF26, 0xBEC8, 0xDF66, 0xCF23, 0xBEC4,
    0x9E42, 0xBEC2, 0x9E41, 0xBEC1, 0x8E2C, 0xC717, 0x9E6C, 0x8E26, 0xBEEC, 0x9E66,
    0x8E23, 0xBEE6, 0x9E63, 0xBEE3, 0x8617, 0x8E37, 0x9E77, 0xBEF7, 0xDF28, 0xEF96,
    0xDF24, 0xEF93, 0xDF22, 0xDF21, 0x9E28, 0xCF16, 0xBE68, 0x9E24, 0xCF13, 0xBE64,
    0xDF33, 0xBE62, 0x9E21, 0xBE61, 0x8E16, 0x9E36, 0x8E13, 0xBE76, 0x9E33, 0xBE73,
    0xDF14, 0xEF8B, 0xDF12, 0xDF11, 0x9E14, 0xCF0B, 0xBE34, 0x9E12, 0xBE32, 0x9E11,
    0xBE31, 0x8E0B, 0x9E1B, 0xBE3B, 0xDF0A, 0xDF09, 0x9E0A, 0xBE1A, 0x9E09, 0xBE19,
    0x815E, 0x814F, 0x835C, 0xC1AF, 0x834E, 0x8347, 0x812F, 0x836F, 0x8758, 0xC3AE,
    0x874C, 0xC3A7, 0x8746, 0x8743, 0x832E, 0x876E, 0x8327, 0x8767, 0x8F50, 0xC7AC,
    0xE3D7, 0x8F48, 0xC7A6, 0x8F44, 0xC7A3, 0x8F42, 0x8F41, 0x872C, 0xC397, 0x8F6C,
    0xC7B7, 0x8F66, 0x8723, 0x8F63, 0x8317, 0x8737, 0x8F77, 0xCFA8, 0xE7D6, 0xCFA4,
    0xE7D3, 0xCFA2, 0xCFA1, 0x8F28, 0xC796, 0x9F68, 0xCFB6, 0xC793, 0x9F64, 0x8F22,
    0x9F62, 0x8F21, 0x9F61, 0x8716, 0x8F36, 0x8713, 0x9F76, 0x8F33, 0x9F73, 0xEFD4,
    0xF7EB, 0xEFD2, 0xEFD1, 0xCF94, 0xE7CB, 0xDFB4, 0xCF92, 0xDFB2, 0xCF91, 0xDFB1,
    0x8F14, 0xC78B, 0x9F34, 0x8F12, 0xBF74, 0x9F32, 0x8F11, 0xBF72, 0x9F31, 0xBF71,
    0x870B, 0x8F1B, 0x9F3B, 0xBF7B, 0xEFCA, 0xEFC9, 0xCF8A, 0xDF9A, 0xCF89, 0xDF99,
    0x8F0A, 0x9F1A, 0x8F09, 0xBF3A, 0x9F19, 0xBF39, 0xEFC5, 0xCF85, 0xDF8D, 0x8F05,
    0x9F0D, 0xBF1D, 0x81AE, 0x81A7, 0x83AC, 0xC1D7, 0x83A6, 0x83A3, 0x8197, 0x83B7,
    0x87A8, 0xC3D6, 0x87A4, 0xC3D3, 0x87A2, 0x87A1, 0x8396, 0x87B6, 0x8393, 0x87B3,
    0xC7D4, 0xE3EB, 0xC7D2, 0xC7D1, 0x8794, 0xC3CB, 0x8FB4, 0xC7DB, 0x8FB2, 0x8791,
    0x8FB1, 0x838B, 0x879B, 0x8FBB, 0xE7EA, 0xE7E9, 0xC7CA, 0xCFDA, 0xC7C9, 0xCFD9,
    0x878A, 0x8F9A, 0x8789, 0x9FBA, 0x8F99, 0x9FB9, 0xE7E5, 0xC7C5, 0xCFCD, 0x8785,
    0x8F8D, 0x9F9D, 0x81D6, 0x81D3, 0x83D4, 0xC1EB, 0x83D2, 0x83D1, 0x81CB, 0x83DB,
    0xC3EA, 0xC3E9, 0x83CA, 0x87DA, 0x83C9, 0x87D9, 0xE3F5
};

/* MicroPDF417 coefficients from ISO/IEC 24728:2006 Annex F */
INTERNAL_DATA const unsigned short pdf_Microcoeffs[344] = {
    /* k = 7 */
    76, 925, 537, 597, 784, 691, 437,

    /* k = 8 */
    237, 308, 436, 284, 646, 653, 428, 379,

    /* k = 9 */
    567, 527, 622, 257, 289, 362, 501, 441, 205,

    /* k = 10 */
    377, 457, 64, 244, 826, 841, 818, 691, 266, 612,

    /* k = 11 */
    462, 45, 565, 708, 825, 213, 15, 68, 327, 602, 904,

    /* k = 12 */
    597, 864, 757, 201, 646, 684, 347, 127, 388, 7, 69, 851,

    /* k = 13 */
    764, 713, 342, 384, 606, 583, 322, 592, 678, 204, 184, 394, 692,

    /* k = 14 */
    669, 677, 154, 187, 241, 286, 274, 354, 478, 915, 691, 833, 105, 215,

    /* k = 15 */
    460, 829, 476, 109, 904, 664, 230, 5, 80, 74, 550, 575, 147, 868, 642,

    /* k = 16 */
    274, 562, 232, 755, 599, 524, 801, 132, 295, 116, 442, 428, 295, 42, 176, 65,

    /* k = 18 */
    279, 577, 315, 624, 37, 855, 275, 739, 120, 297, 312, 202, 560, 321, 233, 756,
    760, 573,

    /* k = 21 */
    108, 519, 781, 534, 129, 425, 681, 553, 422, 716, 763, 693, 624, 610, 310, 691,
    347, 165, 193, 259, 568,

    /* k = 26 */
    443, 284, 887, 544, 788, 93, 477, 760, 331, 608, 269, 121, 159, 830, 446, 893,
    699, 245, 441, 454, 325, 858, 131, 847, 764, 169,

    /* k = 32 */
    361, 575, 922, 525, 176, 586, 640, 321, 536, 742, 677, 742, 687, 284, 193, 517,
    273, 494, 263, 147, 593, 800, 571, 320, 803, 133, 231, 390, 685, 330, 63, 410,

    /* k = 38 */
    234, 228, 438, 848, 133, 703, 529, 721, 788, 322, 280, 159, 738, 586, 388, 684,
    445, 680, 245, 595, 614, 233, 812, 32, 284, 658, 745, 229, 95, 689, 920, 771,
    554, 289, 231, 125, 117, 518,

    /* k = 44 */
    476, 36, 659, 848, 678, 64, 764, 840, 157, 915, 470, 876, 109, 25, 632, 405,
    417, 436, 714, 60, 376, 97, 413, 706, 446, 21, 3, 773, 569, 267, 272, 213,
    31, 560, 231, 758, 103, 271, 572, 436, 339, 730, 82, 285,

    /* k = 50 */
    923, 797, 576, 875, 156, 706, 63, 81, 257, 874, 411, 416, 778, 50, 205, 303,
    188, 535, 909, 155, 637, 230, 534, 96, 575, 102, 264, 233, 919, 593, 865, 26,
    579, 623, 766, 146, 10, 739, 246, 127, 71, 244, 211, 477, 920, 876, 427, 820,
    718, 435
};

/* rows, columns, error codewords, k-offset of valid MicroPDF417 sizes from ISO/IEC 24728:2006 */
INTERNAL_DATA const unsigned short pdf_MicroVariants[170] = {
    1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
    11, 14, 17, 20, 24, 28, 8, 11, 14, 17, 20, 23, 26, 6, 8, 10, 12, 15, 20, 26, 32, 38, 44, 4, 6, 8, 10, 12, 15, 20, 26, 32, 38, 44,
    7, 7, 7, 8, 8, 8, 8, 9, 9, 10, 11, 13, 15, 12, 14, 16, 18, 21, 26, 32, 38, 44, 50, 8, 12, 14, 16, 18, 21, 26, 32, 38, 44, 50,
    0, 0, 0, 7, 7, 7, 7, 15, 15, 24, 34, 57, 84, 45, 70, 99, 115, 133, 154, 180, 212, 250, 294, 7, 45, 70, 99, 115, 133, 154, 180, 212, 250, 294
};
/* rows, columns, error codewords, k-offset */

/* following is Left RAP, Centre RAP, Right RAP and Start Cluster from ISO/IEC 24728:2006 tables 10, 11 and 12 */
INTERNAL_DATA const char pdf_RAPTable[136] = {
    1, 8, 36, 19, 9, 25, 1, 1, 8, 36, 19, 9, 27, 1, 7, 15, 25, 37, 1, 1, 21, 15, 1, 47, 1, 7, 15, 25, 37, 1, 1, 21, 15, 1,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 7, 15, 25, 37, 17, 9, 29, 31, 25, 19, 1, 7, 15, 25, 37, 17, 9, 29, 31, 25,
    9, 8, 36, 19, 17, 33, 1, 9, 8, 36, 19, 17, 35, 1, 7, 15, 25, 37, 33, 17, 37, 47, 49, 43, 1, 7, 15, 25, 37, 33, 17, 37, 47, 49,
    0, 3, 6, 0, 6, 0, 0, 0, 3, 6, 0, 6, 6, 0, 0, 6, 0, 0, 0, 0, 6, 6, 0, 3, 0, 0, 6, 0, 0, 0, 0, 6, 6, 0
};

/* Left and Right Row Address Pattern from Table 2 */
INTERNAL_DATA const unsigned short pdf_rap_side[52] = {
    0x322, 0x3A2, 0x3B2, 0x332, 0x372, 0x37A, 0x33A, 0x3BA, 0x39A, 0x3DA,
    0x3CA, 0x38A, 0x30A, 0x31A, 0x312, 0x392, 0x3D2, 0x3D6, 0x3D4, 0x394,
    0x3B4, 0x3A4, 0x3A6, 0x3AE, 0x3AC, 0x3A8, 0x328, 0x32C, 0x32E, 0x326,
    0x336, 0x3B6, 0x396, 0x316, 0x314, 0x334, 0x374, 0x364, 0x366, 0x36E,
    0x36C, 0x368, 0x348, 0x358, 0x35C, 0x35E, 0x34E, 0x34C, 0x344, 0x346,
    0x342, 0x362
};

/* Centre Row Address Pattern from Table 2 */
INTERNAL_DATA const unsigned short pdf_rap_centre[52] = {
    0x2CE, 0x24E, 0x26E, 0x22E, 0x226, 0x236, 0x216, 0x212, 0x21A, 0x23A,
    0x232, 0x222, 0x262, 0x272, 0x27A, 0x2FA, 0x2F2, 0x2F6, 0x276, 0x274,
    0x264, 0x266, 0x246, 0x242, 0x2C2, 0x2E2, 0x2E6, 0x2E4, 0x2EC, 0x26C,
    0x22C, 0x228, 0x268, 0x2E8, 0x2C8, 0x2CC, 0x2C4, 0x2C6, 0x286, 0x28E,
    0x28C, 0x29C, 0x298, 0x2B8, 0x2B0, 0x290, 0x2D0, 0x250, 0x258, 0x25C,
    0x2DC, 0x2DE
};

/* vim: set ts=4 sw=4 et : */
#endif /* Z_PDF417_TABS_H */
