//
// File: test_timsframe.cpp
// Created by: Olivier Langella
// Created on: 12/4/2020
//
/*******************************************************************************
 * Copyright (c) 2020 Olivier Langella <olivier.langella@u-psud.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

// make test ARGS="-V -I 22,22"


#define CATCH_CONFIG_MAIN
#include <catch2/catch.hpp>
#include <QDebug>
#include <QString>
#include <iostream>
#include <pappsomspp/vendors/tims/timsframebase.h>
#include <pappsomspp/mzrange.h>
#include <pappsomspp/msfile/msfileaccessor.h>
#include <pappsomspp/msrun/private/timsmsrunreaderms2.h>
#include <pappsomspp/pappsoexception.h>
#include <pappsomspp/processing/filters/filtersuitestring.h>
#include "config.h"

using namespace pappso;
using namespace std;

TEST_CASE("test tims frames", "[timsframe]")
{

  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  cout << endl << "..:: test TIMS frame init begin ::.." << endl;
  //   OboPsiMod test;

  TimsFrameBase frame(1, 671);

  MzRange mz_range(1200.0001, PrecisionFactory::getPpmInstance(10));

  SECTION("test 1")
  {
    double temperature_correction =
      77.0 * (25.4933665127396 - 25.59978165276) +
      (-3.7) * (26.2222755503081 - 27.6311281556638);
    temperature_correction = (double)1.0 + (temperature_correction / 1.0e6);


    frame.setMzCalibration(25.59978165276,
                           27.6311281556638,
                           0.2,
                           24864.0,
                           313.577620892277,
                           157424.07710945,
                           0.000338743021989553,
                           0.0,
                           0.0,

                           25.4933665127396,
                           26.2222755503081,
                           77.0,
                           -3.7);
    frame.setTimsCalibration(2,
                             1,
                             670,
                             207.775676931964,
                             59.2526676368822,
                             33.0,
                             1,
                             0.0209889001473149,
                             131.440113097798,
                             12.9712317295887,
                             2558.71692505931);
    frame.setTime(2402.64305686123);
    frame.setMsMsType(0);
    quint32 index =
      frame.getMzCalibrationInterfaceSPtr().get()->getTofIndexFromMz(
        mz_range.getMz());
    qDebug() << "index=" << index;
    double mz =
      frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(index);
    qDebug() << "mz=" << QString::number(mz, 'g', 10);

    REQUIRE(mz_range.contains(mz));

    // 313792
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(
      (quint32)313793);
    qDebug() << " 313793 => mz=" << QString::number(mz, 'g', 10);
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(
      (quint32)313792);

    REQUIRE(QString::number(mz, 'g', 10) == "1200.002299");
    qDebug() << " 313792 => mz=" << QString::number(mz, 'g', 10);
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(
      (quint32)313791);
    qDebug() << " 313791 => mz=" << QString::number(mz, 'g', 10);


    double one_over_k0 = frame.getOneOverK0Transformation(400);
    qDebug() << " 400 => one_over_k0=" << QString::number(one_over_k0, 'g', 10);
    std::size_t scan_num = frame.getScanNumFromOneOverK0(one_over_k0);

    REQUIRE(scan_num == 400);
    cout << endl << "..:: test 1 OK ::.." << endl;
  }

  SECTION("test 2")
  {
    cout << endl << "..:: second test, new calibration parameters ::.." << endl;


    frame.setMzCalibration(25.3072302808429,
                           25.3072302808429,
                           0.2,
                           24864.0,
                           313.577620892277,
                           157424.07710945,
                           0.000338743021989553,
                           0.0,
                           0.0,

                           25.4933665127396,
                           26.2222755503081,
                           77.0,
                           -3.7);
    /*
          q.value(2).toDouble(),  // MzCalibration.DigitizerTimebase
          q.value(3).toDouble(),  // MzCalibration.DigitizerDelay
          q.value(4).toDouble(),  // MzCalibration.C0
          q.value(5).toDouble(),  // MzCalibration.C1
          q.value(6).toDouble(),  // MzCalibration.C2
          q.value(7).toDouble()); // MzCalibration.C3
          */

    frame.setTimsCalibration(2,
                             1,
                             1537,
                             179.152532325778,
                             90.4208212951646,
                             33.0,
                             1,
                             0.009221,
                             131.053614,
                             9.656516,
                             2095.24199);
    frame.setTime(1.7950832);
    frame.setMsMsType(0);

    std::size_t index =
      frame.getMzCalibrationInterfaceSPtr().get()->getTofIndexFromMz(
        mz_range.getMz());
    qDebug() << "index=" << index;
    double mz =
      frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(index);
    qDebug() << "mz=" << QString::number(mz, 'g', 10);

    REQUIRE(mz_range.contains(mz));

    // 313792
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(
      (quint32)313793);
    qDebug() << " 313793 => mz=" << QString::number(mz, 'g', 10);
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(
      (quint32)313792);
    qDebug() << " 313792 => mz=" << QString::number(mz, 'g', 10);
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(
      (quint32)313791);
    qDebug() << " 313791 => mz=" << QString::number(mz, 'g', 10);


    double one_over_k0 = frame.getOneOverK0Transformation(400);
    qDebug() << " 400 => one_over_k0=" << QString::number(one_over_k0, 'g', 10);
    std::size_t scan_num = frame.getScanNumFromOneOverK0(one_over_k0);

    REQUIRE(scan_num == 400);


    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(
      (quint32)375176);
    qDebug() << " 375176 => mz=" << QString::number(mz, 'g', 10);


    REQUIRE_FALSE(mz < 50.0);
    cout << endl << "..:: test 2 OK ::.." << endl;
  }

#if USEPAPPSOTREE == 1
  cout << endl << "..:: test TIMS TDF parsing ::.." << endl;
  /*
    pappso::MsFileAccessor accessor(
      "/gorgone/pappso/jouy/raw/2020_Tims_TOF/20201102_HeLa/"
      "11-2-2020_1_QC_HeLa10ng_219.d/analysis.tdf",
      "a1");
      */
  /*
  pappso::MsFileAccessor accessor(
    "/gorgone/pappso/fichiers_fabricants/Bruker/tims_doc/tdf-sdk/example_data/"
    "200ngHeLaPASEF_2min_compressed.d/analysis.tdf",
    "a1");
    */
  pappso::MsFileAccessor accessor(
    "/home/langella/developpement/git/bruker/"
    "200ngHeLaPASEF_2min_compressed.d/"
    "analysis.tdf",
    "a1");
  accessor.setPreferedFileReaderType(pappso::MzFormat::brukerTims,
                                     pappso::FileReaderType::tims_ms2);

  pappso::MsRunReaderSPtr p_msreader =
    accessor.msRunReaderSp(accessor.getMsRunIds().front());

  pappso::TimsMsRunReaderMs2 *tims2_reader =
    dynamic_cast<pappso::TimsMsRunReaderMs2 *>(p_msreader.get());
  if(tims2_reader != nullptr)
    {
      tims2_reader->setMs2BuiltinCentroid(true);
      tims2_reader->setMs2FilterCstSPtr(std::make_shared<FilterSuiteString>(
        FilterSuiteString("chargeDeconvolution|0.02dalton")));
    }

  pappso::QualifiedMassSpectrum mass_spectrum =
    tims2_reader->qualifiedMassSpectrum(7);


  cout << endl
       << "mass_spectrum.getMsLevel()=" << mass_spectrum.getMsLevel() << endl;

  cout << endl
       << "mass_spectrum.getPrecursorMz()=" << mass_spectrum.getPrecursorMz()
       << endl;
  cout << endl
       << "mass_spectrum.getMassSpectrumId().getSpectrumIndex()="
       << mass_spectrum.getMassSpectrumId().getSpectrumIndex() << endl;

       /*
  cout << endl
       << mass_spectrum.getMassSpectrumCstSPtr()
            .get()
            ->toString()
            .toStdString()
            .c_str()
       << endl;
*/
  /*
   *
22:
/home/langella/developpement/git/pappsomspp/src/pappsomspp/vendors/tims/timsframebase.cpp@513,
pappso::TimsFrameBase::getTraceFromCumulatedScansBuiltinCentroid():
375176 22:
/home/langella/developpement/git/pappsomspp/src/pappsomspp/vendors/tims/timsframebase.cpp@518,
pappso::TimsFrameBase::getTraceFromCumulatedScansBuiltinCentroid():
99944.2 22:
/home/langella/developpement/git/pappsomspp/src/pappsomspp/vendors/tims/timsframebase.cpp@253,
pappso::TimsFrameBase::getMzFromTof(): m.length()= 3 22:
/home/langella/developpement/git/pappsomspp/src/pappsomspp/vendors/tims/timsframebase.cpp@520,
pappso::TimsFrameBase::getTraceFromCumulatedScansBuiltinCentroid(): 41.3832

polynome :
-101811 2523.58 -0.002221 313.267
=>  m1= 42.058

*/

  /*
22:
/home/langella/developpement/git/pappsomspp/src/pappsomspp/vendors/tims/timsframebase.cpp@513,
pappso::TimsFrameBase::getTraceFromCumulatedScansBuiltinCentroid():
374144 22:
/home/langella/developpement/git/pappsomspp/src/pappsomspp/vendors/tims/timsframebase.cpp@518,
pappso::TimsFrameBase::getTraceFromCumulatedScansBuiltinCentroid():
99692.8 22:
/home/langella/developpement/git/pappsomspp/src/pappsomspp/vendors/tims/timsframebase.cpp@192,
pappso::TimsFrameBase::getMzFromTof(): 0 22:
/home/langella/developpement/git/pappsomspp/src/pappsomspp/vendors/tims/timsframebase.cpp@253,
pappso::TimsFrameBase::getMzFromTof(): m.length()= 2 22:
/home/langella/developpement/git/pappsomspp/src/pappsomspp/vendors/tims/timsframebase.cpp@520,
pappso::TimsFrameBase::getTraceFromCumulatedScansBuiltinCentroid():
1554.74

polynome :
-100327 2520.37 0.000338744
=> m1= 1584.54

*/

  REQUIRE_FALSE(mass_spectrum.getMassSpectrumCstSPtr().get()->back().x < 40.0);
#elif USEPAPPSOTREE == 1

  cout << endl << "..:: NO test TIMS TDF parsing ::.." << endl;

#endif
}
