#!/bin/bash
#
# Copyright (c) 2015-2016, NVIDIA CORPORATION.  All rights reserved.
#
# This program is free software; you can redistribute it and/or modify it
# under the terms and conditions of the GNU General Public License,
# version 2, as published by the Free Software Foundation.
#
# This program is distributed in the hope it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# See file CREDITS for list of people who contributed to this
# project.
#

Usage ()
{
	cat << EOF
Usage: ./sign.sh <soc> <boot_image> <rsa_priv_key>
  Where,
	soc: tegra124, tegra210
	boot_image: image generated by cbootimage,
	priv_key: rsa key file in .pem format.
EOF
	exit 1;
}

set -e

soc=$1		# tegra124, tegra210
if [[ "${soc}" = tegra124 ]]; then
	bl_block_offset=16384;  # emmc: 16384, spi_flash: 32768: default: emmc
	bct_signed_offset=1712;
	bct_signed_length=6480;
elif [[ "${soc}" = tegra210 ]]; then
	bl_block_offset=32768;  # emmc: 16384, spi_flash: 32768: default: spi
	bct_signed_offset=1296;
	bct_signed_length=8944;
else
	echo "Error: Invalid target device: soc = $soc";
	Usage;
fi;
bct_length=$(($bct_signed_offset + $bct_signed_length));

# more error check
if [ $# -lt 3 ]; then
	echo "Error: Missing parameter(s)";
	Usage;
fi;

#
# In case to add more parameters in the future, we keep the last two as
# IMAGE_FILE and KEY_FILE
#
argv=($@);
IMAGE_FILE=${argv[$#-2]};
KEY_FILE=${argv[$#-1]};
TARGET_IMAGE=$IMAGE_FILE
CONFIG_FILE=config.tmp

CBOOTIMAGE=../src/cbootimage
BCT_DUMP=../src/bct_dump
OBJCOPY=objcopy
OPENSSL=openssl
DD=dd
RM=rm
MV=mv
XXD=xxd
CUT=cut

echo "Sign ${soc} ${IMAGE_FILE} with key ${KEY_FILE}"

echo "Get bl length "
BL_LENGTH=`$BCT_DUMP $IMAGE_FILE | grep "Bootloader\[0\].Length"\
 | awk -F ' ' '{print $4}' | awk -F ';' '{print $1}'`

echo "Extract bootloader to $IMAGE_FILE.bl.tosig, length $BL_LENGTH"
$DD bs=1 skip=${bl_block_offset} if=$IMAGE_FILE of=$IMAGE_FILE.bl.tosig \
	 count=$BL_LENGTH

echo "Calculate rsa signature for bootloader and save to $IMAGE_FILE.bl.sig"
$OPENSSL dgst -sha256 -sigopt rsa_padding_mode:pss -sigopt rsa_pss_saltlen:-1 \
 -sign $KEY_FILE -out $IMAGE_FILE.bl.sig $IMAGE_FILE.bl.tosig

echo "Update bootloader's rsa signature, aes hash and bct's aes hash"
echo "RsaPssSigBlFile = $IMAGE_FILE.bl.sig;" > $CONFIG_FILE
echo "RehashBl;" >> $CONFIG_FILE
$CBOOTIMAGE -s ${soc} -u $CONFIG_FILE $IMAGE_FILE $IMAGE_FILE.tmp

echo "Extract the part of bct which needs to be rsa signed"
$DD bs=1 if=$IMAGE_FILE.tmp of=$IMAGE_FILE.bct.tosig skip=${bct_signed_offset} \
 count=${bct_signed_length}

echo "Calculate rsa signature for bct and save to $IMAGE_FILE.bct.sig"
$OPENSSL dgst -sha256 -sigopt rsa_padding_mode:pss -sigopt rsa_pss_saltlen:-1 \
 -sign $KEY_FILE -out $IMAGE_FILE.bct.sig $IMAGE_FILE.bct.tosig

echo "Create public key modulus from key file $KEY_FILE and save to $KEY_FILE.mod"
$OPENSSL rsa -in $KEY_FILE -noout -modulus -out $KEY_FILE.mod
# remove prefix
$CUT -d= -f2 < $KEY_FILE.mod > $KEY_FILE.mod.tmp

# convert from hexdecimal to binary
$XXD -r -p -l 256 $KEY_FILE.mod.tmp $KEY_FILE.mod.bin

echo "Update bct's rsa signature and modulus"
echo "RsaPssSigBctFile = $IMAGE_FILE.bct.sig;" > $CONFIG_FILE
echo "RsaKeyModulusFile = $KEY_FILE.mod.bin;" >> $CONFIG_FILE
echo ""
$CBOOTIMAGE -s ${soc} -u $CONFIG_FILE $IMAGE_FILE.tmp $TARGET_IMAGE

echo ""
$DD bs=1 if=$TARGET_IMAGE of=${soc}.bct count=${bct_length}
echo ""
echo "Signed bct ${soc}.bct has been successfully generated!";

#echo "Get rid of all temporary files: *.sig, *.tosig, *.tmp, *.mod, *.mod.bin"
$RM -f *.sig *.tosig *.tmp *.mod *.mod.bin
